/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2002 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    wrdt_x.c - WRD Tracer for X Window

    Written by Takanori Watanabe <takawata@shidahara1.planet.kobe-u.ac.jp>
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#include <stdio.h>
#ifndef NO_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include "timidity.h"
#include "common.h"
#include "instrum.h"
#include "playmidi.h"
#include "readmidi.h"
#include "controls.h"
#include "wrd.h"
#include "x_wrdwindow.h"
#include "x_mag.h"
#ifdef ENABLE_SHERRY
#include "x_sherry.h"
#endif /* ENABLE_SHERRY */


static int wrdt_open(tmdy_struct_ex_t *tmdy_struct, char *dummy);
static void wrdt_apply(tmdy_struct_ex_t *tmdy_struct, int cmd, int wrd_argc, int wrd_argv[]);
static void wrdt_update_events(tmdy_struct_ex_t *tmdy_struct);
static int wrdt_start(tmdy_struct_ex_t *tmdy_struct, int wrd_mode);
static void wrdt_end(tmdy_struct_ex_t *tmdy_struct);
static void wrdt_close(tmdy_struct_ex_t *tmdy_struct);
static int current_wrd_mode = WRD_TRACE_NOTHING;
static char *save_open_flag;

#define wrdt x_wrdt_mode

WRDTracer wrdt =
{
    "X WRD tracer", 'x',
    0,
    wrdt_open,
    wrdt_apply,
#ifdef ENABLE_SHERRY
    x_sry_wrdt_apply,
#else
    NULL,
#endif
    wrdt_update_events,
    wrdt_start,
    wrdt_end,
    wrdt_close
};

static int inkey_flag;
#define LINEBUF 1024
static char line_buf[LINEBUF];

static int wrdt_open(tmdy_struct_ex_t *tmdy_struct, char *arg)
{
    save_open_flag = arg;
    wrdt.opened = 1;
    return 0;
}

static void wrdt_update_events(tmdy_struct_ex_t *tmdy_struct)
{
    if(current_wrd_mode == WRD_TRACE_MIMPI)
	WinEvent(tmdy_struct);
#ifdef ENABLE_SHERRY
    else if(current_wrd_mode == WRD_TRACE_SHERRY)
	x_sry_event(tmdy_struct);
#endif
}

static int wrdt_start(tmdy_struct_ex_t *tmdy_struct, int mode)
{
    int last_mode = current_wrd_mode;

    current_wrd_mode = mode;
    switch(mode)
    {
      case WRD_TRACE_NOTHING:
	switch(last_mode)
	{
	  case WRD_TRACE_MIMPI:
	    CloseWRDWindow(tmdy_struct);
	    break;
	  case WRD_TRACE_SHERRY:
#ifdef ENABLE_SHERRY
	    CloseSryWindow(tmdy_struct);
#endif /* ENABLE_SHERRY */
	    break;
	}
	break;

      case WRD_TRACE_MIMPI:
#ifdef ENABLE_SHERRY
	if(last_mode == WRD_TRACE_SHERRY)
	    CloseSryWindow(tmdy_struct);
#endif /* ENABLE_SHERRY */
	  if(OpenWRDWindow(tmdy_struct, save_open_flag) == -1)
	  {
	      current_wrd_mode = WRD_TRACE_NOTHING;
	      return -1; /* Error */
	  }
	break;

      case WRD_TRACE_SHERRY:
	if(last_mode == WRD_TRACE_MIMPI)
	    CloseWRDWindow(tmdy_struct);
#ifdef ENABLE_SHERRY
	if(OpenSryWindow(tmdy_struct, save_open_flag) == -1)
	{
	    current_wrd_mode = WRD_TRACE_NOTHING;
	    return -1; /* Error */
	}
#endif /* ENABLE_SHERRY */
	break;
    }

    return 0;
}

static void wrdt_end(tmdy_struct_ex_t *tmdy_struct)
{
  inkey_flag = 0;
  mag_deletetab(tmdy_struct);
  x_VRel(tmdy_struct);
}

static void wrdt_close(tmdy_struct_ex_t *tmdy_struct)
{
    if(wrdt.opened)
    {
	EndWin(tmdy_struct);
#ifdef ENABLE_SHERRY
	x_sry_close(tmdy_struct);
#endif /* ENABLE_SHERRY */
	wrdt.opened = 0;
	inkey_flag = 0;
    }
}

static char *wrd_event2string(tmdy_struct_ex_t *tmdy_struct, int id)
{
    char *name;

    name = TMDY_READMIDI->event2string(tmdy_struct, id);
    if(name != NULL)
	return name + 1;
    return "";
}

static void print_ecmd(tmdy_struct_ex_t *tmdy_struct, char *cmd, int *args, int narg)
{
    char *p;

    p = (char *)TMDY_UTILS->mblock->new_segment(tmdy_struct, &TMDY_COMMON->tmpbuffer, MIN_MBLOCK_SIZE);
    sprintf(p, "^%s(", cmd);

    if(*args == WRD_NOARG)
	strcat(p, "*");
    else
	sprintf(p + strlen(p), "%d", *args);
    args++;
    narg--;
    while(narg > 0)
    {
	if(*args == WRD_NOARG)
	    strcat(p, ",*");
	else
	    sprintf(p + strlen(p), ",%d", *args);
	args++;
	narg--;
    }
    strcat(p, ")");
    TMDY_CONTROLS->ctl->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG, "%s", p);
    TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, &TMDY_COMMON->tmpbuffer);
}

static void load_default_graphics(tmdy_struct_ex_t *tmdy_struct, char *fn)
{
    MBlockList pool;
    char *file, *p;
    int is_caseUpper;
    magdata *m;

    TMDY_UTILS->mblock->init_mblock(tmdy_struct, &pool);
    file = (char *)TMDY_UTILS->mblock->new_segment(tmdy_struct, &pool, strlen(fn) + 5);
    strcpy(file, fn);
    if((p = strrchr(file, '.')) == NULL)
	goto done;
    is_caseUpper = ('A' <= p[1] && p[1] <= 'Z');

    /* try load default MAG file */
    strcpy(p + 1, is_caseUpper ? "MAG" : "mag");
    if((m = mag_create(tmdy_struct, file)) != NULL)
    {
	x_Mag(tmdy_struct, m, WRD_NOARG, WRD_NOARG, 0, 0);
	goto done;
    }

    /* try load default PHO file */
    strcpy(p + 1, is_caseUpper ? "PHO" : "pho");
    x_PLoad(tmdy_struct, file);

 done:
    TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, &pool);
}

static void wrdt_apply(tmdy_struct_ex_t *tmdy_struct, int cmd, int wrd_argc, int wrd_args[])
{
    char *p;
    char *text;
    int i, len;
    static int txtclr_preserve=0;

    if(cmd == WRD_MAGPRELOAD){
	/* Load MAG file */
	magdata *m;
	char *p = wrd_event2string(tmdy_struct, wrd_args[0]);
	TMDY_CONTROLS->ctl->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY, "Loading magfile: [%s]", p);
	m = mag_create(tmdy_struct, p);
	if(m == NULL)
	    TMDY_CONTROLS->ctl->cmsg(tmdy_struct, CMSG_WARNING, VERB_NOISY, "Can't load magfile: %s", p);
	else
	{
	    TMDY_CONTROLS->ctl->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG, "MAG %s: [%d,%d,%d,%d]",
		      p, m->xorig, m->xend, m->yorig, m->yend);
	}
	return;
    }
    if(cmd == WRD_PHOPRELOAD){
	/* Load PHO file - Not implemented */
	return;
    }
#if 0
    if(inkey_flag)
	fprintf(file_out,"* ");
#endif
    switch(cmd)
    {
      case WRD_LYRIC:
	p = wrd_event2string(tmdy_struct, wrd_args[0]);
	len = strlen(p);
	text = (char *)TMDY_UTILS->mblock->new_segment(tmdy_struct, &TMDY_COMMON->tmpbuffer, SAFE_CONVERT_LENGTH(len));

	/*This must be not good thing,but as far as I know no wrd file 
	  written in EUC-JP code can be found,And Hankaku kana required 
	  for layout of charactor and ASCII art.*/

	TMDY_COMMON->code_convert(tmdy_struct, p, text, SAFE_CONVERT_LENGTH(len), "SJIS", "JISK");
	AddLine(tmdy_struct, text,0);
	TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, &TMDY_COMMON->tmpbuffer);
	break;
      case WRD_NL: /* Newline */
	AddLine(tmdy_struct, "\n",0);
	break;
      case WRD_COLOR:
	txtclr_preserve=wrd_args[0];
	/*This length is at most 20 ; this is much lesser than LINEBUF*/
	snprintf(line_buf,LINEBUF,"\033[%dm", txtclr_preserve);
	AddLine(tmdy_struct, line_buf,0);
	break;
      case WRD_END: /* Never call */
	break;
      case WRD_ESC:
	AddLine(tmdy_struct, "\033[",0);
	AddLine(tmdy_struct, wrd_event2string(tmdy_struct, wrd_args[0]),0);
	break;
      case WRD_EXEC:
	/*I don't spawn another program*/
	TMDY_CONTROLS->ctl->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
		  "@EXEC(%s)", wrd_event2string(tmdy_struct, wrd_args[0]));
	break;
      case WRD_FADE:
	TMDY_CONTROLS->ctl->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
		  "@FADE(%d,%d,%d)", wrd_args[0], wrd_args[1], wrd_args[2]);
	x_Fade(tmdy_struct, wrd_args,wrd_argc-1,-1,-1);
	break;
      case WRD_FADESTEP:
#if 0
	TMDY_CONTROLS->ctl->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
		  "@FADESTEP(%d/%d)", wrd_args[0], wrd_args[1]);
#endif
	x_Fade(tmdy_struct, NULL,0,wrd_args[0],wrd_args[1]);
	break;
      case WRD_GCIRCLE:
	TMDY_CONTROLS->ctl->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
		  "@GCIRCLE(%d,%d,%d,%d,%d,%d)",
		  wrd_args[0], wrd_args[1], wrd_args[2], wrd_args[3],
		  wrd_args[4], wrd_args[5]);
	x_GCircle(tmdy_struct, wrd_args,wrd_argc-1);
	break;
      case WRD_GCLS:
	x_Gcls(tmdy_struct, wrd_args[0]);
	break;
      case WRD_GINIT:
	TMDY_CONTROLS->ctl->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG, "@GINIT()");
	break;
      case WRD_GLINE:
	TMDY_CONTROLS->ctl->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
		  "@GLINE(%d,%d,%d,%d,%d,%d,%d)",
	       wrd_args[0], wrd_args[1], wrd_args[2], wrd_args[3], wrd_args[4],
	       wrd_args[5], wrd_args[6]);
	/*GRPH_LINE_MODE=1*/
	x_Gline(tmdy_struct, wrd_args,wrd_argc-1);
	break;
      case WRD_GMODE:
	TMDY_CONTROLS->ctl->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
		  "@GMODE(%d)", wrd_args[0]);
	x_GMode(tmdy_struct, wrd_args[0]);
	break;
      case WRD_GMOVE:
	TMDY_CONTROLS->ctl->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
		  "@GMOVE(%d,%d,%d,%d,%d,%d,%d,%d,%d)",
	       wrd_args[0], wrd_args[1], wrd_args[2], wrd_args[3], wrd_args[4],
	       wrd_args[5], wrd_args[6], wrd_args[7], wrd_args[8]);
	wrd_args[0] &= ~0x7;
	wrd_args[4] &= ~0x7;  
	wrd_args[2] |= 0x7;
 	x_GMove(tmdy_struct, wrd_args[0], wrd_args[1], wrd_args[2],
		wrd_args[3], wrd_args[4],wrd_args[5],
		wrd_args[6], wrd_args[7], wrd_args[8]);
	break;
      case WRD_GON:
	TMDY_CONTROLS->ctl->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
		  "@GON(%d)", wrd_args[0]);
	x_Gon(tmdy_struct, wrd_args[0]);
	break;
      case WRD_GSCREEN:
	TMDY_CONTROLS->ctl->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
		  "@GSCREEN(%d,%d)", wrd_args[0], wrd_args[1]);
	x_Gscreen(tmdy_struct, wrd_args[0],wrd_args[1]);
	break;
      case WRD_INKEY:
	inkey_flag = 1;
	TMDY_CONTROLS->ctl->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG, "@INKEY - begin");
	break;
      case WRD_OUTKEY:
	inkey_flag = 0;
	TMDY_CONTROLS->ctl->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG, "@INKEY - end");
	break;
      case WRD_LOCATE:
	/*Length is At most 40*/
	snprintf(line_buf,LINEBUF,"\033[%d;%dH", wrd_args[1], wrd_args[0]);
	AddLine(tmdy_struct, line_buf,0);
	break;
      case WRD_LOOP: /* Never call */
	break;
      case WRD_MAG:
  	{
	  magdata *m;
	  p = (char *)TMDY_UTILS->mblock->new_segment(tmdy_struct, &TMDY_COMMON->tmpbuffer, MIN_MBLOCK_SIZE);
	  strcpy(p, "@MAG(");
	  strcat(p, wrd_event2string(tmdy_struct, wrd_args[0]));
	  strcat(p, ",");
	  for(i = 1; i < 3; i++)
	    {
	      if(wrd_args[i] == WRD_NOARG)
		strcat(p, "*,");
	      else
		sprintf(p + strlen(p), "%d,", wrd_args[i]);
	    }
	  sprintf(p + strlen(p), "%d,%d)", wrd_args[3], wrd_args[4]);
	  TMDY_CONTROLS->ctl->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG, "%s", p);
	  m=mag_search(tmdy_struct, wrd_event2string(tmdy_struct, wrd_args[0]));
	  if(m!=NULL){
	    x_Mag(tmdy_struct, m,wrd_args[1],wrd_args[2],wrd_args[3],wrd_args[4]);
	  }
	  TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, &TMDY_COMMON->tmpbuffer);
	  break;
  	}
      case WRD_MIDI: /* Never call */
	break;
      case WRD_OFFSET: /* Never call */
	break;
      case WRD_PAL:
	x_Pal(tmdy_struct, wrd_args,wrd_argc-1);
	break;
      case WRD_PALCHG:
	TMDY_CONTROLS->ctl->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
		  "@PALCHG(%s)", wrd_event2string(tmdy_struct, wrd_args[0]));
	break;
      case WRD_PALREV:
	TMDY_CONTROLS->ctl->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
		  "@PALREV(%d)", wrd_args[0]);
	x_Palrev(tmdy_struct, wrd_args[0]);
	break;
      case WRD_PATH:
	TMDY_CONTROLS->ctl->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
		  "@PATH(%s)", wrd_event2string(tmdy_struct, wrd_args[0]));
	TMDY_WRD->wrd_add_path(tmdy_struct, wrd_event2string(tmdy_struct, wrd_args[0]), 0);
	break;
      case WRD_PLOAD:
	TMDY_CONTROLS->ctl->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
		  "@PLOAD(%s)", wrd_event2string(tmdy_struct, wrd_args[0]));
	x_PLoad(tmdy_struct, wrd_event2string(tmdy_struct, wrd_args[0]));
	break;
      case WRD_REM:
	p = wrd_event2string(tmdy_struct, wrd_args[0]);
	len = strlen(p);
	text = (char *)TMDY_UTILS->mblock->new_segment(tmdy_struct, &TMDY_COMMON->tmpbuffer, SAFE_CONVERT_LENGTH(len));
	TMDY_COMMON->code_convert(tmdy_struct, p, text, SAFE_CONVERT_LENGTH(len), NULL, NULL);
	TMDY_CONTROLS->ctl->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG, "@REM %s", text);
	TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, &TMDY_COMMON->tmpbuffer);
	break;
      case WRD_REMARK:
	TMDY_CONTROLS->ctl->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
		  "@REMARK(%s)", wrd_event2string(tmdy_struct, wrd_args[0]));
	break;
      case WRD_REST: /* Never call */
	break;
      case WRD_SCREEN: /* Not supported */
	break;
      case WRD_SCROLL:
	TMDY_CONTROLS->ctl->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
		  "@SCROLL(%d,%d,%d,%d,%d,%d,%d)",
		  wrd_args[0], wrd_args[1], wrd_args[2], wrd_args[3],
		  wrd_args[4], wrd_args[5], wrd_args[6]);
	break;
      case WRD_STARTUP:
	TMDY_CONTROLS->ctl->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
		  "@STARTUP(%d)", wrd_args[0]);
	TMDY_WRD->wrd_init_path(tmdy_struct);
	inkey_flag = 0;
	x_Startup(tmdy_struct, wrd_args[0]);
	load_default_graphics(tmdy_struct, TMDY_READMIDI->current_file_info->filename);
	break;
      case WRD_STOP: /* Never call */
	break;
      case WRD_TCLS:
	TMDY_CONTROLS->ctl->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
		  "@TCLS(%d,%d,%d,%d,%d,%d)",
		  wrd_args[0], wrd_args[1], wrd_args[2], wrd_args[3],
		  wrd_args[4], wrd_args[5]);
	{
	  char fillbuf[1024];
	  int xdiff;
	  xdiff=wrd_args[2]-wrd_args[0]+1;
	  if(xdiff>80)
	    xdiff = 79-wrd_args[0]+1;
	  if(xdiff<0||xdiff>80)
	    break;
	  fillbuf[0]=0x1b;
	  fillbuf[1]='[';
	  fillbuf[2]='s';
	  fillbuf[3]=0;
	  AddLine(tmdy_struct, fillbuf,0);
	  i=wrd_args[4];
	  /*This Length is no more than 1024*/
	  sprintf(fillbuf,"\033[%dm",i);
	  AddLine(tmdy_struct, fillbuf,0);
	  memset(fillbuf,wrd_args[5],xdiff);/*X2-X1*/
	  fillbuf[xdiff]=0;
	  for(i=wrd_args[1];i<=wrd_args[3];i++){/*Y1 to Y2*/
	    snprintf(line_buf,LINEBUF,"\033[%d;%dH",i,wrd_args[0]);
/*X1to....*/
	    AddLine(tmdy_struct, line_buf,0);
	    AddLine(tmdy_struct, fillbuf,0);
	  }
	  fillbuf[0]=0x1b;
	  fillbuf[1]='[';
	  fillbuf[2]='u';
	  fillbuf[3]=0;
	  AddLine(tmdy_struct, fillbuf,0);
	}
	break;
      case WRD_TON:
	TMDY_CONTROLS->ctl->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
		  "@TON(%d)", wrd_args[0]);
	x_Ton(tmdy_struct, wrd_args[0]);
	break;
      case WRD_WAIT: /* Never call */
	break;
      case WRD_WMODE: /* Never call */
	break;

	/* Ensyutsukun */
      case WRD_eFONTM:
	print_ecmd(tmdy_struct, "FONTM", wrd_args, 1);
	break;
      case WRD_eFONTP:
	print_ecmd(tmdy_struct, "FONTP", wrd_args, 4);
	break;
      case WRD_eFONTR:
	print_ecmd(tmdy_struct, "FONTR", wrd_args, 17);
	break;
      case WRD_eGSC:
	print_ecmd(tmdy_struct, "GSC", wrd_args, 1);
	break;
      case WRD_eLINE:
	print_ecmd(tmdy_struct, "LINE", wrd_args, 1);
	break;
      case WRD_ePAL:
	print_ecmd(tmdy_struct, "PAL", wrd_args, 2);
	break;
      case WRD_eREGSAVE:
	print_ecmd(tmdy_struct, "REGSAVE", wrd_args, 17);
	break;
      case WRD_eSCROLL:
	print_ecmd(tmdy_struct, "SCROLL",wrd_args, 2);
	break;
      case WRD_eTEXTDOT:
	print_ecmd(tmdy_struct, "TEXTDOT", wrd_args, 1);
	break;
      case WRD_eTMODE:
	print_ecmd(tmdy_struct, "TMODE", wrd_args, 1);
	break;
      case WRD_eTSCRL:
	print_ecmd(tmdy_struct, "TSCRL", wrd_args, 0);
	break;
      case WRD_eVCOPY:
	print_ecmd(tmdy_struct, "VCOPY", wrd_args, 9);
	x_VCopy(tmdy_struct, wrd_args[0],wrd_args[1],wrd_args[2],wrd_args[3]
		,wrd_args[4],wrd_args[5],wrd_args[6],wrd_args[7],wrd_args[8]);
	break;
      case WRD_eVSGET:
	print_ecmd(tmdy_struct, "VSGET", wrd_args, 4);
	x_VSget(tmdy_struct, wrd_args,4);
	break;
      case WRD_eVSRES:
	print_ecmd(tmdy_struct, "VSRES", wrd_args, 0);
	x_VRel(tmdy_struct);
	break;
      case WRD_eXCOPY:
	if(wrd_argc < 9)
	    break;
	x_XCopy(tmdy_struct, wrd_args[0],wrd_args[1],wrd_args[2],wrd_args[3],wrd_args[4],
		wrd_args[5],wrd_args[6],wrd_args[7],wrd_args[8],
		wrd_args + 9, wrd_argc - 9);
	break;

	/* Extensionals */
      case WRD_START_SKIP:
	if(current_wrd_mode == WRD_TRACE_MIMPI)
	    x_RedrawControl(tmdy_struct, 0);
#ifdef ENABLE_SHERRY
	else if(current_wrd_mode == WRD_TRACE_SHERRY)
	    x_sry_redraw_ctl(tmdy_struct, 0);
#endif /* ENABLE_SHERRY */
	break;
      case WRD_END_SKIP:
	if(current_wrd_mode == WRD_TRACE_MIMPI)
	    x_RedrawControl(tmdy_struct, 1);
#ifdef ENABLE_SHERRY
	else if(current_wrd_mode == WRD_TRACE_SHERRY)
	    x_sry_redraw_ctl(tmdy_struct, 1);
#endif /* ENABLE_SHERRY */
	break;
#ifdef ENABLE_SHERRY
      case WRD_SHERRY_UPDATE:
	x_sry_update(tmdy_struct);
	break;
#endif /* ENABLE_SHERRY */
    }
    WinFlush(tmdy_struct);
}
