/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2002 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    Most of this file is taken from the MikMod sound library, which is
    (c) 1998, 1999 Miodrag Vallat and others - see file libunimod/AUTHORS
    for complete list.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* Interface to libunimod + module player */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#ifdef SUNOS
extern long int random (void);
#endif

#ifndef NO_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif


#include "timidity.h"
#include "common.h"
#include "instrum.h"
#include "playmidi.h"
#include "readmidi.h"
#include "tables.h"
#include "mod.h"
#include "output.h"
//#include "controls.h"
//#include "unimod.h"
//#include "unimod_priv.h"
#include "mod2midi.h"

#include "mod_prv.h"

static BOOL mod_do_play (tmdy_struct_ex_t* tmdy_struct, MODULE *);

int 
load_module_file (tmdy_struct_ex_t* tmdy_struct, struct timidity_file *tf, int mod_type)
{
  MODULE *mf;

#ifdef LOOKUP_HACK
  ML_8bitsamples = 1;
#else
  ML_8bitsamples = 0;
#endif
  ML_monosamples = 1;

  TMDY_UNIMOD->ML_RegisterAllLoaders (tmdy_struct);
  mf = TMDY_UNIMOD->ML_Load (tmdy_struct, tf->url, MOD_NUM_VOICES, 0);
  if (ML_errno)
    return 1;

  TMDY_READMIDI->current_file_info->file_type = mod_type;
  TMDY_MOD2MIDI->load_module_samples(tmdy_struct, mf->samples, mf->numsmp, mod_type == IS_MOD_FILE);
  mod_do_play (tmdy_struct, mf);
  TMDY_UNIMOD->ML_Free (tmdy_struct, mf);
  return 0;
}


int 
get_module_type (tmdy_struct_ex_t* tmdy_struct, char *fn)
{
  if (TMDY_COMMON->check_file_extension(tmdy_struct, fn, ".mod", 1))	/* Most common first */
    return IS_MOD_FILE;

  if (TMDY_COMMON->check_file_extension(tmdy_struct, fn, ".xm", 1)
      || TMDY_COMMON->check_file_extension(tmdy_struct, fn, ".s3m", 1)
      || TMDY_COMMON->check_file_extension(tmdy_struct, fn, ".it", 1)
      || TMDY_COMMON->check_file_extension(tmdy_struct, fn, ".669", 1)	/* Then the others in alphabetic order */
      || TMDY_COMMON->check_file_extension(tmdy_struct, fn, ".amf", 1)
      || TMDY_COMMON->check_file_extension(tmdy_struct, fn, ".dsm", 1)
      || TMDY_COMMON->check_file_extension(tmdy_struct, fn, ".far", 1)
      || TMDY_COMMON->check_file_extension(tmdy_struct, fn, ".gdm", 1)
      || TMDY_COMMON->check_file_extension(tmdy_struct, fn, ".imf", 1)
      || TMDY_COMMON->check_file_extension(tmdy_struct, fn, ".med", 1)
      || TMDY_COMMON->check_file_extension(tmdy_struct, fn, ".mtm", 1)
      || TMDY_COMMON->check_file_extension(tmdy_struct, fn, ".stm", 1)
      || TMDY_COMMON->check_file_extension(tmdy_struct, fn, ".stx", 1)
      || TMDY_COMMON->check_file_extension(tmdy_struct, fn, ".ult", 1)
      || TMDY_COMMON->check_file_extension(tmdy_struct, fn, ".uni", 1))

    return IS_S3M_FILE;

  return IS_OTHER_FILE;
}

char *
get_module_title (tmdy_struct_ex_t* tmdy_struct, struct timidity_file *tf, int mod_type)
{
  return TMDY_UNIMOD->ML_LoadTitle (tmdy_struct, tf->url);
}

/*========== Playing */


//MODULE *pf = NULL;		/* modfile being played */
//static MP_CONTROL *a;		/* current AUDTMP we're working on */
//static MP_STATUS mp;		/* player status */

static const UBYTE VibratoTable[32] =
{
  0, 24, 49, 74, 97, 120, 141, 161, 180, 197, 212, 224, 235, 244, 250, 253,
  255, 253, 250, 244, 235, 224, 212, 197, 180, 161, 141, 120, 97, 74, 49, 24
};

static const UBYTE avibtab[128] =
{
  0, 1, 3, 4, 6, 7, 9, 10, 12, 14, 15, 17, 18, 20, 21, 23,
  24, 25, 27, 28, 30, 31, 32, 34, 35, 36, 38, 39, 40, 41, 42, 44,
  45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 54, 55, 56, 57, 57, 58,
  59, 59, 60, 60, 61, 61, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63,
  64, 63, 63, 63, 63, 63, 63, 63, 62, 62, 62, 61, 61, 60, 60, 59,
  59, 58, 57, 57, 56, 55, 54, 54, 53, 52, 51, 50, 49, 48, 47, 46,
  45, 44, 42, 41, 40, 39, 38, 36, 35, 34, 32, 31, 30, 28, 27, 25,
  24, 23, 21, 20, 18, 17, 15, 14, 12, 10, 9, 7, 6, 4, 3, 1
};

static const SBYTE PanbrelloTable[256] =
{
  0, 2, 3, 5, 6, 8, 9, 11, 12, 14, 16, 17, 19, 20, 22, 23,
  24, 26, 27, 29, 30, 32, 33, 34, 36, 37, 38, 39, 41, 42, 43, 44,
  45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 56, 57, 58, 59,
  59, 60, 60, 61, 61, 62, 62, 62, 63, 63, 63, 64, 64, 64, 64, 64,
  64, 64, 64, 64, 64, 64, 63, 63, 63, 62, 62, 62, 61, 61, 60, 60,
  59, 59, 58, 57, 56, 56, 55, 54, 53, 52, 51, 50, 49, 48, 47, 46,
  45, 44, 43, 42, 41, 39, 38, 37, 36, 34, 33, 32, 30, 29, 27, 26,
  24, 23, 22, 20, 19, 17, 16, 14, 12, 11, 9, 8, 6, 5, 3, 2,
  0, -2, -3, -5, -6, -8, -9, -11, -12, -14, -16, -17, -19, -20, -22, -23,
  -24, -26, -27, -29, -30, -32, -33, -34, -36, -37, -38, -39, -41, -42, -43, -44,
  -45, -46, -47, -48, -49, -50, -51, -52, -53, -54, -55, -56, -56, -57, -58, -59,
  -59, -60, -60, -61, -61, -62, -62, -62, -63, -63, -63, -64, -64, -64, -64, -64,
  -64, -64, -64, -64, -64, -64, -63, -63, -63, -62, -62, -62, -61, -61, -60, -60,
  -59, -59, -58, -57, -56, -56, -55, -54, -53, -52, -51, -50, -49, -48, -47, -46,
  -45, -44, -43, -42, -41, -39, -38, -37, -36, -34, -33, -32, -30, -29, -27, -26,
  -24, -23, -22, -20, -19, -17, -16, -14, -12, -11, -9, -8, -6, -5, -3, -2
};

/* returns a random value between 0 and ceil-1, ceil must be a power of two */
static int 
getrandom (tmdy_struct_ex_t* tmdy_struct, int ceil)
{
#ifdef HAVE_SRANDOM
  return random () & (ceil - 1);
#else
  return (rand () * ceil) / (RAND_MAX + 1.0);
#endif
}

/*      New Note Action Scoring System :
   --------------------------------
   1)   total-volume (fadevol, chanvol, volume) is the main scorer.
   2)   a looping sample is a bonus x2
   3)   a foreground channel is a bonus x4
   4)   an active envelope with keyoff is a handicap -x2                          */
static int 
MP_FindEmptyChannel (tmdy_struct_ex_t* tmdy_struct)
{
  MP_VOICE *a;
  ULONG t, k, tvol, pp;

  a = TMDY_MOD->mp.voice;
  for (t = 0; t < MOD_NUM_VOICES; t++)
    {
      /* allow us to take over a nonexisting sample */
      if (!a->s){
        return t;
      }

      if (((TMDY_MOD->mp.voice[t].kick == KICK_ABSENT) || (TMDY_MOD->mp.voice[t].kick == KICK_ENV)) &&
	  TMDY_MOD2MIDI->Voice_Stopped(tmdy_struct, t)){
        return t;
      }
    }

  tvol = 0xffffffUL;
  t = 0;
  for (k = 0; k < MOD_NUM_VOICES; k++, a++)
    if ((a->kick == KICK_ABSENT) || (a->kick == KICK_ENV))
      {
	pp = a->totalvol << ((a->s->flags & SF_LOOP) ? 1 : 0);
	if ((a->master) && (a == a->master->slave))
	  pp <<= 2;

	if (pp < tvol)
	  {
	    tvol = pp;
	    t = k;
	  }
      }

  if (tvol > 8000 * 7){
    return -1;
  }

  return t;
}

static UWORD GetPeriod (tmdy_struct_ex_t* tmdy_struct, UWORD note, ULONG speed)
{
  if (TMDY_MOD->pf->flags & UF_XMPERIODS){
    return (TMDY_MOD->pf->flags & UF_LINEAR) ? TMDY_UNIMOD->getlinearperiod (tmdy_struct, note, speed) : TMDY_UNIMOD->getlogperiod (tmdy_struct, note, speed);
  }

  return TMDY_UNIMOD->getoldperiod (tmdy_struct, note, speed);
}


static SWORD 
Interpolate (tmdy_struct_ex_t* tmdy_struct, SWORD p, SWORD p1, SWORD p2, SWORD v1, SWORD v2)
{
  if ((p1 == p2) || (p == p1))
    return v1;
  return v1 + ((SLONG) ((p - p1) * (v2 - v1)) / (p2 - p1));
}


static SWORD 
InterpolateEnv (tmdy_struct_ex_t* tmdy_struct, SWORD p, ENVPT * a, ENVPT * b)
{
  return (Interpolate (tmdy_struct, p, a->pos, b->pos, a->val, b->val));
}

static SWORD 
DoPan (tmdy_struct_ex_t* tmdy_struct, SWORD envpan, SWORD pan)
{
  int newpan;

  newpan = pan + (((envpan - PAN_CENTER) * (128 - abs (pan - PAN_CENTER))) / 128);

  return (newpan < PAN_LEFT) ? PAN_LEFT : (newpan > PAN_RIGHT ? PAN_RIGHT : newpan);
}

static void 
StartEnvelope (tmdy_struct_ex_t* tmdy_struct, ENVPR * t, UBYTE flg, UBYTE pts, UBYTE susbeg, UBYTE susend, UBYTE beg, UBYTE end, ENVPT * p, UBYTE keyoff)
{
  t->flg = flg;
  t->pts = pts;
  t->susbeg = susbeg;
  t->susend = susend;
  t->beg = beg;
  t->end = end;
  t->env = p;
  t->p = 0;
  t->a = 0;
  t->b = ((t->flg & EF_SUSTAIN) && (!(keyoff & KEY_OFF))) ? 0 : 1;

  /* Imago Orpheus sometimes stores an extra initial point in the envelope */
  if ((t->pts >= 2) && (t->env[0].pos == t->env[1].pos))
    {
      t->a++;
      t->b++;
    }

  if (t->b >= t->pts)
    t->b = t->pts - 1;
}

/* This procedure processes all envelope types, include volume, pitch, and
   panning.  Envelopes are defined by a set of points, each with a magnitude
   [relating either to volume, panning position, or pitch modifier] and a tick
   position.

   Envelopes work in the following manner:

   (a) Each tick the envelope is moved a point further in its progression. For
   an accurate progression, magnitudes between two envelope points are
   interpolated.

   (b) When progression reaches a defined point on the envelope, values are
   shifted to interpolate between this point and the next, and checks for
   loops or envelope end are done.

   Misc:
   Sustain loops are loops that are only active as long as the keyoff flag is
   clear.  When a volume envelope terminates, so does the current fadeout.  */
static SWORD 
ProcessEnvelope (tmdy_struct_ex_t* tmdy_struct, ENVPR * t, SWORD v, UBYTE keyoff)
{
  if (t->flg & EF_ON)
    {
      UBYTE a, b;		/* actual points in the envelope */
      UWORD p;			/* the 'tick counter' - real point being played */

      a = t->a;
      b = t->b;
      p = t->p;

      /* if sustain loop on one point (XM type), don't move and don't
         interpolate when the point is reached */
      if ((t->flg & EF_SUSTAIN) && (t->susbeg == t->susend) &&
	  (!(keyoff & KEY_OFF)) && (p == t->env[t->susbeg].pos))
	v = t->env[t->susbeg].val;
      else
	{
	  /* compute the current envelope value between points a and b */
	  if (a == b)
	    v = t->env[a].val;
	  else
	    v = InterpolateEnv (tmdy_struct, p, &t->env[a], &t->env[b]);

	  p++;
	  /* pointer reached point b? */
	  if (p >= t->env[b].pos)
	    {
	      a = b++;		/* shift points a and b */

	      /* Check for loops, sustain loops, or end of envelope. */
	      if ((t->flg & EF_SUSTAIN) && (!(keyoff & KEY_OFF)) && (b > t->susend))
		{
		  a = t->susbeg;
		  b = (t->susbeg == t->susend) ? a : a + 1;
		  p = t->env[a].pos;
		}
	      else if ((t->flg & EF_LOOP) && (b > t->end))
		{
		  a = t->beg;
		  b = (t->beg == t->end) ? a : a + 1;
		  p = t->env[a].pos;
		}
	      else
		{
		  if (b >= t->pts)
		    {
		      if ((t->flg & EF_VOLENV) && (TMDY_MOD->mp.channel != -1))
			{
			  TMDY_MOD->mp.voice[TMDY_MOD->mp.channel].keyoff |= KEY_FADE;
			  if (!v)
			    TMDY_MOD->mp.voice[TMDY_MOD->mp.channel].fadevol = 0;
			}
		      b--;
		      p--;
		    }
		}
	    }
	  t->a = a;
	  t->b = b;
	  t->p = p;
	}
    }
  return v;
}

/*========== Protracker effects */

static void 
DoEEffects (tmdy_struct_ex_t* tmdy_struct, UBYTE dat)
{
  UBYTE nib = dat & 0xf;

  switch (dat >> 4)
    {
    case 0x0:			/* hardware filter toggle, not supported */
      break;
    case 0x1:			/* fineslide up */
      if (TMDY_MOD->a->period)
	if (!TMDY_MOD->mp.vbtick)
	  TMDY_MOD->a->tmpperiod -= (nib << 2);
      break;
    case 0x2:			/* fineslide dn */
      if (TMDY_MOD->a->period)
	if (!TMDY_MOD->mp.vbtick)
	  TMDY_MOD->a->tmpperiod += (nib << 2);
      break;
    case 0x3:			/* glissando ctrl */
      TMDY_MOD->a->glissando = nib;
      break;
    case 0x4:			/* set vibrato waveform */
      TMDY_MOD->a->wavecontrol &= 0xf0;
      TMDY_MOD->a->wavecontrol |= nib;
      break;
    case 0x5:			/* set finetune */
      if (TMDY_MOD->a->period)
	{
	  if (TMDY_MOD->pf->flags & UF_XMPERIODS)
	    TMDY_MOD->a->speed = nib + 128;
	  else
	    TMDY_MOD->a->speed = finetune[nib];
	  TMDY_MOD->a->tmpperiod = GetPeriod (tmdy_struct, (UWORD) TMDY_MOD->a->note << 1, TMDY_MOD->a->speed);
	}
      break;
    case 0x6:			/* set patternloop */
      if (TMDY_MOD->mp.vbtick)
	break;
      if (nib)
	{			/* set reppos or repcnt ? */
	  /* set repcnt, so check if repcnt already is set, which means we
	     are already looping */
	  if (TMDY_MOD->a->pat_repcnt)
	    TMDY_MOD->a->pat_repcnt--;	/* already looping, decrease counter */
	  else
	    {
#if 0
	      /* this would make walker.xm, shipped with Xsoundtracker,
	         play correctly, but it's better to remain compatible
	         with FT2 */
	      if ((!(TMDY_MOD->pf->flags & UF_NOWRAP)) || (TMDY_MOD->a->pat_reppos != POS_NONE))
#endif
		TMDY_MOD->a->pat_repcnt = nib;	/* not yet looping, so set repcnt */
	    }

	  if (TMDY_MOD->a->pat_repcnt)
	    {			/* jump to reppos if repcnt>0 */
	      if (TMDY_MOD->a->pat_reppos == POS_NONE)
		TMDY_MOD->a->pat_reppos = TMDY_MOD->mp.patpos - 1;
	      if (TMDY_MOD->a->pat_reppos == -1)
		{
		  TMDY_MOD->mp.pat_repcrazy = 1;
		  TMDY_MOD->mp.patpos = 0;
		}
	      else
		TMDY_MOD->mp.patpos = TMDY_MOD->a->pat_reppos;
	    }
	  else
	    TMDY_MOD->a->pat_reppos = POS_NONE;
	}
      else
	TMDY_MOD->a->pat_reppos = TMDY_MOD->mp.patpos - 1;	/* set reppos - can be (-1) */
      break;
    case 0x7:			/* set tremolo waveform */
      TMDY_MOD->a->wavecontrol &= 0x0f;
      TMDY_MOD->a->wavecontrol |= nib << 4;
      break;
    case 0x8:			/* set panning */
      if (nib <= 8)
	nib <<= 4;
      else
	nib *= 17;
      TMDY_MOD->a->panning = TMDY_MOD->pf->panning[TMDY_MOD->mp.channel] = nib;
      break;
    case 0x9:			/* retrig note */
      /* only retrigger if data nibble > 0 */
      if (nib)
	{
	  if (!TMDY_MOD->a->retrig)
	    {
	      /* when retrig counter reaches 0, reset counter and restart
	         the sample */
	      if (TMDY_MOD->a->period)
		TMDY_MOD->a->kick = KICK_NOTE;
	      TMDY_MOD->a->retrig = nib;
	    }
	  TMDY_MOD->a->retrig--;		/* countdown */
	}
      break;
    case 0xa:			/* fine volume slide up */
      if (TMDY_MOD->mp.vbtick)
	break;
      TMDY_MOD->a->tmpvolume += nib;
      if (TMDY_MOD->a->tmpvolume > 64)
	TMDY_MOD->a->tmpvolume = 64;
      break;
    case 0xb:			/* fine volume slide dn  */
      if (TMDY_MOD->mp.vbtick)
	break;
      TMDY_MOD->a->tmpvolume -= nib;
      if (TMDY_MOD->a->tmpvolume < 0)
	TMDY_MOD->a->tmpvolume = 0;
      break;
    case 0xc:			/* cut note */
      /* When mp.vbtick reaches the cut-note value, turn the volume to
         zero ( Just like on the amiga) */
      if (TMDY_MOD->mp.vbtick >= nib)
	TMDY_MOD->a->tmpvolume = 0;	/* just turn the volume down */
      break;
    case 0xd:			/* note delay */
      /* delay the start of the sample until mp.vbtick==nib */
      if (!TMDY_MOD->mp.vbtick)
	TMDY_MOD->a->notedelay = nib;
      else if (TMDY_MOD->a->notedelay)
	TMDY_MOD->a->notedelay--;
      break;
    case 0xe:			/* pattern delay */
      if (TMDY_MOD->mp.vbtick)
	break;
      if (!TMDY_MOD->mp.patdly2)
	TMDY_MOD->mp.patdly = nib + 1;	/* only once, when vbtick=0 */
      break;
    case 0xf:			/* invert loop, not supported  */
      break;
    }
}

static void 
DoVibrato (tmdy_struct_ex_t* tmdy_struct)
{
  UBYTE q;
  UWORD temp = 0;

  q = (TMDY_MOD->a->vibpos >> 2) & 0x1f;

  switch (TMDY_MOD->a->wavecontrol & 3)
    {
    case 0:			/* sine */
      temp = VibratoTable[q];
      break;
    case 1:			/* ramp down */
      q <<= 3;
      if (TMDY_MOD->a->vibpos < 0)
	q = 255 - q;
      temp = q;
      break;
    case 2:			/* square wave */
      temp = 255;
      break;
    case 3:			/* random wave */
      temp = getrandom (tmdy_struct,256);
      break;
    }

  temp *= TMDY_MOD->a->vibdepth;
  temp >>= 7;
  temp <<= 2;

  if (TMDY_MOD->a->vibpos >= 0)
    TMDY_MOD->a->period = TMDY_MOD->a->tmpperiod + temp;
  else
    TMDY_MOD->a->period = TMDY_MOD->a->tmpperiod - temp;

  if (TMDY_MOD->mp.vbtick)
    TMDY_MOD->a->vibpos += TMDY_MOD->a->vibspd;
}

static void 
DoTremolo (tmdy_struct_ex_t* tmdy_struct)
{
  UBYTE q;
  UWORD temp = 0;

  q = (TMDY_MOD->a->trmpos >> 2) & 0x1f;

  switch ((TMDY_MOD->a->wavecontrol >> 4) & 3)
    {
    case 0:			/* sine */
      temp = VibratoTable[q];
      break;
    case 1:			/* ramp down */
      q <<= 3;
      if (TMDY_MOD->a->trmpos < 0)
	q = 255 - q;
      temp = q;
      break;
    case 2:			/* square wave */
      temp = 255;
      break;
    case 3:			/* random wave */
      temp = getrandom (tmdy_struct,256);
      break;
    }
  temp *= TMDY_MOD->a->trmdepth;
  temp >>= 6;

  if (TMDY_MOD->a->trmpos >= 0)
    {
      TMDY_MOD->a->volume = TMDY_MOD->a->tmpvolume + temp;
      if (TMDY_MOD->a->volume > 64)
	TMDY_MOD->a->volume = 64;
    }
  else
    {
      TMDY_MOD->a->volume = TMDY_MOD->a->tmpvolume - temp;
      if (TMDY_MOD->a->volume < 0)
	TMDY_MOD->a->volume = 0;
    }

  if (TMDY_MOD->mp.vbtick)
    TMDY_MOD->a->trmpos += TMDY_MOD->a->trmspd;
}

static void 
DoVolSlide (tmdy_struct_ex_t* tmdy_struct, UBYTE dat)
{
  if (!TMDY_MOD->mp.vbtick){
    return;
  }

  if (dat & 0xf)
    {
      TMDY_MOD->a->tmpvolume -= (dat & 0x0f);
      if (TMDY_MOD->a->tmpvolume < 0)
	TMDY_MOD->a->tmpvolume = 0;
    }
  else
    {
      TMDY_MOD->a->tmpvolume += (dat >> 4);
      if (TMDY_MOD->a->tmpvolume > 64)
	TMDY_MOD->a->tmpvolume = 64;
    }
}

static void 
DoToneSlide (tmdy_struct_ex_t* tmdy_struct)
{
  if (TMDY_MOD->mp.vbtick)
    {
      int dist;

      /* We have to slide TMDY_MOD->a->period towards TMDY_MOD->a->wantedperiod, so compute the
         difference between those two values */
      dist = TMDY_MOD->a->period - TMDY_MOD->a->wantedperiod;

      /* if they are equal or if portamentospeed is too big ... */
      if ((!dist) || TMDY_MOD->a->portspeed > abs (dist))
	/* ...make tmpperiod equal tperiod */
	TMDY_MOD->a->tmpperiod = TMDY_MOD->a->period = TMDY_MOD->a->wantedperiod;
      else if (dist > 0)
	{
	  TMDY_MOD->a->tmpperiod -= TMDY_MOD->a->portspeed;
	  TMDY_MOD->a->period -= TMDY_MOD->a->portspeed;	/* dist>0, slide up */
	}
      else
	{
	  TMDY_MOD->a->tmpperiod += TMDY_MOD->a->portspeed;
	  TMDY_MOD->a->period += TMDY_MOD->a->portspeed;	/* dist<0, slide down */
	}
    }
  else
    TMDY_MOD->a->tmpperiod = TMDY_MOD->a->period;
}

static void 
DoArpeggio (tmdy_struct_ex_t* tmdy_struct, UBYTE dat)
{
  UBYTE note = TMDY_MOD->a->note;

  if (dat)
    {
      switch (TMDY_MOD->mp.vbtick % 3)
	{
	case 1:
	  note += (dat >> 4);
	  break;
	case 2:
	  note += (dat & 0xf);
	  break;
	}
      TMDY_MOD->a->period = GetPeriod (tmdy_struct, (UWORD) note << 1, TMDY_MOD->a->speed);
      TMDY_MOD->a->ownper = 1;
    }
}

/*========== Scream Tracker effects */

static void 
DoS3MVolSlide (tmdy_struct_ex_t* tmdy_struct, UBYTE inf)
{
  UBYTE lo, hi;

  if (inf)
    TMDY_MOD->a->s3mvolslide = inf;
  else
    inf = TMDY_MOD->a->s3mvolslide;

  lo = inf & 0xf;
  hi = inf >> 4;

  if (!lo)
    {
      if ((TMDY_MOD->mp.vbtick) || (TMDY_MOD->pf->flags & UF_S3MSLIDES))
	TMDY_MOD->a->tmpvolume += hi;
    }
  else if (!hi)
    {
      if ((TMDY_MOD->mp.vbtick) || (TMDY_MOD->pf->flags & UF_S3MSLIDES))
	TMDY_MOD->a->tmpvolume -= lo;
    }
  else if (lo == 0xf)
    {
      if (!TMDY_MOD->mp.vbtick)
	TMDY_MOD->a->tmpvolume += (hi ? hi : 0xf);
    }
  else if (hi == 0xf)
    {
      if (!TMDY_MOD->mp.vbtick)
	TMDY_MOD->a->tmpvolume -= (lo ? lo : 0xf);
    }
  else{
    return;
  }

  if (TMDY_MOD->a->tmpvolume < 0)
    TMDY_MOD->a->tmpvolume = 0;
  else if (TMDY_MOD->a->tmpvolume > 64)
    TMDY_MOD->a->tmpvolume = 64;
}

static void 
DoS3MSlideDn (tmdy_struct_ex_t* tmdy_struct, UBYTE inf)
{
  UBYTE hi, lo;

  if (inf)
    TMDY_MOD->a->slidespeed = inf;
  else
    inf = TMDY_MOD->a->slidespeed;

  hi = inf >> 4;
  lo = inf & 0xf;

  if (hi == 0xf)
    {
      if (!TMDY_MOD->mp.vbtick)
	TMDY_MOD->a->tmpperiod += (UWORD) lo << 2;
    }
  else if (hi == 0xe)
    {
      if (!TMDY_MOD->mp.vbtick)
	TMDY_MOD->a->tmpperiod += lo;
    }
  else
    {
      if (TMDY_MOD->mp.vbtick)
	TMDY_MOD->a->tmpperiod += (UWORD) inf << 2;
    }
}

static void 
DoS3MSlideUp (tmdy_struct_ex_t* tmdy_struct, UBYTE inf)
{
  UBYTE hi, lo;

  if (inf)
    TMDY_MOD->a->slidespeed = inf;
  else
    inf = TMDY_MOD->a->slidespeed;

  hi = inf >> 4;
  lo = inf & 0xf;

  if (hi == 0xf)
    {
      if (!TMDY_MOD->mp.vbtick)
	TMDY_MOD->a->tmpperiod -= (UWORD) lo << 2;
    }
  else if (hi == 0xe)
    {
      if (!TMDY_MOD->mp.vbtick)
	TMDY_MOD->a->tmpperiod -= lo;
    }
  else
    {
      if (TMDY_MOD->mp.vbtick)
	TMDY_MOD->a->tmpperiod -= (UWORD) inf << 2;
    }
}

static void 
DoS3MTremor (tmdy_struct_ex_t* tmdy_struct, UBYTE inf)
{
  UBYTE on, off;

  if (inf)
    TMDY_MOD->a->s3mtronof = inf;
  else
    {
      inf = TMDY_MOD->a->s3mtronof;
      if (!inf){
			return;
	  }
    }

  if (!TMDY_MOD->mp.vbtick){
    return;
  }
  
  on = (inf >> 4) + 1;
  off = (inf & 0xf) + 1;
  TMDY_MOD->a->s3mtremor %= (on + off);
  TMDY_MOD->a->volume = (TMDY_MOD->a->s3mtremor < on) ? TMDY_MOD->a->tmpvolume : 0;
  TMDY_MOD->a->s3mtremor++;
}

static void 
DoS3MRetrig (tmdy_struct_ex_t* tmdy_struct, UBYTE inf)
{
  if (inf)
    {
      TMDY_MOD->a->s3mrtgslide = inf >> 4;
      TMDY_MOD->a->s3mrtgspeed = inf & 0xf;
    }

  /* only retrigger if low nibble > 0 */
  if (TMDY_MOD->a->s3mrtgspeed > 0)
    {
      if (!TMDY_MOD->a->retrig)
	{
	  /* when retrig counter reaches 0, reset counter and restart the
	     sample */
	  if (TMDY_MOD->a->kick != KICK_NOTE)
	    TMDY_MOD->a->kick = KICK_KEYOFF;
	  TMDY_MOD->a->retrig = TMDY_MOD->a->s3mrtgspeed;

	  if ((TMDY_MOD->mp.vbtick) || (TMDY_MOD->pf->flags & UF_S3MSLIDES))
	    {
	      switch (TMDY_MOD->a->s3mrtgslide)
		{
		case 1:
		case 2:
		case 3:
		case 4:
		case 5:
		  TMDY_MOD->a->tmpvolume -= (1 << (TMDY_MOD->a->s3mrtgslide - 1));
		  break;
		case 6:
		  TMDY_MOD->a->tmpvolume = (2 * TMDY_MOD->a->tmpvolume) / 3;
		  break;
		case 7:
		  TMDY_MOD->a->tmpvolume >>= 1;
		  break;
		case 9:
		case 0xa:
		case 0xb:
		case 0xc:
		case 0xd:
		  TMDY_MOD->a->tmpvolume += (1 << (TMDY_MOD->a->s3mrtgslide - 9));
		  break;
		case 0xe:
		  TMDY_MOD->a->tmpvolume = (3 * TMDY_MOD->a->tmpvolume) >> 1;
		  break;
		case 0xf:
		  TMDY_MOD->a->tmpvolume = TMDY_MOD->a->tmpvolume << 1;
		  break;
		}
	      if (TMDY_MOD->a->tmpvolume < 0)
		TMDY_MOD->a->tmpvolume = 0;
	      else if (TMDY_MOD->a->tmpvolume > 64)
		TMDY_MOD->a->tmpvolume = 64;
	    }
	}
      TMDY_MOD->a->retrig--;		/* countdown  */
    }
}

static void 
DoS3MSpeed (tmdy_struct_ex_t* tmdy_struct, UBYTE speed)
{
  if (TMDY_MOD->mp.vbtick || TMDY_MOD->mp.patdly2){
    return;
  }

  if (speed > 128)
    speed -= 128;
  if (speed)
    {
      TMDY_MOD->mp.sngspd = speed;
      TMDY_MOD->mp.vbtick = 0;
    }
}

static void 
DoS3MTempo (tmdy_struct_ex_t* tmdy_struct, UBYTE tempo)
{
  if (TMDY_MOD->mp.vbtick || TMDY_MOD->mp.patdly2){
    return;
  }

  TMDY_MOD->mp.newbpm = (tempo < 32) ? 32 : tempo;
}

static void 
DoS3MFineVibrato (tmdy_struct_ex_t* tmdy_struct)
{
  UBYTE q;
  UWORD temp = 0;

  q = (TMDY_MOD->a->vibpos >> 2) & 0x1f;

  switch (TMDY_MOD->a->wavecontrol & 3)
    {
    case 0:			/* sine */
      temp = VibratoTable[q];
      break;
    case 1:			/* ramp down */
      q <<= 3;
      if (TMDY_MOD->a->vibpos < 0)
	q = 255 - q;
      temp = q;
      break;
    case 2:			/* square wave */
      temp = 255;
      break;
    case 3:			/* random */
      temp = getrandom (tmdy_struct,256);
      break;
    }

  temp *= TMDY_MOD->a->vibdepth;
  temp >>= 8;

  if (TMDY_MOD->a->vibpos >= 0)
    TMDY_MOD->a->period = TMDY_MOD->a->tmpperiod + temp;
  else
    TMDY_MOD->a->period = TMDY_MOD->a->tmpperiod - temp;

  TMDY_MOD->a->vibpos += TMDY_MOD->a->vibspd;
}

static void 
DoS3MTremolo (tmdy_struct_ex_t* tmdy_struct)
{
  UBYTE q;
  UWORD temp = 0;

  q = (TMDY_MOD->a->trmpos >> 2) & 0x1f;

  switch ((TMDY_MOD->a->wavecontrol >> 4) & 3)
    {
    case 0:			/* sine */
      temp = VibratoTable[q];
      break;
    case 1:			/* ramp down */
      q <<= 3;
      if (TMDY_MOD->a->trmpos < 0)
	q = 255 - q;
      temp = q;
      break;
    case 2:			/* square wave */
      temp = 255;
      break;
    case 3:			/* random */
      temp = getrandom (tmdy_struct,256);
      break;
    }

  temp *= TMDY_MOD->a->trmdepth;
  temp >>= 7;

  if (TMDY_MOD->a->trmpos >= 0)
    {
      TMDY_MOD->a->volume = TMDY_MOD->a->tmpvolume + temp;
      if (TMDY_MOD->a->volume > 64)
	TMDY_MOD->a->volume = 64;
    }
  else
    {
      TMDY_MOD->a->volume = TMDY_MOD->a->tmpvolume - temp;
      if (TMDY_MOD->a->volume < 0)
	TMDY_MOD->a->volume = 0;
    }

  if (TMDY_MOD->mp.vbtick)
    TMDY_MOD->a->trmpos += TMDY_MOD->a->trmspd;
}

/*========== Fast Tracker effects */

static void 
DoXMVolSlide (tmdy_struct_ex_t* tmdy_struct, UBYTE inf)
{
  UBYTE lo, hi;

  TMDY_MOD->mp.explicitslides = 2;

  if (inf)
    TMDY_MOD->a->s3mvolslide = inf;
  else
    inf = TMDY_MOD->a->s3mvolslide;

  if (!TMDY_MOD->mp.vbtick){
    return;
  }

  lo = inf & 0xf;
  hi = inf >> 4;

  if (!hi)
    {
      TMDY_MOD->a->tmpvolume -= lo;
      if (TMDY_MOD->a->tmpvolume < 0)
	TMDY_MOD->a->tmpvolume = 0;
    }
  else
    {
      TMDY_MOD->a->tmpvolume += hi;
      if (TMDY_MOD->a->tmpvolume > 64)
	TMDY_MOD->a->tmpvolume = 64;
    }
}

static void 
DoXMGlobalSlide (tmdy_struct_ex_t* tmdy_struct, UBYTE inf)
{
  if (TMDY_MOD->mp.vbtick)
    {
      if (inf)
	TMDY_MOD->mp.globalslide = inf;
      else
	inf = TMDY_MOD->mp.globalslide;
      if (inf & 0xf0)
	inf &= 0xf0;
      TMDY_MOD->mp.volume = TMDY_MOD->mp.volume + ((inf >> 4) - (inf & 0xf)) * 2;

      if (TMDY_MOD->mp.volume < 0)
	TMDY_MOD->mp.volume = 0;
      else if (TMDY_MOD->mp.volume > 128)
	TMDY_MOD->mp.volume = 128;
    }
}

static void 
DoXMPanSlide (tmdy_struct_ex_t* tmdy_struct, UBYTE inf)
{
  UBYTE lo, hi;
  SWORD pan;

  if (inf)
    TMDY_MOD->a->pansspd = inf;
  else
    inf = TMDY_MOD->a->pansspd;

  if (!TMDY_MOD->mp.vbtick){
    return;
  }

  lo = inf & 0xf;
  hi = inf >> 4;

  /* slide right has absolute priority */
  if (hi)
    lo = 0;

  pan = ((TMDY_MOD->a->panning == PAN_SURROUND) ? PAN_CENTER : TMDY_MOD->a->panning) + hi - lo;

  TMDY_MOD->a->panning = (pan < PAN_LEFT) ? PAN_LEFT : (pan > PAN_RIGHT ? PAN_RIGHT : pan);
}

static void 
DoXMExtraFineSlideUp (tmdy_struct_ex_t* tmdy_struct, UBYTE inf)
{
  if (!TMDY_MOD->mp.vbtick)
    {
      TMDY_MOD->a->period -= inf;
      TMDY_MOD->a->tmpperiod -= inf;
    }
}

static void 
DoXMExtraFineSlideDown (tmdy_struct_ex_t* tmdy_struct, UBYTE inf)
{
  if (!TMDY_MOD->mp.vbtick)
    {
      TMDY_MOD->a->period += inf;
      TMDY_MOD->a->tmpperiod += inf;
    }
}

/*========== Impulse Tracker effects */

static void 
DoITChanVolSlide (tmdy_struct_ex_t* tmdy_struct, UBYTE inf)
{
  UBYTE lo, hi;

  if (inf)
    TMDY_MOD->a->chanvolslide = inf;
  inf = TMDY_MOD->a->chanvolslide;

  lo = inf & 0xf;
  hi = inf >> 4;

  if (!hi)
    TMDY_MOD->a->chanvol -= lo;
  else if (!lo)
    {
      TMDY_MOD->a->chanvol += hi;
    }
  else if (hi == 0xf)
    {
      if (!TMDY_MOD->mp.vbtick)
	TMDY_MOD->a->chanvol -= lo;
    }
  else if (lo == 0xf)
    {
      if (!TMDY_MOD->mp.vbtick)
	TMDY_MOD->a->chanvol += hi;
    }

  if (TMDY_MOD->a->chanvol < 0)
    TMDY_MOD->a->chanvol = 0;
  if (TMDY_MOD->a->chanvol > 64)
    TMDY_MOD->a->chanvol = 64;
}

static void 
DoITGlobalSlide (tmdy_struct_ex_t* tmdy_struct, UBYTE inf)
{
  UBYTE lo, hi;

  if (inf)
    TMDY_MOD->mp.globalslide = inf;
  inf = TMDY_MOD->mp.globalslide;

  lo = inf & 0xf;
  hi = inf >> 4;

  if (!lo)
    {
      if (TMDY_MOD->mp.vbtick)
	TMDY_MOD->mp.volume += hi;
    }
  else if (!hi)
    {
      if (TMDY_MOD->mp.vbtick)
	TMDY_MOD->mp.volume -= lo;
    }
  else if (lo == 0xf)
    {
      if (!TMDY_MOD->mp.vbtick)
	TMDY_MOD->mp.volume += hi;
    }
  else if (hi == 0xf)
    {
      if (!TMDY_MOD->mp.vbtick)
	TMDY_MOD->mp.volume -= lo;
    }

  if (TMDY_MOD->mp.volume < 0)
    TMDY_MOD->mp.volume = 0;
  if (TMDY_MOD->mp.volume > 128)
    TMDY_MOD->mp.volume = 128;
}

static void 
DoITPanSlide (tmdy_struct_ex_t* tmdy_struct, UBYTE inf)
{
  UBYTE lo, hi;
  SWORD pan;

  if (inf)
    TMDY_MOD->a->pansspd = inf;
  else
    inf = TMDY_MOD->a->pansspd;

  lo = inf & 0xf;
  hi = inf >> 4;

  pan = (TMDY_MOD->a->panning == PAN_SURROUND) ? PAN_CENTER : TMDY_MOD->a->panning;

  if (!hi)
    pan += lo << 2;
  else if (!lo)
    {
      pan -= hi << 2;
    }
  else if (hi == 0xf)
    {
      if (!TMDY_MOD->mp.vbtick)
	pan += lo << 2;
    }
  else if (lo == 0xf)
    {
      if (!TMDY_MOD->mp.vbtick)
	pan -= hi << 2;
    }
  TMDY_MOD->a->panning =			/*pf->panning[TMDY_MOD->mp.channel]= */
    (pan < PAN_LEFT) ? PAN_LEFT : (pan > PAN_RIGHT ? PAN_RIGHT : pan);
}

static void 
DoITTempo (tmdy_struct_ex_t* tmdy_struct, UBYTE tempo)
{
  SWORD temp = TMDY_MOD->mp.newbpm;

  if (TMDY_MOD->mp.patdly2){
    return;
  }

  if (tempo & 0x10)
    temp += (tempo & 0x0f);
  else
    temp -= tempo;

  TMDY_MOD->mp.newbpm = (temp > 255) ? 255 : (temp < 1 ? 1 : temp);
}

static void 
DoITVibrato (tmdy_struct_ex_t* tmdy_struct)
{
  UBYTE q;
  UWORD temp = 0;

  q = (TMDY_MOD->a->vibpos >> 2) & 0x1f;

  switch (TMDY_MOD->a->wavecontrol & 3)
    {
    case 0:			/* sine */
      temp = VibratoTable[q];
      break;
    case 1:			/* square wave */
      temp = 255;
      break;
    case 2:			/* ramp down */
      q <<= 3;
      if (TMDY_MOD->a->vibpos < 0)
	q = 255 - q;
      temp = q;
      break;
    case 3:			/* random */
      temp = getrandom (tmdy_struct,256);
      break;
    }

  temp *= TMDY_MOD->a->vibdepth;
  temp >>= 8;
  temp <<= 2;

  if (TMDY_MOD->a->vibpos >= 0)
    TMDY_MOD->a->period = TMDY_MOD->a->tmpperiod + temp;
  else
    TMDY_MOD->a->period = TMDY_MOD->a->tmpperiod - temp;

  TMDY_MOD->a->vibpos += TMDY_MOD->a->vibspd;
}

static void 
DoITFineVibrato (tmdy_struct_ex_t* tmdy_struct)
{
  UBYTE q;
  UWORD temp = 0;

  q = (TMDY_MOD->a->vibpos >> 2) & 0x1f;

  switch (TMDY_MOD->a->wavecontrol & 3)
    {
    case 0:			/* sine */
      temp = VibratoTable[q];
      break;
    case 1:			/* square wave */
      temp = 255;
      break;
    case 2:			/* ramp down */
      q <<= 3;
      if (TMDY_MOD->a->vibpos < 0)
	q = 255 - q;
      temp = q;
      break;
    case 3:			/* random */
      temp = getrandom (tmdy_struct,256);
      break;
    }

  temp *= TMDY_MOD->a->vibdepth;
  temp >>= 8;

  if (TMDY_MOD->a->vibpos >= 0)
    TMDY_MOD->a->period = TMDY_MOD->a->tmpperiod + temp;
  else
    TMDY_MOD->a->period = TMDY_MOD->a->tmpperiod - temp;

  TMDY_MOD->a->vibpos += TMDY_MOD->a->vibspd;
}

static void 
DoITTremor (tmdy_struct_ex_t* tmdy_struct, UBYTE inf)
{
  UBYTE on, off;

  if (inf)
    TMDY_MOD->a->s3mtronof = inf;
  else
    {
      inf = TMDY_MOD->a->s3mtronof;
      if (!inf){
		return;
      }
    }

  if (!TMDY_MOD->mp.vbtick){
    return;
  }

  on = (inf >> 4);
  off = (inf & 0xf);

  TMDY_MOD->a->s3mtremor %= (on + off);
  TMDY_MOD->a->volume = (TMDY_MOD->a->s3mtremor < on) ? TMDY_MOD->a->tmpvolume : 0;
  TMDY_MOD->a->s3mtremor++;
}

static void 
DoITPanbrello (tmdy_struct_ex_t* tmdy_struct)
{
  UBYTE q;
  SLONG temp = 0;

  q = TMDY_MOD->a->panbpos;

  switch (TMDY_MOD->a->panbwave)
    {
    case 0:			/* sine */
      temp = PanbrelloTable[q];
      break;
    case 1:			/* square wave */
      temp = (q < 0x80) ? 64 : 0;
      break;
    case 2:			/* ramp down */
      q <<= 3;
      temp = q;
      break;
    case 3:			/* random */
      if (TMDY_MOD->a->panbpos >= TMDY_MOD->a->panbspd)
	{
	  TMDY_MOD->a->panbpos = 0;
	  temp = getrandom (tmdy_struct,256);
	}
    }

  temp *= TMDY_MOD->a->panbdepth;
  temp = (temp / 8) + TMDY_MOD->pf->panning[TMDY_MOD->mp.channel];

  TMDY_MOD->a->panning = (temp < PAN_LEFT) ? PAN_LEFT : (temp > PAN_RIGHT ? PAN_RIGHT : temp);
  TMDY_MOD->a->panbpos += TMDY_MOD->a->panbspd;
}

static void 
DoITToneSlide (tmdy_struct_ex_t* tmdy_struct)
{
  /* if we don't come from another note, ignore the slide and play the note
     as is */
  if (!TMDY_MOD->a->oldnote){
    return;
  }

  if (TMDY_MOD->mp.vbtick)
    {
      int dist;

      /* We have to slide TMDY_MOD->a->period towards TMDY_MOD->a->wantedperiod, compute the
         difference between those two values */
      dist = TMDY_MOD->a->period - TMDY_MOD->a->wantedperiod;

      /* if they are equal or if portamentospeed is too big... */
      if ((!dist) || ((TMDY_MOD->a->portspeed << 2) > abs (dist)))
	/* ... make tmpperiod equal tperiod */
	TMDY_MOD->a->tmpperiod = TMDY_MOD->a->period = TMDY_MOD->a->wantedperiod;
      else if (dist > 0)
	{
	  TMDY_MOD->a->tmpperiod -= TMDY_MOD->a->portspeed << 2;
	  TMDY_MOD->a->period -= TMDY_MOD->a->portspeed << 2;	/* dist>0 slide up */
	}
      else
	{
	  TMDY_MOD->a->tmpperiod += TMDY_MOD->a->portspeed << 2;
	  TMDY_MOD->a->period += TMDY_MOD->a->portspeed << 2;	/* dist<0 slide down */
	}
    }
  else
    TMDY_MOD->a->tmpperiod = TMDY_MOD->a->period;
}

static void DoNNAEffects (tmdy_struct_ex_t* tmdy_struct, UBYTE dat);
/* Impulse/Scream Tracker Sxx effects.
   All Sxx effects share the same memory space. */
static void 
DoSSEffects (tmdy_struct_ex_t* tmdy_struct, UBYTE dat)
{
  UBYTE inf, c;

  inf = dat & 0xf;
  c = dat >> 4;

  if (!dat)
    {
      c = TMDY_MOD->a->sseffect;
      inf = TMDY_MOD->a->ssdata;
    }
  else
    {
      TMDY_MOD->a->sseffect = c;
      TMDY_MOD->a->ssdata = inf;
    }

  switch (c)
    {
    case SS_GLISSANDO:		/* S1x set glissando voice */
      DoEEffects (tmdy_struct, 0x30 | inf);
      break;
    case SS_FINETUNE:		/* S2x set finetune */
      DoEEffects (tmdy_struct, 0x50 | inf);
      break;
    case SS_VIBWAVE:		/* S3x set vibrato waveform */
      DoEEffects (tmdy_struct, 0x40 | inf);
      break;
    case SS_TREMWAVE:		/* S4x set tremolo waveform */
      DoEEffects (tmdy_struct, 0x70 | inf);
      break;
    case SS_PANWAVE:		/* S5x panbrello */
      TMDY_MOD->a->panbwave = inf;
      break;
    case SS_FRAMEDELAY:	/* S6x delay x number of frames (patdly) */
      DoEEffects (tmdy_struct, 0xe0 | inf);
      break;
    case SS_S7EFFECTS:		/* S7x instrument / NNA commands */
      DoNNAEffects (tmdy_struct, inf);
      break;
    case SS_PANNING:		/* S8x set panning position */
      DoEEffects (tmdy_struct, 0x80 | inf);
      break;
    case SS_SURROUND:		/* S9x set surround sound */
      TMDY_MOD->a->panning = TMDY_MOD->pf->panning[TMDY_MOD->mp.channel] = PAN_SURROUND;
      break;
    case SS_HIOFFSET:		/* SAy set high order sample offset yxx00h */
      if (!TMDY_MOD->mp.vbtick)
	{
	  TMDY_MOD->a->hioffset = inf << 16;
	  TMDY_MOD->a->start = TMDY_MOD->a->hioffset | TMDY_MOD->a->soffset;

	  if ((TMDY_MOD->a->s) && (TMDY_MOD->a->start > TMDY_MOD->a->s->length))
	    TMDY_MOD->a->start = TMDY_MOD->a->s->flags & (SF_LOOP | SF_BIDI) ? TMDY_MOD->a->s->loopstart : TMDY_MOD->a->s->length;
	}
      break;
    case SS_PATLOOP:		/* SBx pattern loop */
      DoEEffects (tmdy_struct, 0x60 | inf);
      break;
    case SS_NOTECUT:		/* SCx notecut */
      DoEEffects (tmdy_struct, 0xC0 | inf);
      break;
    case SS_NOTEDELAY:		/* SDx notedelay */
      DoEEffects (tmdy_struct, 0xD0 | inf);
      break;
    case SS_PATDELAY:		/* SEx patterndelay */
      DoEEffects (tmdy_struct, 0xE0 | inf);
      break;
    }
}

/* Impulse Tracker Volume/Pan Column effects.
   All volume/pan column effects share the same memory space. */
static void 
DoVolEffects (tmdy_struct_ex_t* tmdy_struct, UBYTE c)
{
  UBYTE inf = TMDY_UNIMOD->UniGetByte (tmdy_struct);

  if ((!c) && (!inf))
    {
      c = TMDY_MOD->a->voleffect;
      inf = TMDY_MOD->a->voldata;
    }
  else
    {
      TMDY_MOD->a->voleffect = c;
      TMDY_MOD->a->voldata = inf;
    }

  if (c)
    switch (c)
      {
      case VOL_VOLUME:
	if (TMDY_MOD->mp.vbtick)
	  break;
	if (inf > 64)
	  inf = 64;
	TMDY_MOD->a->tmpvolume = inf;
	break;
      case VOL_PANNING:
	TMDY_MOD->a->panning = /*pf->panning[TMDY_MOD->mp.channel]= */ inf;
	break;
      case VOL_VOLSLIDE:
	DoS3MVolSlide (tmdy_struct, inf);
	break;
      case VOL_PITCHSLIDEDN:
	if (TMDY_MOD->a->period)
	  DoS3MSlideDn (tmdy_struct, inf);
	break;
      case VOL_PITCHSLIDEUP:
	if (TMDY_MOD->a->period)
	  DoS3MSlideUp (tmdy_struct, inf);
	break;
      case VOL_PORTAMENTO:
	if (inf)
	  TMDY_MOD->a->slidespeed = inf;
	if (TMDY_MOD->a->period)
	  {
	    if ((!TMDY_MOD->mp.vbtick) || (TMDY_MOD->a->newsamp))
	      {
		TMDY_MOD->a->kick = KICK_NOTE;
		TMDY_MOD->a->start = -1;
	      }
	    else
	      TMDY_MOD->a->kick = (TMDY_MOD->a->kick == KICK_NOTE) ? KICK_ENV : KICK_ABSENT;
	    DoITToneSlide (tmdy_struct);
	    TMDY_MOD->a->ownper = 1;
	  }
	break;
      case VOL_VIBRATO:
	if (!TMDY_MOD->mp.vbtick)
	  {
	    if (inf & 0x0f)
	      TMDY_MOD->a->vibdepth = inf & 0xf;
	    if (inf & 0xf0)
	      TMDY_MOD->a->vibspd = (inf & 0xf0) >> 2;
	  }
	if (TMDY_MOD->a->period)
	  {
	    DoITVibrato (tmdy_struct);
	    TMDY_MOD->a->ownper = 1;
	  }
	break;
      }
}

/*========== UltraTracker effects */

static void 
DoULTSampleOffset (tmdy_struct_ex_t* tmdy_struct)
{
  UWORD offset = TMDY_UNIMOD->UniGetWord (tmdy_struct);

  if (offset)
    TMDY_MOD->a->ultoffset = offset;

  TMDY_MOD->a->start = TMDY_MOD->a->ultoffset << 2;
  if ((TMDY_MOD->a->s) && (TMDY_MOD->a->start > TMDY_MOD->a->s->length))
    TMDY_MOD->a->start = TMDY_MOD->a->s->flags & (SF_LOOP | SF_BIDI) ? TMDY_MOD->a->s->loopstart : TMDY_MOD->a->s->length;
}

/*========== OctaMED effects */

static void 
DoMEDSpeed (tmdy_struct_ex_t* tmdy_struct)
{
  UWORD speed = TMDY_UNIMOD->UniGetWord (tmdy_struct);

  TMDY_MOD->mp.newbpm = speed;
}

/*========== General player functions */

static void 
pt_playeffects (tmdy_struct_ex_t* tmdy_struct)
{
  UBYTE dat, c, oldc = 0;

  while ((c = TMDY_UNIMOD->UniGetByte (tmdy_struct)))
    {
      int oldsliding = TMDY_MOD->a->sliding;

      TMDY_MOD->a->sliding = 0;

      /* libunimod doesn't *quite* do Ultimate Soundtracker portas correctly */
      if (strcmp(of.modtype, "Ultimate Soundtracker") == 0)
        {
      if (c == 5 && oldc == 4)
        {
          oldc = 5;
          TMDY_MOD->a->sliding = oldsliding;
          TMDY_UNIMOD->UniSkipOpcode (tmdy_struct, c);
          continue;
        }
        oldc = c;
        if (c == 3 || c == 4)
            c++;
      }

      switch (c)
	{
	case UNI_PTEFFECT0:
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if (!TMDY_MOD->mp.vbtick)
	    {
	      if ((!dat) && (TMDY_MOD->pf->flags & UF_ARPMEM))
		dat = TMDY_MOD->a->arpmem;
	      else
	        TMDY_MOD->a->arpmem = dat;
	    }
	  if (TMDY_MOD->a->period)
	    DoArpeggio (tmdy_struct, TMDY_MOD->a->arpmem);
	  break;
	case UNI_PTEFFECT1:
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if ((!TMDY_MOD->mp.vbtick) && (dat))
	    TMDY_MOD->a->slidespeed = (UWORD) dat << 2;
	  if (TMDY_MOD->a->period)
	    if (TMDY_MOD->mp.vbtick)
	      TMDY_MOD->a->tmpperiod -= TMDY_MOD->a->slidespeed;
	  break;
	case UNI_PTEFFECT2:
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if ((!TMDY_MOD->mp.vbtick) && (dat))
	    TMDY_MOD->a->slidespeed = (UWORD) dat << 2;
	  if (TMDY_MOD->a->period)
	    if (TMDY_MOD->mp.vbtick)
	      TMDY_MOD->a->tmpperiod += TMDY_MOD->a->slidespeed;
	  break;
	case UNI_PTEFFECT3:
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if ((!TMDY_MOD->mp.vbtick) && (dat))
	    TMDY_MOD->a->portspeed = (UWORD) dat << 2;
	  if (TMDY_MOD->a->period)
	    {
	      if (!TMDY_MOD->a->fadevol)
		TMDY_MOD->a->kick = (TMDY_MOD->a->kick == KICK_NOTE) ? KICK_NOTE : KICK_KEYOFF;
	      else
		TMDY_MOD->a->kick = (TMDY_MOD->a->kick == KICK_NOTE) ? KICK_ENV : KICK_ABSENT;
	      DoToneSlide (tmdy_struct);
	      TMDY_MOD->a->ownper = 1;
	    }
	  break;
	case UNI_PTEFFECT4:
	case UNI_XMEFFECT4:
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if (!TMDY_MOD->mp.vbtick)
	    {
	      if (dat & 0x0f)
		TMDY_MOD->a->vibdepth = dat & 0xf;
	      if (dat & 0xf0)
		TMDY_MOD->a->vibspd = (dat & 0xf0) >> 2;
	    }
	  else if (TMDY_MOD->a->period)
	    {
	      DoVibrato (tmdy_struct);
	      TMDY_MOD->a->ownper = 1;
	    }
	  break;
	case UNI_PTEFFECT5:
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if (TMDY_MOD->a->period)
	    {
	      if (!TMDY_MOD->a->fadevol)
		TMDY_MOD->a->kick = (TMDY_MOD->a->kick == KICK_NOTE) ? KICK_NOTE : KICK_KEYOFF;
	      else
		TMDY_MOD->a->kick = (TMDY_MOD->a->kick == KICK_NOTE) ? KICK_ENV : KICK_ABSENT;
	      DoToneSlide (tmdy_struct);
	      TMDY_MOD->a->ownper = 1;
	    }
	  DoVolSlide (tmdy_struct, dat);
	  break;
	case UNI_PTEFFECT6:
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if ((TMDY_MOD->a->period) && (TMDY_MOD->mp.vbtick))
	    {
	      DoVibrato (tmdy_struct);
	      TMDY_MOD->a->ownper = 1;
	    }
	  DoVolSlide (tmdy_struct, dat);
	  break;
	case UNI_PTEFFECT7:
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if (!TMDY_MOD->mp.vbtick)
	    {
	      if (dat & 0x0f)
		TMDY_MOD->a->trmdepth = dat & 0xf;
	      if (dat & 0xf0)
		TMDY_MOD->a->trmspd = (dat & 0xf0) >> 2;
	    }
	  if (TMDY_MOD->a->period)
	    {
	      DoTremolo (tmdy_struct);
	      TMDY_MOD->a->ownvol = 1;
	    }
	  break;
	case UNI_PTEFFECT8:
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  TMDY_MOD->a->panning = TMDY_MOD->pf->panning[TMDY_MOD->mp.channel] = dat;
	  break;
	case UNI_PTEFFECT9:
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if (!TMDY_MOD->mp.vbtick)
	    {
	      if (dat)
		TMDY_MOD->a->soffset = (UWORD) dat << 8;
	      TMDY_MOD->a->start = TMDY_MOD->a->hioffset | TMDY_MOD->a->soffset;

	      if ((TMDY_MOD->a->s) && (TMDY_MOD->a->start > TMDY_MOD->a->s->length))
		TMDY_MOD->a->start = TMDY_MOD->a->s->flags & (SF_LOOP | SF_BIDI) ? TMDY_MOD->a->s->loopstart : TMDY_MOD->a->s->length;
	    }
	  break;
	case UNI_PTEFFECTA:
	  DoVolSlide (tmdy_struct, TMDY_UNIMOD->UniGetByte (tmdy_struct));
	  break;
	case UNI_PTEFFECTB:
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if ((TMDY_MOD->mp.vbtick) || (TMDY_MOD->mp.patdly2))
	    break;
	  /* Vincent Voois uses a nasty trick in "Universal Bolero" */
	  if (dat == TMDY_MOD->mp.sngpos && TMDY_MOD->mp.patbrk == TMDY_MOD->mp.patpos)
	    break;
	  if ((!TMDY_MOD->mp.patbrk) && ((dat < TMDY_MOD->mp.sngpos) ||
			  ((TMDY_MOD->mp.sngpos == TMDY_MOD->pf->numpos - 1) && (!TMDY_MOD->mp.patbrk)) ||
			   ((dat == TMDY_MOD->mp.sngpos) && (TMDY_MOD->pf->flags & UF_NOWRAP))))
	    {
	      /* if we don't loop, better not to skip the end of the
	         pattern, after all... so:
	         TMDY_MOD->mp.patbrk=0; */
	      TMDY_MOD->mp.posjmp = 3;
	    }
	  else
	    {
	      /* if we were fading, adjust... */
	      if (TMDY_MOD->mp.sngpos == (TMDY_MOD->pf->numpos - 1))
		TMDY_MOD->mp.volume = TMDY_MOD->pf->initvolume > 128 ? 128 : TMDY_MOD->pf->initvolume;
	      
	      TMDY_MOD->mp.sngpos = dat;
	      TMDY_MOD->mp.posjmp = 2;
	      TMDY_MOD->mp.patpos = 0;
	    }
	  break;
	case UNI_PTEFFECTC:
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if (TMDY_MOD->mp.vbtick)
	    break;
	  if (dat == (UBYTE) - 1)
	    TMDY_MOD->a->anote = dat = 0;	/* note cut */
	  else if (dat > 64)
	    dat = 64;
	  TMDY_MOD->a->tmpvolume = dat;
	  break;
	case UNI_PTEFFECTD:
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if ((TMDY_MOD->mp.vbtick) || (TMDY_MOD->mp.patdly2))
	    break;
	  if ((TMDY_MOD->pf->positions[TMDY_MOD->mp.sngpos] != 255) &&
	      (dat > TMDY_MOD->pf->pattrows[TMDY_MOD->pf->positions[TMDY_MOD->mp.sngpos]]))
	    dat = TMDY_MOD->pf->pattrows[TMDY_MOD->pf->positions[TMDY_MOD->mp.sngpos]];
	  TMDY_MOD->mp.patbrk = dat;
	  if (!TMDY_MOD->mp.posjmp)
	    {
	      /* don't ask me to explain this code - it makes
	       * backwards.s3m and children.xm (heretic's version) play
	       * correctly, among others. Take that for granted, or write
	       * the page of comments yourself... you might need some
	       * aspirin - Miod */
	      if ((TMDY_MOD->mp.sngpos == TMDY_MOD->pf->numpos - 1) && (dat) &&
		  ((TMDY_MOD->pf->positions[TMDY_MOD->mp.sngpos] == (TMDY_MOD->pf->numpat - 1)
		    && (TMDY_MOD->pf->flags & UF_NOWRAP))))
		{
		  /* printf("%d -- Pattern 0!\n", __LINE__); */
		  TMDY_MOD->mp.sngpos = 0;
		  TMDY_MOD->mp.posjmp = 2;
		}
	      else
		TMDY_MOD->mp.posjmp = 3;
	    }
	  break;
	case UNI_PTEFFECTE:
	  DoEEffects (tmdy_struct, TMDY_UNIMOD->UniGetByte (tmdy_struct));
	  break;
	case UNI_PTEFFECTF:
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if (TMDY_MOD->mp.vbtick || TMDY_MOD->mp.patdly2)
	    break;
	  if (dat > 0x20)
	    TMDY_MOD->mp.newbpm = dat;
	  else if (dat)
	    {
	      TMDY_MOD->mp.sngspd = (dat > 32) ? 32 : dat;
	      TMDY_MOD->mp.vbtick = 0;
	    }
	  break;
	case UNI_S3MEFFECTA:
	  DoS3MSpeed (tmdy_struct, TMDY_UNIMOD->UniGetByte (tmdy_struct));
	  break;
	case UNI_S3MEFFECTD:
	  DoS3MVolSlide (tmdy_struct, TMDY_UNIMOD->UniGetByte (tmdy_struct));
	  break;
	case UNI_S3MEFFECTE:
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if (TMDY_MOD->a->period)
	    DoS3MSlideDn (tmdy_struct, dat);
	  break;
	case UNI_S3MEFFECTF:
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if (TMDY_MOD->a->period)
	    DoS3MSlideUp (tmdy_struct, dat);
	  break;
	case UNI_S3MEFFECTI:
	  DoS3MTremor (tmdy_struct, TMDY_UNIMOD->UniGetByte (tmdy_struct));
	  TMDY_MOD->a->ownvol = 1;
	  break;
	case UNI_S3MEFFECTQ:
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if (TMDY_MOD->a->period)
	    DoS3MRetrig (tmdy_struct, dat);
	  break;
	case UNI_S3MEFFECTR:
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if (!TMDY_MOD->mp.vbtick)
	    {
	      if (dat & 0x0f)
		TMDY_MOD->a->trmdepth = dat & 0xf;
	      if (dat & 0xf0)
		TMDY_MOD->a->trmspd = (dat & 0xf0) >> 2;
	    }
	  DoS3MTremolo (tmdy_struct);
	  TMDY_MOD->a->ownvol = 1;
	  break;
	case UNI_S3MEFFECTT:
	  DoS3MTempo (tmdy_struct, TMDY_UNIMOD->UniGetByte (tmdy_struct));
	  break;
	case UNI_S3MEFFECTU:
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if (!TMDY_MOD->mp.vbtick)
	    {
	      if (dat & 0x0f)
		TMDY_MOD->a->vibdepth = dat & 0xf;
	      if (dat & 0xf0)
		TMDY_MOD->a->vibspd = (dat & 0xf0) >> 2;
	    }
	  else if (TMDY_MOD->a->period)
	    {
	      DoS3MFineVibrato (tmdy_struct);
	      TMDY_MOD->a->ownper = 1;
	    }
	  break;
	case UNI_KEYOFF:
	  TMDY_MOD->a->keyoff |= KEY_OFF;
	  if ((!(TMDY_MOD->a->volflg & EF_ON)) || (TMDY_MOD->a->volflg & EF_LOOP))
	    TMDY_MOD->a->keyoff = KEY_KILL;
	  break;
	case UNI_KEYFADE:
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if ((TMDY_MOD->mp.vbtick >= dat) || (TMDY_MOD->mp.vbtick == TMDY_MOD->mp.sngspd - 1))
	    {
	      TMDY_MOD->a->keyoff = KEY_KILL;
	      if (!(TMDY_MOD->a->volflg & EF_ON))
		TMDY_MOD->a->fadevol = 0;
	    }
	  break;
	case UNI_VOLEFFECTS:
	  DoVolEffects (tmdy_struct, TMDY_UNIMOD->UniGetByte (tmdy_struct));
	  break;
	case UNI_XMEFFECTA:
	  DoXMVolSlide (tmdy_struct, TMDY_UNIMOD->UniGetByte (tmdy_struct));
	  break;
	case UNI_XMEFFECTE1:	/* XM fineslide up */
	  if (TMDY_MOD->mp.vbtick)
	    break;
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if (!TMDY_MOD->mp.vbtick)
	    {
	      if (dat)
		TMDY_MOD->a->fportupspd = dat;
	      if (TMDY_MOD->a->period)
		TMDY_MOD->a->tmpperiod -= (TMDY_MOD->a->fportupspd << 2);
	    }
	  break;
	case UNI_XMEFFECTE2:	/* XM fineslide dn */
	  if (TMDY_MOD->mp.vbtick)
	    break;
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if (!TMDY_MOD->mp.vbtick)
	    {
	      if (dat)
		TMDY_MOD->a->fportdnspd = dat;
	      if (TMDY_MOD->a->period)
		TMDY_MOD->a->tmpperiod += (TMDY_MOD->a->fportdnspd << 2);
	    }
	  break;
	case UNI_XMEFFECTEA:	/* fine volume slide up */
	  if (TMDY_MOD->mp.vbtick)
	    break;
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if (dat)
	    TMDY_MOD->a->fslideupspd = dat;
	  TMDY_MOD->a->tmpvolume += TMDY_MOD->a->fslideupspd;
	  if (TMDY_MOD->a->tmpvolume > 64)
	    TMDY_MOD->a->tmpvolume = 64;
	  break;
	case UNI_XMEFFECTEB:	/* fine volume slide dn */
	  if (TMDY_MOD->mp.vbtick)
	    break;
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if (dat)
	    TMDY_MOD->a->fslidednspd = dat;
	  TMDY_MOD->a->tmpvolume -= TMDY_MOD->a->fslidednspd;
	  if (TMDY_MOD->a->tmpvolume < 0)
	    TMDY_MOD->a->tmpvolume = 0;
	  break;
	case UNI_XMEFFECTG:
	  TMDY_MOD->mp.volume = TMDY_UNIMOD->UniGetByte (tmdy_struct) << 1;
	  if (TMDY_MOD->mp.volume > 128)
	    TMDY_MOD->mp.volume = 128;
	  break;
	case UNI_XMEFFECTH:
	  DoXMGlobalSlide (tmdy_struct, TMDY_UNIMOD->UniGetByte (tmdy_struct));
	  break;
	case UNI_XMEFFECTL:
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if ((!TMDY_MOD->mp.vbtick) && (TMDY_MOD->a->i))
	    {
	      UWORD points;
	      INSTRUMENT *i = TMDY_MOD->a->i;
	      MP_VOICE *aout;

	      if ((aout = TMDY_MOD->a->slave))
		{
		  if (aout->venv.env) {
		    points = i->volenv[i->volpts - 1].pos;
		    aout->venv.p = aout->venv.env[(dat > points) ? points : dat].pos;
		  }
		  if (aout->penv.env) {
		    points = i->panenv[i->panpts - 1].pos;
		    aout->penv.p = aout->penv.env[(dat > points) ? points : dat].pos;
		  }
		}
	    }
	  break;
	case UNI_XMEFFECTP:
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  DoXMPanSlide (tmdy_struct, dat);
	  break;
	case UNI_XMEFFECTX1:
	  if (TMDY_MOD->mp.vbtick)
	    break;
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if (dat)
	    TMDY_MOD->a->ffportupspd = dat;
	  else
	    dat = TMDY_MOD->a->ffportupspd;
	  if (TMDY_MOD->a->period)
	    {
	      DoXMExtraFineSlideUp (tmdy_struct, dat);
	      TMDY_MOD->a->ownper = 1;
	    }
	  break;
	case UNI_XMEFFECTX2:
	  if (TMDY_MOD->mp.vbtick)
	    break;
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if (dat)
	    TMDY_MOD->a->ffportdnspd = dat;
	  else
	    dat = TMDY_MOD->a->ffportdnspd;
	  if (TMDY_MOD->a->period)
	    {
	      DoXMExtraFineSlideDown (tmdy_struct, dat);
	      TMDY_MOD->a->ownper = 1;
	    }
	  break;
	case UNI_ITEFFECTG:
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if (dat)
	    {
	      TMDY_MOD->a->portspeed = dat;
	    }
	  if (TMDY_MOD->a->period)
	    {
	      if ((!TMDY_MOD->mp.vbtick) && (TMDY_MOD->a->newsamp))
		{
		  TMDY_MOD->a->kick = KICK_NOTE;
		  TMDY_MOD->a->start = -1;
		}
	      else
		TMDY_MOD->a->kick = (TMDY_MOD->a->kick == KICK_NOTE) ? KICK_ENV : KICK_ABSENT;
	      DoITToneSlide (tmdy_struct);
	      TMDY_MOD->a->ownper = 1;
	    }
	  break;
	case UNI_ITEFFECTH:	/* IT vibrato */
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if (!TMDY_MOD->mp.vbtick)
	    {
	      if (dat & 0x0f)
		TMDY_MOD->a->vibdepth = dat & 0xf;
	      if (dat & 0xf0)
		TMDY_MOD->a->vibspd = (dat & 0xf0) >> 2;
	    }
	  if (TMDY_MOD->a->period)
	    {
	      DoITVibrato (tmdy_struct);
	      TMDY_MOD->a->ownper = 1;
	    }
	  break;
	case UNI_ITEFFECTI:	/* IT tremor */
	  DoITTremor (tmdy_struct, TMDY_UNIMOD->UniGetByte (tmdy_struct));
	  TMDY_MOD->a->ownvol = 1;
	  break;
	case UNI_ITEFFECTM:
	  TMDY_MOD->a->chanvol = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if (TMDY_MOD->a->chanvol > 64)
	    TMDY_MOD->a->chanvol = 64;
	  else if (TMDY_MOD->a->chanvol < 0)
	    TMDY_MOD->a->chanvol = 0;
	  break;
	case UNI_ITEFFECTN:	/* slide / fineslide channel volume */
	  DoITChanVolSlide (tmdy_struct, TMDY_UNIMOD->UniGetByte (tmdy_struct));
	  break;
	case UNI_ITEFFECTP:	/* slide / fineslide channel panning */
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  DoITPanSlide (tmdy_struct, dat);
	  break;
	case UNI_ITEFFECTT:	/* slide / fineslide tempo */
	  DoITTempo (tmdy_struct, TMDY_UNIMOD->UniGetByte (tmdy_struct));
	  break;
	case UNI_ITEFFECTU:	/* fine vibrato */
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if (!TMDY_MOD->mp.vbtick)
	    {
	      if (dat & 0x0f)
		TMDY_MOD->a->vibdepth = dat & 0xf;
	      if (dat & 0xf0)
		TMDY_MOD->a->vibspd = (dat & 0xf0) >> 2;
	    }
	  if (TMDY_MOD->a->period)
	    {
	      DoITFineVibrato (tmdy_struct);
	      TMDY_MOD->a->ownper = 1;
	    }
	  break;
	case UNI_ITEFFECTW:	/* slide / fineslide global volume */
	  DoITGlobalSlide (tmdy_struct, TMDY_UNIMOD->UniGetByte (tmdy_struct));
	  break;
	case UNI_ITEFFECTY:	/* panbrello */
	  dat = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	  if (!TMDY_MOD->mp.vbtick)
	    {
	      if (dat & 0x0f)
		TMDY_MOD->a->panbdepth = (dat & 0xf);
	      if (dat & 0xf0)
		TMDY_MOD->a->panbspd = (dat & 0xf0) >> 4;
	    }
	  DoITPanbrello (tmdy_struct);
	  break;
	case UNI_ITEFFECTS0:
	  DoSSEffects (tmdy_struct, TMDY_UNIMOD->UniGetByte (tmdy_struct));
	  break;
	case UNI_ITEFFECTZ:
	  /* FIXME not yet implemented */
	  TMDY_UNIMOD->UniSkipOpcode (tmdy_struct, UNI_ITEFFECTZ);
	  break;
	case UNI_ULTEFFECT9:
	  DoULTSampleOffset (tmdy_struct);
	  break;
	case UNI_MEDSPEED:
	  DoMEDSpeed (tmdy_struct);
	  break;
	case UNI_MEDEFFECTF1:
	  DoEEffects (tmdy_struct, 0x90 | (TMDY_MOD->mp.sngspd / 2));
	  break;
	case UNI_MEDEFFECTF2:
	  DoEEffects (tmdy_struct, 0xd0 | (TMDY_MOD->mp.sngspd / 2));
	  break;
	case UNI_MEDEFFECTF3:
	  DoEEffects (tmdy_struct, 0x90 | (TMDY_MOD->mp.sngspd / 3));
	  break;
	/* case UNI_NOTE: */
	/* case UNI_INSTRUMENT: */
	default:
	  TMDY_MOD->a->sliding = oldsliding;
	  TMDY_UNIMOD->UniSkipOpcode (tmdy_struct, c);
	  break;
	}
    }
}

static void 
DoNNAEffects (tmdy_struct_ex_t* tmdy_struct, UBYTE dat)
{
  int t;
  MP_VOICE *aout;

  dat &= 0xf;
  aout = (TMDY_MOD->a->slave) ? TMDY_MOD->a->slave : NULL;

  switch (dat)
    {
    case 0x0:			/* past note cut */
      for (t = 0; t < MOD_NUM_VOICES; t++)
	if (TMDY_MOD->mp.voice[t].master == TMDY_MOD->a)
	  TMDY_MOD->mp.voice[t].fadevol = 0;
      break;
    case 0x1:			/* past note off */
      for (t = 0; t < MOD_NUM_VOICES; t++)
	if (TMDY_MOD->mp.voice[t].master == TMDY_MOD->a)
	  {
	    TMDY_MOD->mp.voice[t].keyoff |= KEY_OFF;
	    if ((!(TMDY_MOD->mp.voice[t].venv.flg & EF_ON)) ||
		(TMDY_MOD->mp.voice[t].venv.flg & EF_LOOP))
	      TMDY_MOD->mp.voice[t].keyoff = KEY_KILL;
	  }
      break;
    case 0x2:			/* past note fade */
      for (t = 0; t < MOD_NUM_VOICES; t++)
	if (TMDY_MOD->mp.voice[t].master == TMDY_MOD->a)
	  TMDY_MOD->mp.voice[t].keyoff |= KEY_FADE;
      break;
    case 0x3:			/* set NNA note cut */
      TMDY_MOD->a->nna = (TMDY_MOD->a->nna & ~NNA_MASK) | NNA_CUT;
      break;
    case 0x4:			/* set NNA note continue */
      TMDY_MOD->a->nna = (TMDY_MOD->a->nna & ~NNA_MASK) | NNA_CONTINUE;
      break;
    case 0x5:			/* set NNA note off */
      TMDY_MOD->a->nna = (TMDY_MOD->a->nna & ~NNA_MASK) | NNA_OFF;
      break;
    case 0x6:			/* set NNA note fade */
      TMDY_MOD->a->nna = (TMDY_MOD->a->nna & ~NNA_MASK) | NNA_FADE;
      break;
    case 0x7:			/* disable volume envelope */
      if (aout)
	aout->volflg &= ~EF_ON;
      break;
    case 0x8:			/* enable volume envelope  */
      if (aout)
	aout->volflg |= EF_ON;
      break;
    case 0x9:			/* disable panning envelope */
      if (aout)
	aout->panflg &= ~EF_ON;
      break;
    case 0xa:			/* enable panning envelope */
      if (aout)
	aout->panflg |= EF_ON;
      break;
    case 0xb:			/* disable pitch envelope */
      if (aout)
	aout->pitflg &= ~EF_ON;
      break;
    case 0xc:			/* enable pitch envelope */
      if (aout)
	aout->pitflg |= EF_ON;
      break;
    }
}

static void 
pt_UpdateVoices (tmdy_struct_ex_t* tmdy_struct)
{
  SWORD envpan, envvol, envpit;
  UWORD playperiod;
  SLONG vibval, vibdpt;
  ULONG tmpvol;
  BOOL  kick_voice;

  MP_VOICE *aout;
  INSTRUMENT *i;
  SAMPLE *s;

  TMDY_MOD->mp.totalchn = TMDY_MOD->mp.realchn = 0;
  for (TMDY_MOD->mp.channel = 0; TMDY_MOD->mp.channel < MOD_NUM_VOICES; TMDY_MOD->mp.channel++)
    {
      aout = &TMDY_MOD->mp.voice[TMDY_MOD->mp.channel];
      i = aout->i;
      s = aout->s;

      if ((!s) || (!s->length))
	continue;

      if (aout->period < 14 || aout->period > 50000)
        {
          TMDY_MOD2MIDI->Voice_Stop(tmdy_struct, TMDY_MOD->mp.channel);
          continue;
        }

      kick_voice = 0;
      if ((aout->kick == KICK_NOTE) || (aout->kick == KICK_KEYOFF))
	{
	  kick_voice = 1;
	  aout->fadevol = 32768;
	  aout->aswppos = 0;
	}

      /* check for a dead note (fadevol=0) */
      if (!aout->fadevol || kick_voice)
	TMDY_MOD2MIDI->Voice_Stop(tmdy_struct, TMDY_MOD->mp.channel);

      if (i && ((aout->kick == KICK_NOTE) || (aout->kick == KICK_ENV)))
	{
	  StartEnvelope (tmdy_struct, &aout->venv, aout->volflg, i->volpts, i->volsusbeg,
	       i->volsusend, i->volbeg, i->volend, i->volenv, aout->keyoff);
	  StartEnvelope (tmdy_struct, &aout->penv, aout->panflg, i->panpts, i->pansusbeg,
	       i->pansusend, i->panbeg, i->panend, i->panenv, aout->keyoff);
	  StartEnvelope (tmdy_struct, &aout->cenv, aout->pitflg, i->pitpts, i->pitsusbeg,
	       i->pitsusend, i->pitbeg, i->pitend, i->pitenv, aout->keyoff);
	  if (aout->cenv.flg & EF_ON)
	    aout->masterperiod = GetPeriod (tmdy_struct, (UWORD) aout->note << 1, aout->master->speed);
	}
      aout->kick = KICK_ABSENT;

      envvol = (!(aout->volflg & EF_ON)) ? 256 :
	ProcessEnvelope (tmdy_struct, &aout->venv, 256, aout->keyoff);
      envpan = (!(aout->panflg & EF_ON)) ? PAN_CENTER :
	ProcessEnvelope (tmdy_struct, &aout->penv, PAN_CENTER, aout->keyoff);
      envpit = (!(aout->pitflg & EF_ON)) ? 32 :
	ProcessEnvelope (tmdy_struct, &aout->cenv, 32, aout->keyoff);

      tmpvol = aout->fadevol;	/* max 32768 */
      tmpvol *= aout->chanvol;	/* * max 64 */
      tmpvol *= aout->volume;	/* * max 256 */
      tmpvol /= 16384L;		/* tmpvol is max 32768 */
      aout->totalvol = tmpvol >> 2;	/* totalvolume used to determine samplevolume */
      tmpvol *= envvol;		/* * max 256 */
      tmpvol *= TMDY_MOD->mp.volume;	/* * max 128 */
      tmpvol /= 4194304UL;

      TMDY_MOD2MIDI->Voice_SetVolume(tmdy_struct, TMDY_MOD->mp.channel, tmpvol);
      if ((tmpvol) && (aout->master) && (aout->master->slave == aout))
	TMDY_MOD->mp.realchn++;
      TMDY_MOD->mp.totalchn++;

      if (aout->panning == PAN_SURROUND)
	TMDY_MOD2MIDI->Voice_SetPanning(tmdy_struct, TMDY_MOD->mp.channel, PAN_SURROUND);
      else if (aout->penv.flg & EF_ON)
	TMDY_MOD2MIDI->Voice_SetPanning(tmdy_struct, TMDY_MOD->mp.channel, DoPan (tmdy_struct, envpan, aout->panning));
      else
	TMDY_MOD2MIDI->Voice_SetPanning(tmdy_struct, TMDY_MOD->mp.channel, aout->panning);

      if (aout->period && s->vibdepth)
	switch (s->vibtype)
	  {
	  case 0:
	    vibval = avibtab[aout->avibpos & 127];
	    if (aout->avibpos & 0x80)
	      vibval = -vibval;
	    break;
	  case 1:
	    vibval = 64;
	    if (aout->avibpos & 0x80)
	      vibval = -vibval;
	    break;
	  case 2:
	    vibval = 63 - (((aout->avibpos + 128) & 255) >> 1);
	    break;
	  default:
	    vibval = (((aout->avibpos + 128) & 255) >> 1) - 64;
	    break;
	  }
      else
	vibval = 0;

      if (s->vibflags & AV_IT)
	{
	  if ((aout->aswppos >> 8) < s->vibdepth)
	    {
	      aout->aswppos += s->vibsweep;
	      vibdpt = aout->aswppos;
	    }
	  else
	    vibdpt = s->vibdepth << 8;
	  vibval = (vibval * vibdpt) >> 16;
	  if (aout->mflag)
	    {
	      if (!(TMDY_MOD->pf->flags & UF_LINEAR))
		vibval >>= 1;
	      aout->period -= vibval;
	    }
	}
      else
	{
	  /* do XM style auto-vibrato */
	  if (!(aout->keyoff & KEY_OFF))
	    {
	      if (aout->aswppos < s->vibsweep)
		{
		  vibdpt = (aout->aswppos * s->vibdepth) / s->vibsweep;
		  aout->aswppos++;
		}
	      else
		vibdpt = s->vibdepth;
	    }
	  else
	    {
	      /* keyoff -> depth becomes 0 if final depth wasn't reached or
	         stays at final level if depth WAS reached */
	      if (aout->aswppos >= s->vibsweep)
		vibdpt = s->vibdepth;
	      else
		vibdpt = 0;
	    }
	  vibval = (vibval * vibdpt) >> 8;
	  aout->period -= vibval;
	}

      /* update vibrato position */
      aout->avibpos = (aout->avibpos + s->vibrate) & 0xff;

      /* process pitch envelope */
      playperiod = aout->period;

      if ((aout->pitflg & EF_ON) && (envpit != 32))
	{
	  long p1;

	  envpit -= 32;
	  if ((aout->note << 1) + envpit <= 0)
	    envpit = -(aout->note << 1);

	  p1 = GetPeriod (tmdy_struct, ((UWORD) aout->note << 1) + envpit, aout->master->speed) - aout->masterperiod;
	  if (p1 > 0)
	    {
	      if ((UWORD) (playperiod + p1) <= playperiod)
		{
		  p1 = 0;
		  aout->keyoff |= KEY_OFF;
		}
	    }
	  else if (p1 < 0)
	    {
	      if ((UWORD) (playperiod + p1) >= playperiod)
		{
		  p1 = 0;
		  aout->keyoff |= KEY_OFF;
		}
	    }
	  playperiod += p1;
	}

      if (!aout->fadevol)
	{			/* check for a dead note (fadevol=0) */
	  TMDY_MOD->mp.totalchn--;
	  if ((tmpvol) && (aout->master) && (aout->master->slave == aout))
	    TMDY_MOD->mp.realchn--;
	}
      else
	{
	  TMDY_MOD2MIDI->Voice_SetPeriod(tmdy_struct, TMDY_MOD->mp.channel,
		      TMDY_UNIMOD->getAmigaPeriod (tmdy_struct, TMDY_MOD->pf->flags, playperiod));

	  if (kick_voice)
	    TMDY_MOD2MIDI->Voice_Play(tmdy_struct, TMDY_MOD->mp.channel, s, (aout->start == -1) ? ((s->flags & SF_UST_LOOP) ? s->loopstart : 0) : aout->start);

	  /* if keyfade, start substracting fadeoutspeed from fadevol: */
	  if ((i) && (aout->keyoff & KEY_FADE))
	    {
	      if (aout->fadevol >= i->volfade)
		aout->fadevol -= i->volfade;
	      else
		aout->fadevol = 0;
	    }
	}

      if (TMDY_MOD->mp.bpm != TMDY_MOD->mp.newbpm || TMDY_MOD->mp.sngspd != TMDY_MOD->mp.oldsngspd) {
	TMDY_MOD->mp.bpm = TMDY_MOD->mp.newbpm;
	TMDY_MOD->mp.oldsngspd = TMDY_MOD->mp.sngspd;
        TMDY_MOD2MIDI->Voice_NewTempo(tmdy_struct, TMDY_MOD->mp.bpm, TMDY_MOD->mp.sngspd);
      }
    }
}

/* Handles new notes or instruments */
static void 
pt_Notes (tmdy_struct_ex_t* tmdy_struct)
{
  UBYTE c, inst;
  int tr, funky;		/* funky is set to indicate note or instrument change */

  for (TMDY_MOD->mp.channel = 0; TMDY_MOD->mp.channel < TMDY_MOD->pf->numchn; TMDY_MOD->mp.channel++)
    {
      TMDY_MOD->a = &TMDY_MOD->mp.control[TMDY_MOD->mp.channel];

      if (TMDY_MOD->mp.sngpos >= TMDY_MOD->pf->numpos)
	{
	  tr = TMDY_MOD->pf->numtrk;
	  TMDY_MOD->mp.numrow = 0;
	}
      else
	{
	  tr = TMDY_MOD->pf->patterns[(TMDY_MOD->pf->positions[TMDY_MOD->mp.sngpos] * TMDY_MOD->pf->numchn) + TMDY_MOD->mp.channel];
	  TMDY_MOD->mp.numrow = TMDY_MOD->pf->pattrows[TMDY_MOD->pf->positions[TMDY_MOD->mp.sngpos]];
	}

      TMDY_MOD->a->row = (tr < TMDY_MOD->pf->numtrk) ? TMDY_UNIMOD->UniFindRow (tmdy_struct, TMDY_MOD->pf->tracks[tr], TMDY_MOD->mp.patpos) : NULL;
      TMDY_MOD->a->newsamp = 0;
      if (!TMDY_MOD->mp.vbtick)
	TMDY_MOD->a->notedelay = 0;

      if (!TMDY_MOD->a->row)
	continue;
      TMDY_UNIMOD->UniSetRow (tmdy_struct, TMDY_MOD->a->row);
      funky = 0;

      while ((c = TMDY_UNIMOD->UniGetByte (tmdy_struct)))
	switch (c)
	  {
	  case UNI_NOTE:
	    funky |= 1;
	    TMDY_MOD->a->oldnote = TMDY_MOD->a->anote, TMDY_MOD->a->anote = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	    TMDY_MOD->a->kick = KICK_NOTE;
	    TMDY_MOD->a->start = -1;
	    TMDY_MOD->a->sliding = 0;

	    /* retrig tremolo and vibrato waves ? */
	    if (!(TMDY_MOD->a->wavecontrol & 0x80))
	      TMDY_MOD->a->trmpos = 0;
	    if (!(TMDY_MOD->a->wavecontrol & 0x08))
	      TMDY_MOD->a->vibpos = 0;
	    if (!TMDY_MOD->a->panbwave)
	      TMDY_MOD->a->panbpos = 0;
	    break;
	  case UNI_INSTRUMENT:
	    inst = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	    if (inst >= TMDY_MOD->pf->numins)
	      break;		/* safety valve */
	    funky |= 2;
	    TMDY_MOD->a->i = (TMDY_MOD->pf->flags & UF_INST) ? &TMDY_MOD->pf->instruments[inst] : NULL;
	    TMDY_MOD->a->retrig = 0;
	    TMDY_MOD->a->s3mtremor = 0;
	    TMDY_MOD->a->ultoffset = 0;
	    TMDY_MOD->a->sample = inst;
	    break;
	  default:
	    TMDY_UNIMOD->UniSkipOpcode (tmdy_struct, c);
	    break;
	  }

      if (funky)
	{
	  INSTRUMENT *i;
	  SAMPLE *s;

	  i = TMDY_MOD->a->i;
	  if (i)
	    {
	      if (i->samplenumber[TMDY_MOD->a->anote] >= TMDY_MOD->pf->numsmp)
		continue;
	      s = &TMDY_MOD->pf->samples[i->samplenumber[TMDY_MOD->a->anote]];
	      TMDY_MOD->a->note = i->samplenote[TMDY_MOD->a->anote];
	    }
	  else
	    {
	      TMDY_MOD->a->note = TMDY_MOD->a->anote;
	      s = &TMDY_MOD->pf->samples[TMDY_MOD->a->sample];
	    }

	  if (TMDY_MOD->a->s != s)
	    {
	      TMDY_MOD->a->s = s;
	      TMDY_MOD->a->newsamp = TMDY_MOD->a->period;
	    }

	  /* channel or instrument determined panning ? */
	  TMDY_MOD->a->panning = TMDY_MOD->pf->panning[TMDY_MOD->mp.channel];
	  if (s->flags & SF_OWNPAN)
	    TMDY_MOD->a->panning = s->panning;
	  else if ((i) && (i->flags & IF_OWNPAN))
	    TMDY_MOD->a->panning = i->panning;

	  TMDY_MOD->a->data = s->data;
	  TMDY_MOD->a->speed = s->speed;

	  if (i)
	    {
	      if ((i->flags & IF_PITCHPAN)
		  && (TMDY_MOD->a->panning != PAN_SURROUND))
		{
		  TMDY_MOD->a->panning += ((TMDY_MOD->a->anote - i->pitpancenter) * i->pitpansep) / 8;
		  if (TMDY_MOD->a->panning < PAN_LEFT)
		    TMDY_MOD->a->panning = PAN_LEFT;
		  else if (TMDY_MOD->a->panning > PAN_RIGHT)
		    TMDY_MOD->a->panning = PAN_RIGHT;
		}
	      TMDY_MOD->a->pitflg = i->pitflg;
	      TMDY_MOD->a->volflg = i->volflg;
	      TMDY_MOD->a->panflg = i->panflg;
	      TMDY_MOD->a->nna = i->nnatype;
	      TMDY_MOD->a->dca = i->dca;
	      TMDY_MOD->a->dct = i->dct;
	    }
	  else
	    {
	      TMDY_MOD->a->pitflg = 0;
	      TMDY_MOD->a->volflg = 0;
	      TMDY_MOD->a->panflg = 0;
	      TMDY_MOD->a->nna = 0;
	      TMDY_MOD->a->dca = 0;
	      TMDY_MOD->a->dct = DCT_OFF;
	    }

	  if (funky & 2)	/* instrument change */
	    {
	      /* IT random volume variations: 0:8 bit fixed, and one bit for
	         sign. */
	      TMDY_MOD->a->volume = TMDY_MOD->a->tmpvolume = s->volume;
	      if ((s) && (i))
		{
		  if (i->rvolvar)
		    {
		      TMDY_MOD->a->volume = TMDY_MOD->a->tmpvolume = s->volume +
			((s->volume * ((SLONG) i->rvolvar * (SLONG) getrandom (tmdy_struct,512)
			  )) / 25600);
		      if (TMDY_MOD->a->volume < 0)
			TMDY_MOD->a->volume = TMDY_MOD->a->tmpvolume = 0;
		      else if (TMDY_MOD->a->volume > 64)
			TMDY_MOD->a->volume = TMDY_MOD->a->tmpvolume = 64;
		    }
		  if ((TMDY_MOD->a->panning != PAN_SURROUND))
		    {
		      TMDY_MOD->a->panning += ((TMDY_MOD->a->panning * ((SLONG) i->rpanvar *
					 (SLONG) getrandom (tmdy_struct,512))) / 25600);
		      if (TMDY_MOD->a->panning < PAN_LEFT)
			TMDY_MOD->a->panning = PAN_LEFT;
		      else if (TMDY_MOD->a->panning > PAN_RIGHT)
			TMDY_MOD->a->panning = PAN_RIGHT;
		    }
		}
	    }

	  TMDY_MOD->a->wantedperiod = TMDY_MOD->a->tmpperiod = GetPeriod (tmdy_struct, (UWORD) TMDY_MOD->a->note << 1, TMDY_MOD->a->speed);
	  TMDY_MOD->a->keyoff = KEY_KICK;
	}
    }
}

/* Handles effects */
static void 
pt_EffectsPass1 (tmdy_struct_ex_t* tmdy_struct)
{
  MP_VOICE *aout;

  for (TMDY_MOD->mp.channel = 0; TMDY_MOD->mp.channel < TMDY_MOD->pf->numchn; TMDY_MOD->mp.channel++)
    {
      TMDY_MOD->a = &TMDY_MOD->mp.control[TMDY_MOD->mp.channel];

      if ((aout = TMDY_MOD->a->slave))
	{
	  TMDY_MOD->a->fadevol = aout->fadevol;
	  TMDY_MOD->a->period = aout->period;
	  if (TMDY_MOD->a->kick == KICK_KEYOFF)
	    TMDY_MOD->a->keyoff = aout->keyoff;
	}

      if (!TMDY_MOD->a->row)
	continue;
      TMDY_UNIMOD->UniSetRow (tmdy_struct, TMDY_MOD->a->row);

      TMDY_MOD->a->ownper = TMDY_MOD->a->ownvol = 0;
      TMDY_MOD->mp.explicitslides = 0;
      pt_playeffects (tmdy_struct);

      /* continue volume slide if necessary for XM and IT */
      if (TMDY_MOD->pf->flags & UF_BGSLIDES)
	{
	  if (!TMDY_MOD->mp.explicitslides && TMDY_MOD->a->sliding)
	    DoS3MVolSlide(tmdy_struct, 0);
	  else if (TMDY_MOD->a->tmpvolume)
	    TMDY_MOD->a->sliding = TMDY_MOD->mp.explicitslides;
	}

      if (!TMDY_MOD->a->ownper)
	TMDY_MOD->a->period = TMDY_MOD->a->tmpperiod;
      if (!TMDY_MOD->a->ownvol)
	TMDY_MOD->a->volume = TMDY_MOD->a->tmpvolume;

      if (TMDY_MOD->a->s)
	{
	  if (TMDY_MOD->a->i)
	    TMDY_MOD->a->outvolume = (TMDY_MOD->a->volume * TMDY_MOD->a->s->globvol * TMDY_MOD->a->i->globvol) >> 10;
	  else
	    TMDY_MOD->a->outvolume = (TMDY_MOD->a->volume * TMDY_MOD->a->s->globvol) >> 4;
	  if (TMDY_MOD->a->outvolume > 256)
	    TMDY_MOD->a->volume = 256;
	  else if (TMDY_MOD->a->outvolume < 0)
	    TMDY_MOD->a->outvolume = 0;
	}
    }
}

/* NNA management */
static void 
pt_NNA (tmdy_struct_ex_t* tmdy_struct)
{
  for (TMDY_MOD->mp.channel = 0; TMDY_MOD->mp.channel < TMDY_MOD->pf->numchn; TMDY_MOD->mp.channel++)
    {
      TMDY_MOD->a = &TMDY_MOD->mp.control[TMDY_MOD->mp.channel];

      if (TMDY_MOD->a->kick == KICK_NOTE)
	{
	  BOOL k = 0;

	  if (TMDY_MOD->a->slave)
	    {
	      MP_VOICE *aout;

	      aout = TMDY_MOD->a->slave;
	      if (aout->nna & NNA_MASK)
		{
		  /* Make sure the old MP_VOICE channel knows it has no
		     master now ! */
		  TMDY_MOD->a->slave = NULL;
		  /* assume the channel is taken by NNA */
		  aout->mflag = 0;

		  switch (aout->nna)
		    {
		    case NNA_CONTINUE:		/* continue note, do nothing */
		      break;
		    case NNA_OFF:	/* note off */
		      aout->keyoff |= KEY_OFF;
		      if ((!(aout->volflg & EF_ON)) || (aout->volflg & EF_LOOP))
			aout->keyoff = KEY_KILL;
		      break;
		    case NNA_FADE:
		      aout->keyoff |= KEY_FADE;
		      break;
		    }
		}
	    }

	  if (TMDY_MOD->a->dct != DCT_OFF)
	    {
	      int t;

	      for (t = 0; t < MOD_NUM_VOICES; t++)
		if ((!TMDY_MOD2MIDI->Voice_Stopped(tmdy_struct, t)) &&
		    (TMDY_MOD->mp.voice[t].masterchn == TMDY_MOD->mp.channel) &&
		    (TMDY_MOD->a->sample == TMDY_MOD->mp.voice[t].sample))
		  {
		    k = 0;
		    switch (TMDY_MOD->a->dct)
		      {
		      case DCT_NOTE:
			if (TMDY_MOD->a->note == TMDY_MOD->mp.voice[t].note)
			  k = 1;
			break;
		      case DCT_SAMPLE:
			if (TMDY_MOD->a->data == TMDY_MOD->mp.voice[t].data)
			  k = 1;
			break;
		      case DCT_INST:
			k = 1;
			break;
		      }
		    if (k)
		      switch (TMDY_MOD->a->dca)
			{
			case DCA_CUT:
			  TMDY_MOD->mp.voice[t].fadevol = 0;
			  break;
			case DCA_OFF:
			  TMDY_MOD->mp.voice[t].keyoff |= KEY_OFF;
			  if ((!(TMDY_MOD->mp.voice[t].volflg & EF_ON)) ||
			      (TMDY_MOD->mp.voice[t].volflg & EF_LOOP))
			    TMDY_MOD->mp.voice[t].keyoff = KEY_KILL;
			  break;
			case DCA_FADE:
			  TMDY_MOD->mp.voice[t].keyoff |= KEY_FADE;
			  break;
			}
		  }
	    }
	}			/* if (TMDY_MOD->a->kick==KICK_NOTE) */
    }
}

/* Setup module and NNA voices */
static void 
pt_SetupVoices (tmdy_struct_ex_t* tmdy_struct)
{
  MP_VOICE *aout;

  for (TMDY_MOD->mp.channel = 0; TMDY_MOD->mp.channel < TMDY_MOD->pf->numchn; TMDY_MOD->mp.channel++)
    {
      TMDY_MOD->a = &TMDY_MOD->mp.control[TMDY_MOD->mp.channel];

      if (TMDY_MOD->a->notedelay)
	continue;
      if (TMDY_MOD->a->kick == KICK_NOTE)
	{
	  /* if no channel was cut above, find an empty or quiet channel
	     here */
	  if (TMDY_MOD->pf->flags & UF_NNA)
	    {
	      if (!TMDY_MOD->a->slave)
		{
		  int newchn;

		  if ((newchn = MP_FindEmptyChannel (tmdy_struct)) != -1)
		    TMDY_MOD->a->slave = &TMDY_MOD->mp.voice[TMDY_MOD->a->slavechn = newchn];
		}
	    }
	  else
	    TMDY_MOD->a->slave = &TMDY_MOD->mp.voice[TMDY_MOD->a->slavechn = TMDY_MOD->mp.channel];

	  /* assign parts of MP_VOICE only done for a KICK_NOTE */
	  if ((aout = TMDY_MOD->a->slave))
	    {
	      if (aout->mflag && aout->master)
		aout->master->slave = NULL;
	      aout->master = TMDY_MOD->a;
	      TMDY_MOD->a->slave = aout;
	      aout->masterchn = TMDY_MOD->mp.channel;
	      aout->mflag = 1;
	    }
	}
      else
	aout = TMDY_MOD->a->slave;

      if (aout)
	{
	  aout->i = TMDY_MOD->a->i;
	  aout->s = TMDY_MOD->a->s;
	  aout->sample = TMDY_MOD->a->sample;
	  aout->data = TMDY_MOD->a->data;
	  aout->period = TMDY_MOD->a->period;
	  aout->panning = TMDY_MOD->a->panning;
	  aout->chanvol = TMDY_MOD->a->chanvol;
	  aout->fadevol = TMDY_MOD->a->fadevol;
	  aout->kick = TMDY_MOD->a->kick;
	  aout->start = TMDY_MOD->a->start;
	  aout->volflg = TMDY_MOD->a->volflg;
	  aout->panflg = TMDY_MOD->a->panflg;
	  aout->pitflg = TMDY_MOD->a->pitflg;
	  aout->volume = TMDY_MOD->a->outvolume;
	  aout->keyoff = TMDY_MOD->a->keyoff;
	  aout->note = TMDY_MOD->a->note;
	  aout->nna = TMDY_MOD->a->nna;
	}
      TMDY_MOD->a->kick = KICK_ABSENT;
    }
}

/* second effect pass */
static void 
pt_EffectsPass2 (tmdy_struct_ex_t* tmdy_struct)
{
  UBYTE c;

  for (TMDY_MOD->mp.channel = 0; TMDY_MOD->mp.channel < TMDY_MOD->pf->numchn; TMDY_MOD->mp.channel++)
    {
      TMDY_MOD->a = &TMDY_MOD->mp.control[TMDY_MOD->mp.channel];

      if (!TMDY_MOD->a->row)
	continue;
      TMDY_UNIMOD->UniSetRow (tmdy_struct, TMDY_MOD->a->row);

      while ((c = TMDY_UNIMOD->UniGetByte (tmdy_struct)))
	if (c == UNI_ITEFFECTS0)
	  {
	    c = TMDY_UNIMOD->UniGetByte (tmdy_struct);
	    if ((c >> 4) == SS_S7EFFECTS)
	      DoNNAEffects (tmdy_struct, c & 0xf);
	  }
	else
	  TMDY_UNIMOD->UniSkipOpcode (tmdy_struct, c);
    }
}

static BOOL 
HandleTick (tmdy_struct_ex_t* tmdy_struct)
{
  if ((!TMDY_MOD->pf) || (TMDY_MOD->mp.sngpos >= TMDY_MOD->pf->numpos)){
    return 0;
  }
  
  if (++TMDY_MOD->mp.vbtick >= TMDY_MOD->mp.sngspd)
    {
      if (TMDY_MOD->mp.pat_repcrazy)
	TMDY_MOD->mp.pat_repcrazy = 0;	/* play 2 times row 0 */
      else
	TMDY_MOD->mp.patpos++;
      TMDY_MOD->mp.vbtick = 0;

      /* process pattern-delay. TMDY_MOD->mp.patdly2 is the counter and TMDY_MOD->mp.patdly is
         the command memory. */
      if (TMDY_MOD->mp.patdly)
	TMDY_MOD->mp.patdly2 = TMDY_MOD->mp.patdly, TMDY_MOD->mp.patdly = 0;
      if (TMDY_MOD->mp.patdly2)
	{
	  /* patterndelay active */
	  if (--TMDY_MOD->mp.patdly2)
	    /* so turn back TMDY_MOD->mp.patpos by 1 */
	    if (TMDY_MOD->mp.patpos)
	      TMDY_MOD->mp.patpos--;
	}

      /* do we have to get a new patternpointer ? (when TMDY_MOD->mp.patpos reaches the
         pattern size, or when a patternbreak is active) */
      if (((TMDY_MOD->mp.patpos >= TMDY_MOD->mp.numrow) && (TMDY_MOD->mp.numrow > 0)) && (!TMDY_MOD->mp.posjmp))
	TMDY_MOD->mp.posjmp = 3;

      if (TMDY_MOD->mp.posjmp)
	{
	  TMDY_MOD->mp.patpos = TMDY_MOD->mp.numrow ? (TMDY_MOD->mp.patbrk % TMDY_MOD->mp.numrow) : 0;
	  TMDY_MOD->mp.pat_repcrazy = 0;
	  TMDY_MOD->mp.sngpos += (TMDY_MOD->mp.posjmp - 2);

	  for (TMDY_MOD->mp.channel = 0; TMDY_MOD->mp.channel < TMDY_MOD->pf->numchn; TMDY_MOD->mp.channel++)
	    TMDY_MOD->mp.control[TMDY_MOD->mp.channel].pat_reppos = -1;

	  TMDY_MOD->mp.patbrk = TMDY_MOD->mp.posjmp = 0;
	  /* handle the "---" (end of song) pattern since it can occur
	     *inside* the module in .IT and .S3M */
	  if ((TMDY_MOD->mp.sngpos >= TMDY_MOD->pf->numpos) || (TMDY_MOD->pf->positions[TMDY_MOD->mp.sngpos] == 255)){
	    return 0;
	  }

	  if (TMDY_MOD->mp.sngpos < 0)
	    TMDY_MOD->mp.sngpos = TMDY_MOD->pf->numpos - 1;

	}

      if (!TMDY_MOD->mp.patdly2)
	pt_Notes (tmdy_struct);
    }

  pt_EffectsPass1 (tmdy_struct);
  if (TMDY_MOD->pf->flags & UF_NNA)
    pt_NNA (tmdy_struct);
  pt_SetupVoices (tmdy_struct);
  pt_EffectsPass2 (tmdy_struct);

  /* now set up the actual hardware channel playback information */
  pt_UpdateVoices (tmdy_struct);
  return 1;
}

BOOL
mod_do_play (tmdy_struct_ex_t* tmdy_struct, MODULE * mf)
{
  int t;

  /* make sure the player doesn't start with garbage */
  memset(&TMDY_MOD->mp, 0, sizeof(TMDY_MOD->mp));
  TMDY_MOD->mp.control = (MP_CONTROL *) TMDY_COMMON->safe_malloc(tmdy_struct, mf->numchn * sizeof (MP_CONTROL));
  if (!TMDY_MOD->mp.control){
    return 1;
  }

  memset (TMDY_MOD->mp.control, 0, mf->numchn * sizeof (MP_CONTROL));
  for (t = 0; t < mf->numchn; t++)
    {
      TMDY_MOD->mp.control[t].chanvol = mf->chanvol[t];
      TMDY_MOD->mp.control[t].panning = mf->panning[t];
    }

  TMDY_MOD->mp.pat_repcrazy = 0;
  TMDY_MOD->mp.sngpos = 0;
  TMDY_MOD->mp.sngspd = mf->initspeed ? (mf->initspeed <= 32 ? mf->initspeed : 32) : 6;
  TMDY_MOD->mp.volume = mf->initvolume > 128 ? 128 : mf->initvolume;

  TMDY_MOD->mp.oldsngspd = TMDY_MOD->mp.sngspd;
  TMDY_MOD->mp.vbtick = TMDY_MOD->mp.sngspd;
  TMDY_MOD->mp.patdly = 0;
  TMDY_MOD->mp.patdly2 = 0;
  TMDY_MOD->mp.bpm = mf->inittempo <= 32 ? 32 : mf->inittempo;
  TMDY_MOD->mp.newbpm = TMDY_MOD->mp.bpm;

  TMDY_MOD->mp.patpos = 0;
  TMDY_MOD->mp.posjmp = 2;		/* make sure the player fetches the first note */
  TMDY_MOD->mp.numrow = -1;
  TMDY_MOD->mp.patbrk = 0;
  TMDY_MOD->pf = mf;

  TMDY_MOD2MIDI->Voice_StartPlaying(tmdy_struct);
  TMDY_MOD2MIDI->Voice_NewTempo(tmdy_struct, TMDY_MOD->mp.bpm, TMDY_MOD->mp.sngspd);
  do
    TMDY_MOD2MIDI->Voice_TickDone(tmdy_struct);
  while (HandleTick (tmdy_struct));
  TMDY_MOD2MIDI->Voice_TickDone(tmdy_struct);
  TMDY_MOD2MIDI->Voice_EndPlaying(tmdy_struct);

  /* reset all sample pans to center */
  /* mod routines have already adjusted the pan events, so we don't want
     the regular mixing routines to apply them yet again */
  for (t = 0; t < 256; t++)
  {
    if (TMDY_INSTRUM->special_patch[t])
      TMDY_INSTRUM->special_patch[t]->sample->panning = 64;
  }

  /* Done! */
  free (TMDY_MOD->mp.control);
  return 0;
}


int ts_get_module_type (tmdy_struct_ex_t* tmdy_struct, char *fn){
	int ts_buf;
	timidity_mutex_lock(TMDY_MOD->busy);
	ts_buf=get_module_type (tmdy_struct, fn);
	timidity_mutex_unlock(TMDY_MOD->busy);
	return ts_buf;
}
int ts_load_module_file (tmdy_struct_ex_t* tmdy_struct, struct timidity_file *tf, int mod_type){
	int ts_buf;
	timidity_mutex_lock(TMDY_MOD->busy);
	ts_buf=load_module_file (tmdy_struct, tf, mod_type);
	timidity_mutex_unlock(TMDY_MOD->busy);
	return ts_buf;
}
char *ts_get_module_title (tmdy_struct_ex_t* tmdy_struct, struct timidity_file *tf, int mod_type){
	char *ts_buf;
	timidity_mutex_lock(TMDY_MOD->busy);
	ts_buf=get_module_title (tmdy_struct, tf, mod_type);
	timidity_mutex_unlock(TMDY_MOD->busy);
	return ts_buf;
}

mod_ex_t* new_mod(tmdy_struct_ex_t *tmdy_struct){
	int i;
	mod_ex_t* mod_ex;

	mod_ex=(mod_ex_t *)TMDY_COMMON->safe_malloc(tmdy_struct, sizeof(mod_ex_t));
	
	timidity_mutex_init(mod_ex->busy);
	
	mod_ex->get_module_type=ts_get_module_type;
	mod_ex->load_module_file=ts_load_module_file;
	mod_ex->get_module_title=ts_get_module_title;
	
	mod_ex->pf = NULL;
	
	return mod_ex;
}
void destroy_mod(mod_ex_t* mod){	
	timidity_mutex_destroy(mod->busy);
	free(mod);
}
	
