/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2002 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    miditrace.c - Midi audio synchronized tracer
    Written by Masanao Izumo <mo@goice.co.jp>
*/


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#include <stdio.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */
#include <stdlib.h>

#ifndef NO_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif


#include "timidity.h"
#include "common.h"
#include "instrum.h"
#include "playmidi.h"
#include "output.h"
#include "controls.h"
#include "miditrace.h"
#include "wrd.h"
#include "aq.h"

#include "miditrace_prv.h"

enum trace_argtypes
{
    ARG_VOID,
    ARG_INT,
    ARG_INT_INT,
    ARG_VP,
    ARG_CE,
};

typedef struct _MidiTraceList
{
    int32 start;	/* Samples to start */
    int argtype;	/* Argument type */

    union { /* Argument */
	int args[2];
	uint16 ui16;
	CtlEvent ce;
	void *v;
    } a;

    union { /* Handler function */
	void (*f0)(tmdy_struct_ex_t *tmdy_struct);
	void (*f1)(tmdy_struct_ex_t *tmdy_struct, int);
	void (*f2)(tmdy_struct_ex_t *tmdy_struct, int, int);
	void (*fce)(tmdy_struct_ex_t *tmdy_struct, CtlEvent *ce);
	void (*fv)(tmdy_struct_ex_t *tmdy_struct, void *);
    } f;

    struct _MidiTraceList *next; /* Chain link next node */
} MidiTraceList;


//xtern int32 current_sample;

static MidiTraceList *new_trace_node(tmdy_struct_ex_t *tmdy_struct)
{
    MidiTraceList *p;

    if(TMDY_MIDITRACE->midi_trace.free_list == NULL)
	p = (MidiTraceList *)TMDY_UTILS->mblock->new_segment(tmdy_struct, &TMDY_MIDITRACE->midi_trace.pool,
					 sizeof(MidiTraceList));
    else
    {
	p = TMDY_MIDITRACE->midi_trace.free_list;
	TMDY_MIDITRACE->midi_trace.free_list = TMDY_MIDITRACE->midi_trace.free_list->next;	
    }
    return p;
}

static void reuse_trace_node(tmdy_struct_ex_t *tmdy_struct, MidiTraceList *p)
{
    p->next = TMDY_MIDITRACE->midi_trace.free_list;
    TMDY_MIDITRACE->midi_trace.free_list = p;
}

static int32 trace_start_time(tmdy_struct_ex_t *tmdy_struct)
{
    if((TMDY_OUTPUT->play_mode)->flag & PF_CAN_TRACE)
	return TMDY_PLAYMIDI->current_sample;
    return -1;
}

static void run_midi_trace(tmdy_struct_ex_t *tmdy_struct, MidiTraceList *p)
{
    if(!(TMDY_CONTROLS->ctl)->opened)
	return;

    switch(p->argtype)
    {
      case ARG_VOID:
	p->f.f0(tmdy_struct);
	break;
      case ARG_INT:
	p->f.f1(tmdy_struct, p->a.args[0]);
	break;
      case ARG_INT_INT:
	p->f.f2(tmdy_struct, p->a.args[0], p->a.args[1]);
	break;
      case ARG_VP:
	p->f.fv(tmdy_struct, p->a.v);
	break;
      case ARG_CE:
	p->f.fce(tmdy_struct, &p->a.ce);
	break;
    }
}

static MidiTraceList *midi_trace_setfunc(tmdy_struct_ex_t *tmdy_struct, MidiTraceList *node)
{
    MidiTraceList *p;

    if(!(TMDY_CONTROLS->ctl)->trace_playing || node->start < 0)
    {
	run_midi_trace(tmdy_struct, node);
	return NULL;
    }

    p = new_trace_node(tmdy_struct);
    *p = *node;
    p->next = NULL;

    if(TMDY_MIDITRACE->midi_trace.head == NULL)
	TMDY_MIDITRACE->midi_trace.head = TMDY_MIDITRACE->midi_trace.tail = p;
    else
    {
	TMDY_MIDITRACE->midi_trace.tail->next = p;
	TMDY_MIDITRACE->midi_trace.tail = p;
    }

    return p;
}

void push_midi_trace0(tmdy_struct_ex_t *tmdy_struct, void (*f)(tmdy_struct_ex_t*))
{
    MidiTraceList node;
    if(f == NULL)
	return;
    memset(&node, 0, sizeof(node));
    node.start = trace_start_time(tmdy_struct);
    node.argtype = ARG_VOID;
    node.f.f0 = f;
    midi_trace_setfunc(tmdy_struct, &node);
}

void push_midi_trace1(tmdy_struct_ex_t *tmdy_struct, void (*f)(tmdy_struct_ex_t*, int), int arg1)
{
    MidiTraceList node;
    if(f == NULL)
	return;
    memset(&node, 0, sizeof(node));
    node.start = trace_start_time(tmdy_struct);
    node.argtype = ARG_INT;
    node.f.f1 = f;
    node.a.args[0] = arg1;
    midi_trace_setfunc(tmdy_struct, &node);
}

void push_midi_trace2(tmdy_struct_ex_t *tmdy_struct, void (*f)(tmdy_struct_ex_t*, int, int), int arg1, int arg2)
{
    MidiTraceList node;
    if(f == NULL)
	return;
    memset(&node, 0, sizeof(node));
    node.start = trace_start_time(tmdy_struct);
    node.argtype = ARG_INT_INT;
    node.f.f2 = f;
    node.a.args[0] = arg1;
    node.a.args[1] = arg2;
    midi_trace_setfunc(tmdy_struct, &node);
}

void push_midi_trace_ce(tmdy_struct_ex_t *tmdy_struct, void (*f)(tmdy_struct_ex_t *tmdy_struct, CtlEvent *), CtlEvent *ce)
{
    MidiTraceList node;
    if(f == NULL)
	return;
    memset(&node, 0, sizeof(node));
    node.start = trace_start_time(tmdy_struct);
    node.argtype = ARG_CE;
    node.f.fce = f;
    node.a.ce = *ce;
    midi_trace_setfunc(tmdy_struct, &node);
}

void push_midi_time_vp(tmdy_struct_ex_t *tmdy_struct, int32 start, void (*f)(tmdy_struct_ex_t *tmdy_struct, void *), void *vp)
{
    MidiTraceList node;
    if(f == NULL)
	return;
    memset(&node, 0, sizeof(node));
    node.start = start;
    node.argtype = ARG_VP;
    node.f.fv = f;
    node.a.v = vp;
    midi_trace_setfunc(tmdy_struct, &node);
}

int32 trace_loop(tmdy_struct_ex_t *tmdy_struct)
{
    int32 cur, start;
    int ctl_update;
//    static int trace_loop_lasttime = -1;

    if(TMDY_MIDITRACE->midi_trace.trace_loop_hook != NULL)
	TMDY_MIDITRACE->midi_trace.trace_loop_hook(tmdy_struct);

    if(TMDY_MIDITRACE->midi_trace.head == NULL){
		return 0;
	}

    if((cur = current_trace_samples(tmdy_struct)) == -1 || !(TMDY_CONTROLS->ctl)->trace_playing)
	cur = 0x7fffffff; /* apply all trace event */

    ctl_update = 0;
    start = TMDY_MIDITRACE->midi_trace.head->start;
    while(TMDY_MIDITRACE->midi_trace.head && cur >= TMDY_MIDITRACE->midi_trace.head->start
	  && cur > 0) /* privent flying start */
    {
	MidiTraceList *p;

	p = TMDY_MIDITRACE->midi_trace.head;
	run_midi_trace(tmdy_struct, p);
	if(p->argtype == ARG_CE)
	    ctl_update = 1;
	TMDY_MIDITRACE->midi_trace.head = TMDY_MIDITRACE->midi_trace.head->next;
	reuse_trace_node(tmdy_struct, p);
    }

    if(ctl_update)
	TMDY_PLAYMIDI->ctl_mode_event(tmdy_struct, CTLE_REFRESH, 0, 0, 0);

    if(TMDY_MIDITRACE->midi_trace.head == NULL)
    {
	TMDY_MIDITRACE->midi_trace.tail = NULL;
	return 0; /* No more tracing */
    }

    if(!ctl_update)
    {
	if(TMDY_MIDITRACE->trace_loop_lasttime == cur)
	    TMDY_MIDITRACE->midi_trace.head->start--;	/* avoid infinite loop */
	TMDY_MIDITRACE->trace_loop_lasttime = cur;
    }

    return 1; /* must call trace_loop() continued */
}

void trace_offset(tmdy_struct_ex_t *tmdy_struct, int offset)
{
    TMDY_MIDITRACE->midi_trace.offset = offset;
}

void trace_flush(tmdy_struct_ex_t *tmdy_struct)
{
    TMDY_MIDITRACE->midi_trace.flush_flag = 1;
    TMDY_WRD->wrd_midi_event(tmdy_struct, WRD_START_SKIP, WRD_NOARG);
    while(TMDY_MIDITRACE->midi_trace.head)
    {
	MidiTraceList *p;

	p = TMDY_MIDITRACE->midi_trace.head;
	run_midi_trace(tmdy_struct, p);
	TMDY_MIDITRACE->midi_trace.head = TMDY_MIDITRACE->midi_trace.head->next;
	reuse_trace_node(tmdy_struct, p);
    }
    TMDY_WRD->wrd_midi_event(tmdy_struct, WRD_END_SKIP, WRD_NOARG);
    TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, &TMDY_MIDITRACE->midi_trace.pool);
    TMDY_MIDITRACE->midi_trace.head = TMDY_MIDITRACE->midi_trace.tail = TMDY_MIDITRACE->midi_trace.free_list = NULL;
    TMDY_PLAYMIDI->ctl_mode_event(tmdy_struct, CTLE_REFRESH, 0, 0, 0);
    TMDY_MIDITRACE->midi_trace.flush_flag = 0;
}

void set_trace_loop_hook(tmdy_struct_ex_t *tmdy_struct, void (* f)(tmdy_struct_ex_t *tmdy_struct))
{
    TMDY_MIDITRACE->midi_trace.trace_loop_hook = f;
}

int32 current_trace_samples(tmdy_struct_ex_t *tmdy_struct)
{
    int32 sp;
    if((sp = (TMDY_AQ->aq_samples)(tmdy_struct)) == -1){
		return -1;
	}
    return TMDY_MIDITRACE->midi_trace.offset + (TMDY_AQ->aq_samples)(tmdy_struct);
}

void init_midi_trace(tmdy_struct_ex_t *tmdy_struct)
{
    memset(&TMDY_MIDITRACE->midi_trace, 0, sizeof(TMDY_MIDITRACE->midi_trace));
    TMDY_UTILS->mblock->init_mblock(tmdy_struct, &TMDY_MIDITRACE->midi_trace.pool);
}

int32 trace_wait_samples(tmdy_struct_ex_t *tmdy_struct)
{
    int32 s;

    if(TMDY_MIDITRACE->midi_trace.head == NULL){
		return -1;
	}
    if((s = current_trace_samples(tmdy_struct)) == -1){
		return 0;
	}
    s = TMDY_MIDITRACE->midi_trace.head->start - s;
    if(s < 0)
	s = 0;
    return s;
}


void ts_init_midi_trace(tmdy_struct_ex_t *tmdy_struct){
	timidity_mutex_lock(TMDY_MIDITRACE->busy);
	init_midi_trace(tmdy_struct);
	timidity_mutex_unlock(TMDY_MIDITRACE->busy);
}
void ts_push_midi_trace0(tmdy_struct_ex_t *tmdy_struct, void (*f)(tmdy_struct_ex_t*)){
	timidity_mutex_lock(TMDY_MIDITRACE->busy);
	push_midi_trace0(tmdy_struct, f);
	timidity_mutex_unlock(TMDY_MIDITRACE->busy);
}
void ts_push_midi_trace1(tmdy_struct_ex_t *tmdy_struct, void (*f)(tmdy_struct_ex_t*, int), int arg1){
	timidity_mutex_lock(TMDY_MIDITRACE->busy);
	push_midi_trace1(tmdy_struct, f, arg1);
	timidity_mutex_unlock(TMDY_MIDITRACE->busy);
}
void ts_push_midi_trace2(tmdy_struct_ex_t *tmdy_struct, void (*f)(tmdy_struct_ex_t*, int, int), int arg1, int arg2){
	timidity_mutex_lock(TMDY_MIDITRACE->busy);
	push_midi_trace2(tmdy_struct, f, arg1, arg2);
	timidity_mutex_unlock(TMDY_MIDITRACE->busy);
}
void ts_push_midi_trace_ce(tmdy_struct_ex_t *tmdy_struct, void (*f)(tmdy_struct_ex_t*, CtlEvent *), CtlEvent *ce){
	timidity_mutex_lock(TMDY_MIDITRACE->busy);
	push_midi_trace_ce(tmdy_struct, *f, ce);
	timidity_mutex_unlock(TMDY_MIDITRACE->busy);
}
void ts_push_midi_time_vp(tmdy_struct_ex_t *tmdy_struct, int32 start, void (*f)(tmdy_struct_ex_t *tmdy_struct, void *), void *vp){
	timidity_mutex_lock(TMDY_MIDITRACE->busy);
	push_midi_time_vp(tmdy_struct, start, f, vp);
	timidity_mutex_unlock(TMDY_MIDITRACE->busy);
}
int32 ts_trace_loop(tmdy_struct_ex_t *tmdy_struct){
	int32 ts_buf;
	timidity_mutex_lock(TMDY_MIDITRACE->busy);
	ts_buf=trace_loop(tmdy_struct);
	timidity_mutex_unlock(TMDY_MIDITRACE->busy);
	return ts_buf;
}
void ts_trace_flush(tmdy_struct_ex_t *tmdy_struct){
	timidity_mutex_lock(TMDY_MIDITRACE->busy);
	trace_flush(tmdy_struct);
	timidity_mutex_unlock(TMDY_MIDITRACE->busy);
}
void ts_trace_offset(tmdy_struct_ex_t *tmdy_struct, int offset){
	timidity_mutex_lock(TMDY_MIDITRACE->busy);
	trace_offset(tmdy_struct, offset);
	timidity_mutex_unlock(TMDY_MIDITRACE->busy);
}
/*
void ts_trace_nodelay(tmdy_struct_ex_t *tmdy_struct, int nodelay){
	timidity_mutex_lock(TMDY_MIDITRACE->busy);
	trace_nodelay(tmdy_struct, nodelay);
	timidity_mutex_unlock(TMDY_MIDITRACE->busy);
}
*/
void ts_set_trace_loop_hook(tmdy_struct_ex_t *tmdy_struct, void (* f)(tmdy_struct_ex_t *tmdy_struct)){
	timidity_mutex_lock(TMDY_MIDITRACE->busy);
	set_trace_loop_hook(tmdy_struct, f);
	timidity_mutex_unlock(TMDY_MIDITRACE->busy);
}
int32 ts_current_trace_samples(tmdy_struct_ex_t *tmdy_struct){
	int32 ts_buf;
	timidity_mutex_lock(TMDY_MIDITRACE->busy);
	ts_buf=current_trace_samples(tmdy_struct);
	timidity_mutex_unlock(TMDY_MIDITRACE->busy);
	return ts_buf;
}
int32 ts_trace_wait_samples(tmdy_struct_ex_t *tmdy_struct){
	int32 ts_buf;
	timidity_mutex_lock(TMDY_MIDITRACE->busy);
	ts_buf=trace_wait_samples(tmdy_struct);
	timidity_mutex_unlock(TMDY_MIDITRACE->busy);
	return ts_buf;
}


miditrace_ex_t* new_miditrace(tmdy_struct_ex_t *tmdy_struct){
	int i;
	miditrace_ex_t* miditrace_ex;

	miditrace_ex=(miditrace_ex_t *)TMDY_COMMON->safe_malloc(tmdy_struct, sizeof(miditrace_ex_t));
	timidity_mutex_init(miditrace_ex->busy);
	
miditrace_ex->init_midi_trace=ts_init_midi_trace;

miditrace_ex->push_midi_trace0=ts_push_midi_trace0;
miditrace_ex->push_midi_trace1=ts_push_midi_trace1;
miditrace_ex->push_midi_trace2=ts_push_midi_trace2;
miditrace_ex->push_midi_trace_ce=ts_push_midi_trace_ce;
miditrace_ex->push_midi_time_vp=ts_push_midi_time_vp;

miditrace_ex->trace_loop=ts_trace_loop;
miditrace_ex->trace_flush=ts_trace_flush;
miditrace_ex->trace_offset=ts_trace_offset;
//miditrace_ex->trace_nodelay=ts_trace_nodelay;
miditrace_ex->set_trace_loop_hook=ts_set_trace_loop_hook;
miditrace_ex->current_trace_samples=ts_current_trace_samples;
miditrace_ex->trace_wait_samples=ts_trace_wait_samples;

/**** private ****/	
miditrace_ex->trace_loop_lasttime = -1;
	
	return miditrace_ex;
}
void destroy_miditrace(miditrace_ex_t* miditrace){	
	timidity_mutex_destroy(miditrace->busy);
	free(miditrace);
}


