/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2002 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef ___URL_H_
#define ___URL_H_
#include "tmdy_struct.h"
/* This header file from liburl-1.8.3.
 * You can get full source from:
 * http://www.goice.co.jp/member/mo/release/index.html#liburl
 */


#define URL_LIB_VERSION "1.9.5"

/* Define if you want to enable pipe command scheme ("command|") */
#define PIPE_SCHEME_ENABLE

/* Define if you want to appended on a user's home directory if a filename
 * is beginning with '~'
 */
#if !defined(__MACOS__) && !defined(TMDY___W32__)
#define TILD_SCHEME_ENABLE
#endif

/* Define if you want to use soft directory cache */
#ifndef URL_DIR_CACHE_DISABLE
#define URL_DIR_CACHE_ENABLE
#endif /* URL_DIR_CACHE_DISABLE */

/* Define if you want to use XOVER command in NNTP */
/* #define URL_NEWS_XOVER_SUPPORT "XOVER", "XOVERVIEW" */

/* M:Must, O:Optional defined */
typedef struct s_URL
{
    int   type;								/* M */

    long  (* url_read)(tmdy_struct_ex_t *tmdy_struct, struct s_URL *url, void *buff, long n);		/* M */
    char *(* url_gets)(tmdy_struct_ex_t *tmdy_struct, struct s_URL *url, char *buff, int n);		/* O */
    int   (* url_fgetc)(tmdy_struct_ex_t *tmdy_struct, struct s_URL *url);				/* O */
    long  (* url_seek)(tmdy_struct_ex_t *tmdy_struct, struct s_URL *url, long offset, int whence);	/* O */
    long  (* url_tell)(tmdy_struct_ex_t *tmdy_struct, struct s_URL *url);				/* O */
    void  (* url_close)(tmdy_struct_ex_t *tmdy_struct, struct s_URL *url);				/* M */

    unsigned long nread;	/* Reset in url_seek, url_rewind,
				   url_set_readlimit */
    unsigned long readlimit;
    int		  eof;		/* Used in url_nread and others */
} *URL;


enum url_errtypes
{
    URLERR_NONE = 10000,	/* < 10000 represent system call's errno */
    URLERR_NOURL,		/* Unknown URL */
    URLERR_OPERM,		/* Operation not permitted */
    URLERR_CANTOPEN,		/* Can't open a URL */
    URLERR_IURLF,		/* Invalid URL form */
    URLERR_URLTOOLONG,		/* URL too long */
    URLERR_NOMAILADDR,		/* No mail address */
    URLERR_MAXNO
};

struct URL_module
{
    /* url type */
    int type;

    /* URL checker */
    int (* name_check)(tmdy_struct_ex_t *tmdy_struct, char *url_string);

    /* Once call just before url_open(). */
    int (* url_init)(tmdy_struct_ex_t *tmdy_struct);

    /* Open specified URL */
    URL (* url_open)(tmdy_struct_ex_t *tmdy_struct, char *url_string);

    /* chain next modules */
    struct URL_module *chain;
};

#define URLm(url, m) (((URL)url)->m)



struct s_url_ex_t {
	timidity_mutex_t  busy;

int (*url_eof)(tmdy_struct_ex_t *tmdy_struct, URL url);
int (*url_getc)(tmdy_struct_ex_t *tmdy_struct, URL url);
	
/* open URL stream */
URL (*url_open)(tmdy_struct_ex_t *tmdy_struct, char *url_string);

/* close URL stream */
void (*url_close)(tmdy_struct_ex_t *tmdy_struct, URL url);

/* read n bytes */
long (*url_read)(tmdy_struct_ex_t *tmdy_struct, URL url, void *buff, long n);
long (*url_safe_read)(tmdy_struct_ex_t *tmdy_struct, URL url, void *buff, long n);
long (*url_nread)(tmdy_struct_ex_t *tmdy_struct, URL url, void *buff, long n);

/* read a line */
/* Like a fgets */
char *(*url_gets)(tmdy_struct_ex_t *tmdy_struct, URL url, char *buff, int n);

/* Allow termination by CR or LF or both. Ignored empty lines.
 * CR or LF is truncated.
 * Success: length of the line.
 * EOF or Error: EOF
 */
int (*url_readline)(tmdy_struct_ex_t *tmdy_struct, URL url, char *buff, int n);

/* read a byte */
int (*url_fgetc)(tmdy_struct_ex_t *tmdy_struct, URL url);


/* seek position */
long (*url_seek)(tmdy_struct_ex_t *tmdy_struct, URL url, long offset, int whence);

/* get the current position */
long (*url_tell)(tmdy_struct_ex_t *tmdy_struct, URL url);

/* skip n bytes */
void (*url_skip)(tmdy_struct_ex_t *tmdy_struct, URL url, long n);

/* seek to first position */
void (*url_rewind)(tmdy_struct_ex_t *tmdy_struct, URL url);

/* dump */
void *(*url_dump)(tmdy_struct_ex_t *tmdy_struct, URL url, long nbytes, long *real_read);

/* set read limit */
void (*url_set_readlimit)(tmdy_struct_ex_t *tmdy_struct, URL url, long readlimit);

/* url_errno to error message */
char *(*url_strerror)(tmdy_struct_ex_t *tmdy_struct, int no);

/* allocate URL structure */
URL (*alloc_url)(tmdy_struct_ex_t *tmdy_struct, int size);

/* Check URL type. */
int (*url_check_type)(tmdy_struct_ex_t *tmdy_struct, char *url_string);

/* replace `~' to user directory */
char *(*url_expand_home_dir)(tmdy_struct_ex_t *tmdy_struct, char *filename);
char *(*url_unexpand_home_dir)(tmdy_struct_ex_t *tmdy_struct, char *filename);

int *url_errno_p;

void (*url_add_module)(tmdy_struct_ex_t *tmdy_struct, struct URL_module *m);
void (*url_add_modules)(tmdy_struct_ex_t *tmdy_struct, struct URL_module *m, ...);

URL (*url_file_open)(tmdy_struct_ex_t *tmdy_struct, char *filename);
//#ifndef __MACOS__
URL (*url_dir_open)(tmdy_struct_ex_t *tmdy_struct, char *directory_name);
//#endif
//#ifdef TMDY_HAVE_POPEN
URL (*url_pipe_open)(tmdy_struct_ex_t *tmdy_struct, char *command);
//#endif
//#ifdef TMDY_SUPPORT_SOCKET
URL (*url_http_open)(tmdy_struct_ex_t *tmdy_struct, char *url_string);
URL (*url_ftp_open)(tmdy_struct_ex_t *tmdy_struct, char *url_string);
URL (*url_newsgroup_open)(tmdy_struct_ex_t *tmdy_struct, char *url_string);
URL (*url_news_open)(tmdy_struct_ex_t *tmdy_struct, char *url_string);
//#endif

/* No URL_module */
URL (*url_mem_open)(tmdy_struct_ex_t *tmdy_struct, char *memory, long memsiz, int autofree);
URL (*url_inflate_open)(tmdy_struct_ex_t *tmdy_struct, URL instream, long compsize, int autoclose);
URL (*url_buff_open)(tmdy_struct_ex_t *tmdy_struct, URL url, int autoclose);
URL (*url_cache_open)(tmdy_struct_ex_t *tmdy_struct, URL url, int autoclose);
void (*url_cache_detach)(tmdy_struct_ex_t *tmdy_struct, URL url);
void (*url_cache_disable)(tmdy_struct_ex_t *tmdy_struct, URL url);
URL (*url_uudecode_open)(tmdy_struct_ex_t *tmdy_struct, URL reader, int autoclose);
URL (*url_b64decode_open)(tmdy_struct_ex_t *tmdy_struct, URL reader, int autoclose);
URL (*url_hqxdecode_open)(tmdy_struct_ex_t *tmdy_struct, URL reader, int dataonly, int autoclose);
URL (*url_qsdecode_open)(tmdy_struct_ex_t *tmdy_struct, URL reader, int autoclose);
char *(*url_dir_name)(tmdy_struct_ex_t *tmdy_struct, URL url);
//#ifdef TMDY_SUPPORT_SOCKET
/*
URL (*url_cgi_escape_open)(tmdy_struct_ex_t *tmdy_struct, URL reader, int autoclose);
URL (*url_cgi_unescape_open)(tmdy_struct_ex_t *tmdy_struct, URL reader, int autoclose);
*/
char *(*url_newsgroup_name)(tmdy_struct_ex_t *tmdy_struct, URL url);
int (*url_news_connection_cache)(tmdy_struct_ex_t *tmdy_struct, int flag);
//#endif 

char *url_lib_version;
char *user_mailaddr;
char *url_user_agent;
//#ifdef TMDY_SUPPORT_SOCKET
char *url_http_proxy_host;
unsigned short *url_http_proxy_port_p;
char *url_ftp_proxy_host;
unsigned short *url_ftp_proxy_port_p;
//#endif
int *url_newline_code_p;
int *uudecode_unquote_html_p;

struct URL_module *URL_module_file_p;
//#ifndef __MACOS__
struct URL_module *URL_module_dir_p;
//#endif /* __MACOS__ */
//#ifdef TMDY_SUPPORT_SOCKET
struct URL_module *URL_module_http_p;
struct URL_module *URL_module_ftp_p;
struct URL_module *URL_module_news_p;
struct URL_module *URL_module_newsgroup_p;
//#endif /* SUPPORT_SOCKET */
//#if !defined(__MACOS__) && defined(TMDY_HAVE_POPEN)
struct URL_module *URL_module_pipe_p;
//#endif /* HAVE_POPEN */
};

url_ex_t* new_url(tmdy_struct_ex_t *tmdy_struct);
void destroy_url(url_ex_t* url);


enum url_types
{
    URL_none_t,			/* Undefined URL */
    URL_file_t,			/* File system */
    URL_dir_t,			/* Directory entry */
    URL_http_t,			/* HTTP */
    URL_ftp_t,			/* FTP */
    URL_news_t,			/* NetNews article */
    URL_newsgroup_t,		/* NetNews group */
    URL_pipe_t,			/* Pipe */
    URL_mem_t,			/* On memory */
    URL_buff_t,			/* Buffered stream */
    URL_cache_t,		/* Cached stream */
    URL_uudecode_t,		/* UU decoder */
    URL_b64decode_t,		/* Base64 decoder */
    URL_qsdecode_t,		/* Quoted-string decoder */
    URL_hqxdecode_t,		/* HQX decoder */
    URL_cgi_escape_t,		/* WWW CGI Escape */
    URL_cgi_unescape_t,		/* WWW CGI Unescape */
    URL_arc_t,			/* arc stream */

    URL_inflate_t = 99,		/* LZ77 decode stream */

    URL_extension_t = 100	/* extentional stream >= 100 */
};

enum url_news_conn_type
{
    URL_NEWS_CONN_NO_CACHE,
    URL_NEWS_CONN_CACHE,
    URL_NEWS_CLOSE_CACHE,
    URL_NEWS_GET_FLAG
};

#define IS_URL_SEEK_SAFE(url) ((url)->url_seek != NULL && \
			       (url)->type != URL_buff_t)

#define URL_MAX_READLIMIT ((~(unsigned long)0) >> 1)
#endif /* ___URL_H_ */
