/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2002 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    alsaseq_c.c - ALSA sequencer server interface
        Copyright (c) 2000  Takashi Iwai <tiwai@suse.de>

    This interface provides an ALSA sequencer client which receives 
    events and plays it in real-time.  On this mode, TiMidity works
    as a software (quasi-)real-time MIDI synth engine.

    See doc/C/README.alsaseq for more details.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <sched.h>
#include <sys/types.h>
#include <sys/time.h>
#ifndef NO_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif
#include <math.h>
#include <signal.h>

#if HAVE_ALSA_ASOUNDLIB_H
#include <alsa/asoundlib.h>
#else
#include <sys/asoundlib.h>
#endif

#include "timidity.h"
#include "common.h"
#include "controls.h"
#include "instrum.h"
#include "playmidi.h"
#include "readmidi.h"
#include "recache.h"
#include "output.h"
#include "aq.h"
#include "timer.h"

#include "alsaseq_c.h"
#include "alsaseq_c_prv.h"

static tmdy_struct_ex_t *tmdy_st_alsa_c;

#define MAX_PORTS	16

#define TICKTIME_HZ	100

struct seq_context {
	snd_seq_t *handle;	/* The snd_seq handle to /dev/snd/seq */
	int client;		/* The client associated with this context */
	int num_ports;		/* number of ports */
	int port[MAX_PORTS];	/* created sequencer ports */
	int fd;			/* The file descriptor */
	int used;		/* number of current connection */
	int active;		/* */
	int queue;
	snd_seq_queue_status_t *q_status;
};

static struct seq_context alsactx;

#if SND_LIB_MINOR >= 6
/* !! this is a dirty hack.  not sure to work in future !! */
static int snd_seq_file_descriptor(tmdy_struct_ex_t *tmdy_struct, snd_seq_t *handle)
{
	int pfds = snd_seq_poll_descriptors_count(handle, POLLIN);
	if (pfds > 0) {
		struct pollfd pfd;
		if (snd_seq_poll_descriptors(handle, &pfd, 1, POLLIN) >= 0)
			return pfd.fd;
	}
	return -ENXIO;
}

static int alsa_seq_open(tmdy_struct_ex_t *tmdy_struct, snd_seq_t **seqp)
{
	return snd_seq_open(seqp, "hw", SND_SEQ_OPEN_DUPLEX, 0);
}

static int alsa_create_port(snd_seq_t *seq, int index)
{
	char name[32];
	int port;

	sprintf(name, "TiMidity port %d", index);
	port = snd_seq_create_simple_port(seq, name,
					  SND_SEQ_PORT_CAP_WRITE|SND_SEQ_PORT_CAP_SUBS_WRITE,
					  SND_SEQ_PORT_TYPE_MIDI_GENERIC);
	if (port < 0) {
		fprintf(stderr, "error in snd_seq_create_simple_port\n");
		return -1;
	}

	return port;
}

#else
static int alsa_seq_open(tmdy_struct_ex_t *tmdy_struct, snd_seq_t **seqp)
{
	return snd_seq_open(seqp, SND_SEQ_OPEN_IN);
}

static int alsa_create_port(snd_seq_t *seq, int index)
{
	snd_seq_port_info_t pinfo;

	memset(&pinfo, 0, sizeof(pinfo));
	sprintf(pinfo.name, "TiMidity port %d", index);
	pinfo.capability = SND_SEQ_PORT_CAP_WRITE|SND_SEQ_PORT_CAP_SUBS_WRITE;
	pinfo.type = SND_SEQ_PORT_TYPE_MIDI_GENERIC;
	strcpy(pinfo.group, SND_SEQ_GROUP_DEVICE);
	if (snd_seq_create_port(alsactx.handle, &pinfo) < 0) {
		fprintf(stderr, "error in snd_seq_create_simple_port\n");
		return -1;
	}
	return pinfo.port;
}

#endif

static void alsa_set_timestamping(tmdy_struct_ex_t *tmdy_struct, struct seq_context *ctxp, int port)
{
#if HAVE_SND_SEQ_PORT_INFO_SET_TIMESTAMPING
	int q;
	snd_seq_port_info_t *pinfo;

	if (ctxp->queue < 0) {
		q = snd_seq_alloc_queue(ctxp->handle);
		ctxp->queue = q;
		if (q < 0)
			return;
		if (snd_seq_queue_status_malloc(&ctxp->q_status) < 0) {
			fprintf(stderr, "no memory!\n");
			exit(1);
		}
	}

	snd_seq_port_info_alloca(&pinfo);
	if (snd_seq_get_port_info(ctxp->handle, port, pinfo) < 0)
		return;
	snd_seq_port_info_set_timestamping(pinfo, 1);
	snd_seq_port_info_set_timestamp_real(pinfo, 1);
	snd_seq_port_info_set_timestamp_queue(pinfo, q);
	if (snd_seq_set_port_info(ctxp->handle, port, pinfo) < 0)
		return;
#endif
}


static int ctl_open(tmdy_struct_ex_t *tmdy_struct, int using_stdin, int using_stdout);
static void ctl_close(tmdy_struct_ex_t *tmdy_struct);
static int ctl_read(tmdy_struct_ex_t *tmdy_struct, int32 *valp);
static int cmsg(tmdy_struct_ex_t *tmdy_struct, int type, int verbosity_level, char *fmt, ...);
static void ctl_event(tmdy_struct_ex_t *tmdy_struct, CtlEvent *e);
static void ctl_pass_playing_list(tmdy_struct_ex_t *tmdy_struct, int n, char *args[]);

/**********************************/
/* export the interface functions */

#define ctl alsaseq_control_mode

ControlMode ctl=
{
    "ALSA sequencer interface", 'A',
    1,0,0,
    0,
    ctl_open,
    ctl_close,
    ctl_pass_playing_list,
    ctl_read,
    cmsg,
    ctl_event
};

/* options */
#define OPT_REALTIMTE_PRIORITY 0
#define OPT_SEQUENCER_PORTS 4

/*
int ASEQ->opt_realtime_priority = 0;
int ASEQ->opt_sequencer_ports = 4;

static int ASEQ->buffer_time_advance;
static long ASEQ->buffer_time_offset;
static long ASEQ->start_time_base;
static long ASEQ->cur_time_offset;
static long ASEQ->last_queue_offset;
static double ASEQ->rate_frac, ASEQ->rate_frac_nsec;
static FILE *ASEQ->outfp;
*/
/*ARGSUSED*/
static int ctl_open(tmdy_struct_ex_t *tmdy_struct, int using_stdin, int using_stdout)
{
	tmdy_st_alsa_c=tmdy_struct;
	
	ctl.opened = 1;
	ctl.flags &= ~(CTLF_LIST_RANDOM|CTLF_LIST_SORT);
	if (using_stdout)
		ASEQ->outfp = stderr;
	else
		ASEQ->outfp = stdout;
	return 0;
}

static void ctl_close(tmdy_struct_ex_t *tmdy_struct)
{
	if (!ctl.opened)
		return;
}

static int ctl_read(tmdy_struct_ex_t *tmdy_struct, int32 *valp)
{
    return RC_NONE;
}

static int cmsg(tmdy_struct_ex_t *tmdy_struct, int type, int verbosity_level, char *fmt, ...)
{
    va_list ap;

    if((type==CMSG_TEXT || type==CMSG_INFO || type==CMSG_WARNING) &&
       ctl.verbosity < verbosity_level)
	return 0;

    if(ASEQ->outfp == NULL)
	ASEQ->outfp = stderr;

    va_start(ap, fmt);
    vfprintf(ASEQ->outfp, fmt, ap);
    fputs(NLS, ASEQ->outfp);
    fflush(ASEQ->outfp);
    va_end(ap);

    return 0;
}

static void ctl_event(tmdy_struct_ex_t *tmdy_struct, CtlEvent *e)
{
}

static RETSIGTYPE sig_timeout(int sig)
{
    signal(SIGALRM, sig_timeout); /* For SysV base */
    /* Expect EINTR */
}

static void doit(tmdy_struct_ex_t *tmdy_struct, struct seq_context *ctxp);
static int do_sequencer(tmdy_struct_ex_t *tmdy_struct, struct seq_context *ctxp);
static int start_sequencer(tmdy_struct_ex_t *tmdy_struct, struct seq_context *ctxp);
static void stop_sequencer(tmdy_struct_ex_t *tmdy_struct, struct seq_context *ctxp);
static void server_reset(tmdy_struct_ex_t *tmdy_struct);

/* reset all when SIGHUP is received */
static RETSIGTYPE sig_reset(int sig)
{
	if (alsactx.active) {
		stop_sequencer(tmdy_st_alsa_c, &alsactx);
		server_reset(tmdy_st_alsa_c);
	}
	signal(SIGHUP, sig_reset);
}
static RETSIGTYPE sig_safe_exit(int sig)
{
	tmdy_st_alsa_c->common->safe_exit(tmdy_st_alsa_c,1);
}
/*
 * set the process to realtime privs
 */
static int set_realtime_priority(tmdy_struct_ex_t *tmdy_struct)
{
	struct sched_param schp;
	int max_prio;

	if (ASEQ->opt_realtime_priority <= 0)
		return 0;

        memset(&schp, 0, sizeof(schp));
	max_prio = sched_get_priority_max(SCHED_FIFO);
	if (max_prio < ASEQ->opt_realtime_priority)
		ASEQ->opt_realtime_priority = max_prio;

	schp.sched_priority = ASEQ->opt_realtime_priority;
        if (sched_setscheduler(0, SCHED_FIFO, &schp) != 0) {
		printf("can't set sched_setscheduler - using normal priority\n");
                return -1;
        }
	/* drop root priv. */
	if (! geteuid() && getuid() != geteuid()) {
		if (setuid(getuid()))
			perror("dropping root priv");
	}
	printf("set SCHED_FIFO(%d)\n", ASEQ->opt_realtime_priority);
        return 0;
}

static void ctl_pass_playing_list(tmdy_struct_ex_t *tmdy_struct, int n, char *args[])
{
	double btime;
	int i, j;

#ifdef SIGPIPE
	signal(SIGPIPE, SIG_IGN);    /* Handle broken pipe */
#endif /* SIGPIPE */

	printf("TiMidity starting in ALSA server mode\n");

	set_realtime_priority(tmdy_struct);

	if (alsa_seq_open(tmdy_struct, &alsactx.handle) < 0) {
		fprintf(stderr, "error in snd_seq_open\n");
		return;
	}
	alsactx.queue = -1;
	alsactx.client = snd_seq_client_id(alsactx.handle);
	alsactx.fd = snd_seq_file_descriptor(tmdy_struct, alsactx.handle);
	snd_seq_set_client_name(alsactx.handle, "TiMidity");
	snd_seq_set_client_pool_input(alsactx.handle, 1000); /* enough? */
	if (ASEQ->opt_sequencer_ports < 1)
		alsactx.num_ports = 1;
	else if (ASEQ->opt_sequencer_ports > MAX_PORTS)
		alsactx.num_ports = MAX_PORTS;
	else
		alsactx.num_ports = ASEQ->opt_sequencer_ports;

	printf("Opening sequencer port:");
	for (i = 0; i < alsactx.num_ports; i++) {
		int port;
		port = alsa_create_port(alsactx.handle, i);
		if (port < 0)
			return;
		alsactx.port[i] = port;
		alsa_set_timestamping(tmdy_struct, &alsactx, port);
		printf(" %d:%d", alsactx.client, alsactx.port[i]);
	}
	printf("\n");

	alsactx.used = 0;
	alsactx.active = 0;

	TMDY_PLAYMIDI->opt_realtime_playing = 1; /* Enable loading patch while playing */
	TMDY_RECACHE->allocate_cache_size = 0; /* Don't use pre-calclated samples */
	TMDY_PLAYMIDI->current_keysig = TMDY_PLAYMIDI->opt_init_keysig;
	TMDY_PLAYMIDI->note_key_offset = 0;

	if (IS_STREAM_TRACE) {
		/* set the audio queue size as minimum as possible, since
		 * we don't have to use audio queue..
		 */
		printf("guha ha\n");
		TMDY_OUTPUT->play_mode->acntl(tmdy_struct, PM_REQ_GETFRAGSIZ, &ASEQ->buffer_time_advance);
		if (!(TMDY_OUTPUT->play_mode->encoding & PE_MONO))
			ASEQ->buffer_time_advance >>= 1;
		if (TMDY_OUTPUT->play_mode->encoding & PE_16BIT)
			ASEQ->buffer_time_advance >>= 1;

		btime = (double)ASEQ->buffer_time_advance / TMDY_OUTPUT->play_mode->rate;
		btime *= 1.01; /* to be sure */
		(TMDY_AQ->aq_set_soft_queue)(tmdy_struct, btime, 0.0);
	} else {
		ASEQ->buffer_time_advance = 0;
	}
	ASEQ->rate_frac = (double)TMDY_OUTPUT->play_mode->rate / 1000000.0;
	ASEQ->rate_frac_nsec = (double)TMDY_OUTPUT->play_mode->rate / 1000000000.0;

	alarm(0);
	signal(SIGALRM, sig_timeout);
	signal(SIGINT, sig_safe_exit);
	signal(SIGTERM, sig_safe_exit);
	signal(SIGHUP, sig_reset);

	if (TMDY_PLAYMIDI->opt_force_keysig != 8) {
		i = TMDY_PLAYMIDI->current_keysig + ((TMDY_PLAYMIDI->current_keysig < 8) ? 7 : -6);
		j = TMDY_PLAYMIDI->opt_force_keysig + ((TMDY_PLAYMIDI->current_keysig < 8) ? 7 : 10);
		while (i != j && i != j + 12) {
			if (++TMDY_PLAYMIDI->note_key_offset > 6)
				TMDY_PLAYMIDI->note_key_offset -= 12;
			i += (i > 10) ? -5 : 7;
		}
	}
	i = TMDY_PLAYMIDI->current_keysig + ((TMDY_PLAYMIDI->current_keysig < 8) ? 7 : -9), j = 0;
	while (i != 7 && i != 19)
		i += (i < 7) ? 5 : -7, j++;
	j += TMDY_PLAYMIDI->note_key_offset, j -= floor(j / 12.0) * 12;
	TMDY_PLAYMIDI->current_freq_table = j;

	TMDY_OUTPUT->play_mode->close_output(tmdy_struct);

	if (ctl.flags & CTLF_DAEMONIZE)
	{
		int pid = fork();
		FILE *pidf;
		switch (pid)
		{
			case 0:			// child is the daemon
				break;
			case -1:		// error status return
				exit(7);
			default:		// no error, doing well
				if ((pidf = fopen( "/var/run/timidity.pid", "w" )) != NULL )
					fprintf( pidf, "%d\n", pid );
				exit(0);
		}
	}

	for (;;) {
		server_reset(tmdy_struct);
		doit(tmdy_struct, &alsactx);
	}
}

/*
 * get the current time in usec from gettimeofday()
 */
static long get_current_time(tmdy_struct_ex_t *tmdy_struct)
{
	struct timeval tv;
	long t;

	gettimeofday(&tv, NULL);
	t = tv.tv_sec * 1000000L + tv.tv_usec;
	return t - ASEQ->start_time_base;
}
	
/*
 * convert from snd_seq_real_time_t to sample count
 */
inline static long queue_time_to_position(tmdy_struct_ex_t *tmdy_struct, const snd_seq_real_time_t *t)
{
	return (long)t->tv_sec * TMDY_OUTPUT->play_mode->rate + (long)(t->tv_nsec * ASEQ->rate_frac_nsec);
}

/*
 * get the current queue position in sample count
 */
static long get_current_queue_position(tmdy_struct_ex_t *tmdy_struct, struct seq_context *ctxp)
{
#if HAVE_SND_SEQ_PORT_INFO_SET_TIMESTAMPING
	snd_seq_get_queue_status(ctxp->handle, ctxp->queue, ctxp->q_status);
	return queue_time_to_position(tmdy_struct, snd_seq_queue_status_get_real_time(ctxp->q_status));
#else
	return 0;
#endif
}

/*
 * update the current position from the event timestamp
 */
static void update_timestamp_from_event(tmdy_struct_ex_t *tmdy_struct, snd_seq_event_t *ev)
{
	long t = queue_time_to_position(tmdy_struct, &ev->time.time) - ASEQ->last_queue_offset;
	if (t < 0) {
		// fprintf(stderr, "timestamp underflow! (delta=%d)\n", (int)t);
		t = 0;
	} else if (ASEQ->buffer_time_advance > 0 && t >= ASEQ->buffer_time_advance) {
		// fprintf(stderr, "timestamp overflow! (delta=%d)\n", (int)t);
		t = ASEQ->buffer_time_advance - 1;
	}
	t += ASEQ->buffer_time_offset;
	if (t >= ASEQ->cur_time_offset)
		ASEQ->cur_time_offset = t;
}

/*
 * update the current position from system time
 */
static void update_timestamp(tmdy_struct_ex_t *tmdy_struct)
{
	ASEQ->cur_time_offset = (long)(get_current_time(tmdy_struct) * ASEQ->rate_frac);
}

static void seq_play_event(tmdy_struct_ex_t *tmdy_struct, MidiEvent *ev)
{
	//JAVE  make channel -Q channels quiet, modified some code from readmidi.c
	int gch;
	gch = GLOBAL_CHANNEL_EVENT_TYPE(ev->type);

	if (gch || !IS_SET_CHANNELMASK(TMDY_READMIDI->quietchannels, ev->channel)){
		//if its a global event or not a masked event
		ev->time = ASEQ->cur_time_offset;
		TMDY_PLAYMIDI->play_event(tmdy_struct, ev);
	}
}

static void stop_playing(tmdy_struct_ex_t *tmdy_struct)
{
	if(TMDY_PLAYMIDI->upper_voices) {
		MidiEvent ev;
		ev.type = ME_EOT;
		ev.a = 0;
		ev.b = 0;
		seq_play_event(tmdy_struct, &ev);
		(TMDY_AQ->aq_flush)(tmdy_struct, 0);
	}
}

static void doit(tmdy_struct_ex_t *tmdy_struct, struct seq_context *ctxp)
{
	for (;;) {
		while (snd_seq_event_input_pending(ctxp->handle, 1)) {
			if (do_sequencer(tmdy_struct, ctxp))
				goto __done;
		}
		if (ctxp->active) {
			MidiEvent ev;

			if (IS_STREAM_TRACE) {
				/* remember the last update position */
				if (ctxp->queue >= 0)
					ASEQ->last_queue_offset = get_current_queue_position(tmdy_struct, ctxp);
				/* advance the buffer position */
				ASEQ->buffer_time_offset += ASEQ->buffer_time_advance;
				ev.time = ASEQ->buffer_time_offset;
			} else {
				/* update the current position */
				if (ctxp->queue >= 0)
					ASEQ->cur_time_offset = get_current_queue_position(tmdy_struct, ctxp);
				else
					update_timestamp(tmdy_struct);
				ev.time = ASEQ->cur_time_offset;
			}
			ev.type = ME_NONE;
			TMDY_PLAYMIDI->play_event(tmdy_struct, &ev);
			(TMDY_AQ->aq_fill_nonblocking)(tmdy_struct);
		}
		if (! ctxp->active || ! IS_STREAM_TRACE) {
			fd_set rfds;
			struct timeval timeout;
			FD_ZERO(&rfds);
			FD_SET(ctxp->fd, &rfds);
			timeout.tv_sec = 0;
			timeout.tv_usec = 10000; /* 10ms */
			if (select(ctxp->fd + 1, &rfds, NULL, NULL, &timeout) < 0)
				goto __done;
		}
	}

__done:
	if (ctxp->active) {
		stop_sequencer(tmdy_struct, ctxp);
	}
}

static void server_reset(tmdy_struct_ex_t *tmdy_struct)
{
	TMDY_READMIDI->readmidi_read_init(tmdy_struct);
	TMDY_PLAYMIDI->playmidi_stream_init(tmdy_struct);
	if ((TMDY_INSTRUM->free_instruments_afterwards))
		(TMDY_INSTRUM->free_instruments)(tmdy_struct, 0);
	TMDY_PLAYMIDI->reduce_voice_threshold = 0; /* Disable auto reduction voice */
	ASEQ->buffer_time_offset = 0;
}

static int start_sequencer(tmdy_struct_ex_t *tmdy_struct, struct seq_context *ctxp)
{
	if (TMDY_OUTPUT->play_mode->open_output(tmdy_struct) < 0) {
		ctl.cmsg(tmdy_struct, CMSG_FATAL, VERB_NORMAL,
			 "Couldn't open %s (`%c')",
			 TMDY_OUTPUT->play_mode->id_name, TMDY_OUTPUT->play_mode->id_character);
		return 0;
	}
	ctxp->active = 1;

	ASEQ->buffer_time_offset = 0;
	ASEQ->last_queue_offset = 0;
	ASEQ->cur_time_offset = 0;
	if (ctxp->queue >= 0) {
		if (snd_seq_start_queue(ctxp->handle, ctxp->queue, NULL) < 0)
			ctxp->queue = -1;
		else
			snd_seq_drain_output(ctxp->handle);
	}
	if (ctxp->queue < 0) {
		ASEQ->start_time_base = 0;
		ASEQ->start_time_base = get_current_time(tmdy_struct);
	}

	return 1;
}

static void stop_sequencer(tmdy_struct_ex_t *tmdy_struct, struct seq_context *ctxp)
{
	stop_playing(tmdy_struct);
	if (ctxp->queue >= 0) {
		snd_seq_stop_queue(ctxp->handle, ctxp->queue, NULL);
		snd_seq_drain_output(ctxp->handle);
	}
	TMDY_OUTPUT->play_mode->close_output(tmdy_struct);
	(TMDY_INSTRUM->free_instruments)(tmdy_struct, 0);
	TMDY_UTILS->mblock->free_global_mblock(tmdy_struct);
	ctxp->used = 0;
	ctxp->active = 0;
}

#define NOTE_CHAN(ev)	((ev)->dest.port * 16 + (ev)->data.note.channel)
#define CTRL_CHAN(ev)	((ev)->dest.port * 16 + (ev)->data.control.channel)

static int do_sequencer(tmdy_struct_ex_t *tmdy_struct, struct seq_context *ctxp)
{
	int n, ne, i;
	MidiEvent ev, evm[260];
	snd_seq_event_t *aevp;

	n = snd_seq_event_input(ctxp->handle, &aevp);
	if (n < 0 || aevp == NULL)
		return 0;

	if (ctxp->active && ctxp->queue >= 0)
		update_timestamp_from_event(tmdy_struct, aevp);
	else if (IS_STREAM_TRACE)
		ASEQ->cur_time_offset = ASEQ->buffer_time_offset;
	else
		update_timestamp(tmdy_struct);

	switch(aevp->type) {
	case SND_SEQ_EVENT_NOTEON:
		ev.channel = NOTE_CHAN(aevp);
		ev.a       = aevp->data.note.note;
		ev.b       = aevp->data.note.velocity;
		if (ev.b == 0)
			ev.type = ME_NOTEOFF;
		else
			ev.type = ME_NOTEON;
		seq_play_event(tmdy_struct, &ev);
		break;

	case SND_SEQ_EVENT_NOTEOFF:
		ev.channel = NOTE_CHAN(aevp);
		ev.a       = aevp->data.note.note;
		ev.b       = aevp->data.note.velocity;
		ev.type = ME_NOTEOFF;
		seq_play_event(tmdy_struct, &ev);
		break;

	case SND_SEQ_EVENT_KEYPRESS:
		ev.channel = NOTE_CHAN(aevp);
		ev.a       = aevp->data.note.note;
		ev.b       = aevp->data.note.velocity;
		ev.type = ME_KEYPRESSURE;
		seq_play_event(tmdy_struct, &ev);
		break;

	case SND_SEQ_EVENT_PGMCHANGE:
		ev.channel = CTRL_CHAN(aevp);
		ev.a = aevp->data.control.value;
		ev.type = ME_PROGRAM;
		seq_play_event(tmdy_struct, &ev);
		break;

	case SND_SEQ_EVENT_CONTROLLER:
		if(TMDY_READMIDI->convert_midi_control_change(tmdy_struct, CTRL_CHAN(aevp),
					       aevp->data.control.param,
					       aevp->data.control.value,
					       &ev))
			seq_play_event(tmdy_struct, &ev);
		break;

	case SND_SEQ_EVENT_CONTROL14:
		if (aevp->data.control.param < 0 || aevp->data.control.param >= 32)
			break;
		if (! TMDY_READMIDI->convert_midi_control_change(tmdy_struct, CTRL_CHAN(aevp),
						  aevp->data.control.param,
						  (aevp->data.control.value >> 7) & 0x7f,
						  &ev))
			break;
		seq_play_event(tmdy_struct, &ev);
		if (! TMDY_READMIDI->convert_midi_control_change(tmdy_struct, CTRL_CHAN(aevp),
						  aevp->data.control.param + 32,
						  aevp->data.control.value & 0x7f,
						  &ev))
			break;
		seq_play_event(tmdy_struct, &ev);
		break;
		    
	case SND_SEQ_EVENT_PITCHBEND:
		ev.type    = ME_PITCHWHEEL;
		ev.channel = CTRL_CHAN(aevp);
		aevp->data.control.value += 0x2000;
		ev.a       = (aevp->data.control.value) & 0x7f;
		ev.b       = (aevp->data.control.value>>7) & 0x7f;
		seq_play_event(tmdy_struct, &ev);
		break;

	case SND_SEQ_EVENT_CHANPRESS:
		ev.type    = ME_CHANNEL_PRESSURE;
		ev.channel = CTRL_CHAN(aevp);
		ev.a       = aevp->data.control.value;
		seq_play_event(tmdy_struct, &ev);
		break;
		
	case SND_SEQ_EVENT_NONREGPARAM:
		/* Break it back into its controler values */
		ev.type = ME_NRPN_MSB;
		ev.channel = CTRL_CHAN(aevp);
		ev.a = (aevp->data.control.param >> 7) & 0x7f;
		seq_play_event(tmdy_struct, &ev);
		ev.type = ME_NRPN_LSB;
		ev.channel = CTRL_CHAN(aevp);
		ev.a = aevp->data.control.param & 0x7f;
		seq_play_event(tmdy_struct, &ev);
		ev.type = ME_DATA_ENTRY_MSB;
		ev.channel = CTRL_CHAN(aevp);
		ev.a = (aevp->data.control.value >> 7) & 0x7f;
		seq_play_event(tmdy_struct, &ev);
		ev.type = ME_DATA_ENTRY_LSB;
		ev.channel = CTRL_CHAN(aevp);
		ev.a = aevp->data.control.value & 0x7f;
		seq_play_event(tmdy_struct, &ev);
		break;

	case SND_SEQ_EVENT_REGPARAM:
		/* Break it back into its controler values */
		ev.type = ME_RPN_MSB;
		ev.channel = CTRL_CHAN(aevp);
		ev.a = (aevp->data.control.param >> 7) & 0x7f;
		seq_play_event(tmdy_struct, &ev);
		ev.type = ME_RPN_LSB;
		ev.channel = CTRL_CHAN(aevp);
		ev.a = aevp->data.control.param & 0x7f;
		seq_play_event(tmdy_struct, &ev);
		ev.type = ME_DATA_ENTRY_MSB;
		ev.channel = CTRL_CHAN(aevp);
		ev.a = (aevp->data.control.value >> 7) & 0x7f;
		seq_play_event(tmdy_struct, &ev);
		ev.type = ME_DATA_ENTRY_LSB;
		ev.channel = CTRL_CHAN(aevp);
		ev.a = aevp->data.control.value & 0x7f;
		seq_play_event(tmdy_struct, &ev);
		break;

	case SND_SEQ_EVENT_SYSEX:
		if (TMDY_READMIDI->parse_sysex_event(tmdy_struct, aevp->data.ext.ptr + 1,
				 aevp->data.ext.len - 1, &ev))
			seq_play_event(tmdy_struct, &ev);
		if (ne = TMDY_READMIDI->parse_sysex_event_multi(tmdy_struct, aevp->data.ext.ptr + 1,
				aevp->data.ext.len - 1, evm))
			for (i = 0; i < ne; i++)
				seq_play_event(tmdy_struct, &evm[i]);
		break;

#if SND_LIB_MINOR >= 6
#define snd_seq_addr_equal(a,b)	((a)->client == (b)->client && (a)->port == (b)->port)
	case SND_SEQ_EVENT_PORT_SUBSCRIBED:
		if (snd_seq_addr_equal(&aevp->data.connect.dest, &aevp->dest)) {
			if (! ctxp->active) {
				if (! start_sequencer(tmdy_struct, ctxp)) {
					snd_seq_free_event(aevp);
					return 0;
				}
			}
			ctxp->used++;
		}
		break;

	case SND_SEQ_EVENT_PORT_UNSUBSCRIBED:
		if (snd_seq_addr_equal(&aevp->data.connect.dest, &aevp->dest)) {
			if (ctxp->active) {
				ctxp->used--;
				if (ctxp->used <= 0) {
					snd_seq_free_event(aevp);
					return 1; /* quit now */
				}
			}
		}
		break;
#else
	case SND_SEQ_EVENT_PORT_USED:
		if (! ctxp->active) {
			if (! start_sequencer(tmdy_struct, ctxp)) {
				snd_seq_free_event(aevp);
				return 0;
			}
		}
		ctxp->used++;
		break;

	case SND_SEQ_EVENT_PORT_UNUSED:
		if (ctxp->active) {
			ctxp->used--;
			if (ctxp->used <= 0) {
				snd_seq_free_event(aevp);
				return 1; /* quit now */
			}
		}
		break;
#endif
		
	default:
		/*printf("Unsupported event %d\n", aevp->type);*/
		break;
	}
	snd_seq_free_event(aevp);
	return 0;
}

/*
 * interface_<id>_loader();
 */
ControlMode *interface_A_loader(void)
{
    return &ctl;
}




alsaseq_c_ex_t* new_alsaseq_c(tmdy_struct_ex_t *tmdy_struct){
	int i;
	alsaseq_c_ex_t* alsaseq_c_ex;

	alsaseq_c_ex=(alsaseq_c_ex_t *)TMDY_COMMON->safe_malloc(tmdy_struct, sizeof(alsaseq_c_ex_t));
	
	timidity_mutex_init(alsaseq_c_ex->busy);

	alsaseq_c_ex->opt_realtime_priority=OPT_REALTIMTE_PRIORITY;
	alsaseq_c_ex->opt_sequencer_ports=OPT_SEQUENCER_PORTS;
	
	return alsaseq_c_ex;
}
void destroy_alsaseq_c(alsaseq_c_ex_t* alsaseq_c){
	timidity_mutex_destroy(alsaseq_c->busy);
	free(alsaseq_c);
}
