/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2002 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    portaudio_a.c by Avatar <avatar@deva.net>
    based on esd_a.c

    Functions to play sound through Portaudio
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#define _GNU_SOURCE
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>

#ifndef NO_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif

//#include <windows.h>
#include <portaudio.h>


#include "timidity.h"
#include "common.h"
#include "output.h"
#include "controls.h"
#include "timer.h"
#include "instrum.h"
#include "playmidi.h"
#include "miditrace.h"

#include "portaudio_a.h"

/* followings are only needed by w32g_util.c */
#define  UTL_DATA_BLOCK_NUM      (64)
#define  UTL_DATA_BLOCK_BITS     DEFAULT_AUDIO_BUFFER_BITS


#define DATA_BLOCK_NUM      (dpm.extra_param[0])
#define SAMPLE_RATE         (48000)

#define FRAMS_PER_BUFFER    (256)

static int open_output(tmdy_struct_ex_t *tmdy_struct); /* 0=success, 1=warning, -1=fatal error */
static void close_output(tmdy_struct_ex_t *tmdy_struct);
static int output_data(tmdy_struct_ex_t *tmdy_struct, char *buf, int32 nbytes);
static int acntl(tmdy_struct_ex_t *tmdy_struct, int request, void *arg);



/* export the playback mode */

#define dpm portaudio_play_mode

PlayMode dpm = {
	(SAMPLE_RATE),
    PE_16BIT|PE_SIGNED,
    PF_PCM_STREAM|PF_BUFF_FRAGM_OPT/*|PF_CAN_TRACE*/,
    -1,
    {32}, /* PF_BUFF_FRAGM_OPT  is need for TWSYNTH */
	"Portaudio Dirver", 'p',
    NULL,
    open_output,
    close_output,
    output_data,
    acntl
};


int paCallback(  void *inputBuffer, void *outputBuffer,

                     unsigned long framesPerBuffer,

                     PaTimestamp outTime, void *userData )

{
	tmdy_struct_ex_t* tmdy_struct;
    unsigned int i;
	int finished = 0;

/* Cast data passed through stream to our structure type. */

	tmdy_struct=(tmdy_struct_ex_t*)userData;

    char *out = (char*)outputBuffer;
	if(PM->pa_data.samplesToGo < framesPerBuffer*PM->data_nbyte*PM->stereo  ){
		for(i=0;i<PM->pa_data.samplesToGo;i++){
			*out++ = *(PM->pa_data.bufpoint)++;
			if( PM->pa_data.buf+PM->bytesPerInBuffer*2 <= PM->pa_data.bufpoint ){
				PM->pa_data.bufpoint=PM->pa_data.buf;
			}
		}
		PM->pa_data.samplesToGo=0;
		for(;i<framesPerBuffer*PM->data_nbyte*PM->stereo;i++){
			*out++=0;
		}
		finished = 0;
	}else{
		for(i=0;i<framesPerBuffer*PM->data_nbyte*PM->stereo;i++){
			*out++=*(PM->pa_data.bufpoint)++;
			if( PM->pa_data.buf+PM->bytesPerInBuffer*2 <= PM->pa_data.bufpoint ){
				PM->pa_data.bufpoint=PM->pa_data.buf;
			}
		}
		PM->pa_data.samplesToGo -= framesPerBuffer*PM->data_nbyte*PM->stereo;
	}

    return finished ;

}


static int open_output(tmdy_struct_ex_t *tmdy_struct)
{
	int framesPerBuffer=FRAMS_PER_BUFFER;
	double rate;
	int n, nrates;
	PaError  err;
	
	
	if(PM->pa_active==0){
		err = Pa_Initialize();
		if( err != paNoError ) goto error;
		PM->pa_active=1;
	}


	dpm.encoding = dpm.encoding  & !((int32)PE_ULAW) & !((int32)PE_ALAW) & !((int32)PE_BYTESWAP);
	dpm.encoding = dpm.encoding|PE_SIGNED;
	PM->stereo=(dpm.encoding & PE_MONO)?1:2;
	PM->data_nbyte=(dpm.encoding & PE_16BIT)?sizeof(int16):sizeof(int8);

	PM->DeviceID=Pa_GetDefaultOutputDeviceID();
	PM->DeviceInfo=Pa_GetDeviceInfo(PM->DeviceID);
	nrates=PM->DeviceInfo->numSampleRates;
	if(nrates!=-1){
		rate=PM->DeviceInfo->sampleRates[nrates-1];
		for(n=nrates-1;n>=0;n--){
			if(dpm.rate <= PM->DeviceInfo->sampleRates[n]) rate=PM->DeviceInfo->sampleRates[n];
		}
	}else{
		rate=dpm.rate;
		if(dpm.rate < PM->DeviceInfo->sampleRates[0]) rate=PM->DeviceInfo->sampleRates[0];
		if(dpm.rate > PM->DeviceInfo->sampleRates[1]) rate=PM->DeviceInfo->sampleRates[1];
	}
	dpm.rate=(int32)rate;

	PM->pa_data.samplesToGo=0;
	PM->pa_data.bufpoint=PM->pa_data.buf;
	PM->pa_data.bufepoint=PM->pa_data.buf;
//	firsttime=1;
	PM->numBuffers=Pa_GetMinNumBuffers( framesPerBuffer, dpm.rate );
	PM->framesPerInBuffer=PM->numBuffers*framesPerBuffer;
	if(PM->framesPerInBuffer<4096) PM->framesPerInBuffer=4096;
	PM->bytesPerInBuffer=PM->framesPerInBuffer*PM->data_nbyte*PM->stereo;
//	printf("%d\n",PM->framesPerInBuffer);
//	printf("%d\n",dpm.rate);
	err = Pa_OpenDefaultStream(
    	&PM->stream,        /* passes back stream pointer */
    	0,              /* no input channels */
    	PM->stereo,              /* 2:stereo 1:mono output */
    	(dpm.encoding & PE_16BIT)?paInt16:paInt8,      /* 16 bit 8bit output */
		(double)dpm.rate,          /* sample rate */
    	framesPerBuffer,            /* frames per buffer */
    	PM->numBuffers,              /* number of buffers, if zero then use default minimum */
    	paCallback, /* specify our custom callback */
    	(void *)tmdy_struct);   /* pass our data through to callback */
	if( err != paNoError ) goto error;
	return 0;

error:
	Pa_Terminate(); PM->pa_active=0;
	TMDY_CONTROLS->ctl->cmsg(tmdy_struct,   CMSG_ERROR, VERB_NORMAL, "PortAudio error: %s\n", Pa_GetErrorText( err ) );
	return -1;
}
static int output_data(tmdy_struct_ex_t *tmdy_struct, char *buf, int32 nbytes)
{
	unsigned int i;
    PaError  err;

//	if(PM->pa_data.samplesToGo > DATA_BLOCK_SIZE){ 
//		Sleep(  (PM->pa_data.samplesToGo - DATA_BLOCK_SIZE)/dpm.rate/4  );
//	}
	for(i=0;i<nbytes;i++){
		*(PM->pa_data.bufepoint)++ = *buf++ ;
		if( PM->pa_data.buf+PM->bytesPerInBuffer*2 <= PM->pa_data.bufepoint ){
			PM->pa_data.bufepoint=PM->pa_data.buf;
		}
	}
	PM->pa_data.samplesToGo += nbytes;

/*
	if(firsttime==1){
		err = Pa_StartStream( PM->stream );

		if( err != paNoError ) goto error;
		firsttime=0;
	}
*/
	if( 0==Pa_StreamActive(PM->stream)){
		err = Pa_StartStream( PM->stream );

		if( err != paNoError ) goto error;
	}
    while(PM->pa_data.samplesToGo > PM->bytesPerInBuffer){ Pa_Sleep(1);};
//	Pa_Sleep( (PM->pa_data.samplesToGo - PM->bytesPerInBuffer)/dpm.rate * 1000);
	return 0;

error:
	Pa_Terminate(); PM->pa_active=0;
	TMDY_CONTROLS->ctl->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL, "PortAudio error: %s\n", Pa_GetErrorText( err ) );
	return -1;
}

static void close_output(tmdy_struct_ex_t *tmdy_struct)
{	
    PaError  err;
    
	if( PM->pa_active==0) return;
	if(Pa_StreamActive(PM->stream)){
		Pa_Sleep(  PM->bytesPerInBuffer/dpm.rate*1000  );
	}	
	err = Pa_StopStream( PM->stream );
//	if( err != paNoError ) goto error;
	err = Pa_CloseStream( PM->stream );
//	if( err != paNoError ) goto error;
	Pa_Terminate(); 
	PM->pa_active=0;
	return;

error:
	Pa_Terminate(); PM->pa_active=0;
	TMDY_CONTROLS->ctl->cmsg(tmdy_struct,   CMSG_ERROR, VERB_NORMAL, "PortAudio error: %s\n", Pa_GetErrorText( err ) );
	return;
}

static int acntl(tmdy_struct_ex_t *tmdy_struct, int request, void *arg)
{
    switch(request)
    {
      case PM_REQ_GETQSIZ:
		 *(int *)arg = PM->bytesPerInBuffer*2;
    	return 0;
		break;
      case PM_REQ_GETFILLABLE:
		 *(int *)arg = PM->bytesPerInBuffer*2-PM->pa_data.samplesToGo;
    	return 0;
		break;
      case PM_REQ_GETFILLED:
		 *(int *)arg = PM->pa_data.samplesToGo;
    	return 0;
		break;
      case PM_REQ_DISCARD:
    	Pa_StopStream( PM->stream );
    	close_output(tmdy_struct);
	    open_output(tmdy_struct);
		return 0;
		break;
      case PM_REQ_FLUSH:
    	close_output(tmdy_struct);
	    open_output(tmdy_struct);
		return 0;
		break;
      case PM_REQ_RATE:  /* NOT WORK */
    	{
    		int i;
    		double sampleRateBack;
    		i = *(int *)arg; /* sample rate in and out */
    		close_output(tmdy_struct);
    		sampleRateBack=dpm.rate;
    		dpm.rate=i;
    		if(0==open_output(tmdy_struct)){
    			return 0;
    		}else{    		
    			dpm.rate=sampleRateBack;
    			open_output(tmdy_struct);
    			return -1;
    		}
    	}
    	break;
    }
    return -1;
}



portaudio_a_ex_t* new_portaudio_a(tmdy_struct_ex_t *tmdy_struct){
	int i;
	portaudio_a_ex_t* portaudio_a_ex;

	portaudio_a_ex=(portaudio_a_ex_t *)TMDY_COMMON->safe_malloc(tmdy_struct, sizeof(portaudio_a_ex_t));
	
	timidity_mutex_init(portaudio_a_ex->busy);
	
	portaudio_a_ex->data_block_num = UTL_DATA_BLOCK_NUM;
	portaudio_a_ex->data_block_bits = UTL_DATA_BLOCK_BITS;
	portaudio_a_ex->stereo=2;
	portaudio_a_ex->bytesPerInBuffer=0;
	portaudio_a_ex->pa_active=0;

	
	return portaudio_a_ex;
}
void destroy_portaudio_a(portaudio_a_ex_t* portaudio_a){
	timidity_mutex_destroy(portaudio_a->busy);
	free(portaudio_a);
}
