! EratosthenesFortran.F  -*-f90-*-
! Parallelized sieve of Eratosthenes
! Time-stamp: <2007-11-15 16:13:20 takeshi>
!
! Usage:
!    $ ./EratosthenesFortran   # print version
!    $ mpdtrace && mpdboot ; mpdrun -np <numprocs> ./EratosthenesFortran <memory_size(s)>
!
! Examples:
!    $ mpdtrace && mpdboot ; mpdrun -np <numprocs> ./EratosthenesFortran 100
!    $ mpdtrace && mpdboot ; mpdrun -np <numprocs> ./EratosthenesFortran 1 10 100 1k 10k 100k 1M 10M 100M 1G
!
! Author:
!    NISHIMATSU Takeshi <t-nissie{at}imr.tohoku.ac.jp>
!
! Copying:
!    Eratosthenesfortran.f is distributed in the hope that they will be useful,
!    but WITHOUT ANY WARRANTY. You can copy, modify and redistribute it but only
!    under the conditions described in the GNU General Public License (the "GPL").
!
! References:
!    [MPICH2]              http://www-unix.mcs.anl.gov/mpi/mpich2/
!    [LAM/MPI]             http://www.lam-mpi.org/
!    [Open MPI]            http://www.open-mpi.org/
!    [MPI-forum]           http://www.mpi-forum.org/
!    [The Binary prefixes] http://physics.nist.gov/cuu/Units/binary.html
!!
#if defined HAVE_CONFIG_H
#  include "config.h"
#endif

#if defined(SR11000)
   function command_argument_count()
     implicit none
     integer command_argument_count
     open(9, file='SIZES')!, status='old', action='read')
     read(9,*) command_argument_count
   end function command_argument_count
   subroutine get_command_argument(i,a)
     implicit none
     integer i
     character (len=*) a
     read(9,*) a
   end subroutine get_command_argument
#elif defined(__PGI)
#  define command_argument_count iargc
#  define get_command_argument getarg
#endif

program EratosthenesFortran
  use i64_module
  use mpi_sp_module
  use sieve_module
  implicit none
  integer argc
  character (len=100) :: argument, argument_without_unit
  integer i, read_error, i_error, command_argument_count
  integer*8 memory_unit, memory_size, n, largest_prime_less_than_n, n_prime
  real*8 last_time

#ifdef MPI_PARALLEL
  integer result_length
  call MPI_INIT(i_error)
  call MPI_COMM_RANK(MPI_COMM_WORLD, mpi_sp_my_rank, i_error)
  call MPI_COMM_SIZE(MPI_COMM_WORLD, mpi_sp_n_processor, i_error)
  call MPI_GET_PROCESSOR_NAME(mpi_sp_my_hostname, result_length, i_error)
#endif

  argc = command_argument_count()
  if (argc.eq.0) then
     if (mpi_sp_my_rank.eq.0) then
        write(6,'(a)') PACKAGE_STRING
        write(6,'(2a,i2,a)') __FILE__,':',__LINE__,': Arguments are required'
        call flush(6)
     end if
     call MPI_Barrier(MPI_COMM_WORLD, i_error)
     call MPI_Abort(MPI_COMM_WORLD, __LINE__, i_error)
  end if

  if (mpi_sp_my_rank.eq.0) then
     write(6,'(a,a)') '#',PACKAGE_STRING
     write(6,'(a)') '#proc      byte/proc                 n           n_prime           largest     time[s]'
     call flush(6)
  end if

  last_time = MPI_Wtime()
  do i=1,argc
     call get_command_argument(i,argument)
     memory_unit = 1_i64   !Suppress compiler warning
     select case(argument(LEN_TRIM(argument):LEN_TRIM(argument)))
     case('i')
        argument_without_unit = argument(:LEN_TRIM(argument)-2)
        select case(argument(LEN_TRIM(argument)-1:LEN_TRIM(argument)-1))
        case('K')
           memory_unit =       1024_i64
        case('M')
           memory_unit =    1048576_i64
        case('G')
           memory_unit = 1073741824_i64
        case default
           if (mpi_sp_my_rank.eq.0) then
              write(6,'(2a,i2,a,a)')  __FILE__,':',__LINE__,&
                   ': Allowed binary prefixes are Ki, Mi and Gi. Illegal argument: ', TRIM(argument)
              call flush(6)
           end if
           call MPI_Barrier(MPI_COMM_WORLD, i_error)
           call MPI_Abort(MPI_COMM_WORLD, __LINE__, i_error)
        end select
     case('k')
        argument_without_unit = argument(:LEN_TRIM(argument)-1)
        memory_unit =       1000_i64
     case('M')
        argument_without_unit = argument(:LEN_TRIM(argument)-1)
        memory_unit =    1000000_i64
     case('G')
        argument_without_unit = argument(:LEN_TRIM(argument)-1)
        memory_unit = 1000000000_i64
     case default
        argument_without_unit = argument
        memory_unit = 1_i64
     end select
     read(argument_without_unit,*,iostat=read_error) memory_size
     if (read_error.ne.0) then
        if (mpi_sp_my_rank.eq.0) then
           write(6,'(2a,i2,a,a)')  __FILE__,':',__LINE__,': Illegal argument: ', TRIM(argument)
           call flush(6)
        end if
        call MPI_Barrier(MPI_COMM_WORLD, i_error)
        call MPI_Abort(MPI_COMM_WORLD, __LINE__, i_error)
     end if
     memory_size = memory_size * memory_unit
     call MPI_Barrier(MPI_COMM_WORLD, i_error)
     call eratosthenes(memory_size, n, largest_prime_less_than_n, n_prime)
     if (mpi_sp_my_rank.eq.(mpi_sp_n_processor-1)) then
        write(6,'(i5,i15,3i18,f12.3)') mpi_sp_n_processor, memory_size, n, n_prime, &
             largest_prime_less_than_n, MPI_Wtime()-last_time
        call flush(6)
        last_time = MPI_Wtime()
     end if
  end do

#ifdef MPI_PARALLEL
  call MPI_FINALIZE(i_error)
#endif
end program EratosthenesFortran
