/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.container;

import java.util.HashMap;
import java.util.Map;

import woolpack.utils.OGE;

/**
 * 簡易コンポーネントコンテナの定義。
 * 本クラスのインスタンスは複数のスレッドで同時に使用できる。
 * @author nakamura
 *
 */
abstract public class ComponentDef { 
	private final ComponentScope scope;
	private final Object singletonValue;
	
	/**
	 * コンストラクタ。
	 * @param scope スコープ。
	 * @throws NullPointerException 引数が null の場合。
	 * @throws RuntimeException スコープが{@link ComponentScope#SINGLETON}かつ{@link ComponentDef#newInstance()}が投げた場合。
	 */
	public ComponentDef(final ComponentScope scope){
		scope.getClass();
		this.scope = scope;
		if(scope == ComponentScope.SINGLETON){
			singletonValue = newInstance();
		}else{
			singletonValue = null;
		}
	}
	
	Object getComponent(
			final String attrName, 
			final String key, 
			final Map<String,Object> request, 
			final Map<String,Object> session, 
			final Map<String,Object> application){
		if(scope == ComponentScope.SINGLETON){
			return singletonValue;
		}
		if(scope == ComponentScope.REQUEST){
			return getComponent(attrName, key, request);
		}
		if(scope == ComponentScope.SESSION){
			return getComponent(attrName, key, session);
		}
		if(scope == ComponentScope.APPLICATION){
			return getComponent(attrName, key, application);
		}
		return newInstance();
	}
	
	private synchronized Object getComponent(final String attrName, final String key, final Map<String,Object> map){
		// TODO スコープに基づいた適切なスコープの同期制御を実装する。
		Map<String,Object> container = (Map<String,Object>)map.get(attrName);
		if(container == null){
			container = new HashMap<String,Object>();
			map.put(attrName, container);
		}
		
		Object result = container.get(key);
		if(result == null){
			result = newInstance();
			container.put(key, result);
		}
		return result;
	}
	
	/**
	 * オブジェクトを生成するために呼び出される(called)。
	 * 生成時に{@link OGE}を使用することにより、型検証のゆるいオブジェクト生成が可能。
	 * @return 生成したオブジェクト。
	 */
	abstract protected Object newInstance();
}
