/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.dom;

import java.util.HashMap;
import java.util.Map;

import woolpack.utils.OGE;

/**
 * {@link DomContext}と固定のオブジェクトの組に対して{@link OGE#getValue(Object)}を実行し、
 * 返却値により委譲先を分岐する{@link DomExpression}。
 * 
 * @author nakamura
 *
 */
public class Exec implements DomExpression {
	/**
	 * {@link DomContext}にアクセスするためのキー。
	 * OGNL "context" により{@link DomContext}にアクセスする。
	 */
	public static final String CONTEXT = "context";
	
	/**
	 * 固定のオブジェクトにアクセスするためのキー。
	 * OGNL "resource" により固定のオブジェクトにアクセスする。
	 */
	public static final String RESOURCE = "resource";
	
	private final Object resource;
	private final OGE oge;
	private final DomExpression trueExpression;
	private final DomExpression falseExpression;

	/**
	 * コンストラクタ。
	 * @param resource 固定のオブジェクト。
	 * @param trueExpression oge の実行結果が{@link Boolean#TRUE}の場合の委譲先。
	 * @param falseExpression oge の実行結果が{@link Boolean#FALSE}の場合の委譲先。
	 * @throws NullPointerException oge,trueExpression,falseExpressionのいずれかが null の場合。
	 */
	public Exec(
			final Object resource,
			final OGE oge, 
			final DomExpression trueExpression, 
			final DomExpression falseExpression){
		oge.getClass();
		trueExpression.getClass();
		falseExpression.getClass();
		
		this.resource = resource;
		this.oge = oge;
		this.trueExpression = trueExpression;
		this.falseExpression = falseExpression;
	}
	/**
	 * コンストラクタ。
	 * @param oge 実行する{@link OGE}。
	 * @param trueExpression oge の実行結果が{@link Boolean#TRUE}の場合の委譲先。
	 * @param falseExpression oge の実行結果が{@link Boolean#FALSE}の場合の委譲先。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public Exec(final OGE oge, final DomExpression trueExpression, final DomExpression falseExpression){
		this(null, oge, trueExpression, falseExpression);
	}

	/**
	 * コンストラクタ。
	 * oge の作用結果が{@link Boolean#FALSE}の場合は委譲しない。
	 * @param oge 実行する{@link OGE}。
	 * @param trueExpression oge の実行結果が{@link Boolean#TRUE}の場合の委譲先。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public Exec(final OGE oge, final DomExpression trueExpression){
		this(oge, trueExpression, DomConstants.NULL);
	}
	
	/**
	 * コンストラクタ。
	 * oge の作用結果にかかわらず委譲しない。
	 * @param oge 実行する{@link OGE}。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public Exec(final OGE oge){
		this(oge, DomConstants.NULL);
	}

	public void interpret(final DomContext context) {
		final Map<String,Object> map = new HashMap<String,Object>(2);
		map.put(RESOURCE, resource);
		map.put(CONTEXT, context);
		
		final Object value = oge.getValue(map);
		if(Boolean.TRUE.equals(value)){
			trueExpression.interpret(context);
		}else if(Boolean.FALSE.equals(value)){
			falseExpression.interpret(context);
		}
	}
}
