/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.utils;

import java.lang.reflect.Array;
import java.lang.reflect.Member;
import java.util.Collection;
import java.util.Map;

import ognl.TypeConverter;

/**
 * 一覧を表すオブジェクトを変換する OGNL 用タイプコンバータ。
 * @author nakamura
 *
 */
public class CollectionTypeConverter implements TypeConverter {
	private final TypeConverter defaultTypeConverter;
	
	/**
	 * コンストラクタ。
	 * @param defaultTypeConverter 単純型を変換するコンバータ。
	 * @throws NullPointerException 引数が null の場合。
	 */
	CollectionTypeConverter(final TypeConverter defaultTypeConverter){
		defaultTypeConverter.getClass();
		this.defaultTypeConverter = defaultTypeConverter;
	}
	
	/**
	 * OGNL の API から呼び出される。(called)。
	 * @param context OGNL のコンテキスト。
	 * @param target 格納先のオブジェクト。
	 * @param member 不明。
	 * @param propertyName 変換元のプロパティ名。
	 * @param beforeValue 変換前のオブジェクト。
	 * @param toType 変換先の型。
	 * @return 変換後のオブジェクト。
	 */
	public Object convertValue(final Map context, final Object target, final Member member, final String propertyName, final Object beforeValue, final Class toType) {
		final Collection collection = UtilsConstants.toCollection(beforeValue);
		if(toType.isArray()){
			// TODO 変換元が配列であっても新しい配列が作成されることの設計を再検討する。
			final Object afterArray = Array.newInstance(toType.getComponentType(), collection.size());
			int i=0;
			for(final Object before:collection){
				final Object after = defaultTypeConverter.convertValue(context, target, member, propertyName, before, toType.getComponentType());
				Array.set(afterArray, i, after);
				i++;
			}
			return afterArray;
		}else if(Collection.class.isAssignableFrom(toType)){
			return collection;
		}else{
			final Object value = collection.iterator().next();
			if(value != null){
				return defaultTypeConverter.convertValue(context, target, member, propertyName, value, toType);
			}else{
				return value;
			}
		}
	}
}
