/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.utils;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import ognl.MethodFailedException;
import ognl.Ognl;
import ognl.OgnlException;
import ognl.TypeConverter;

/**
 * OGNLをクラス化した、オブジェクトグラフ表現(Object Graph Expression)。
 * 本クラスのインスタンスは複数のスレッドで同時に使用できる。
 * @author nakamura
 *
 */
public class OGE {
	// NULL_OGEの前に定義する必要がある。
	private static final Map<String, Object> cache = new HashMap<String, Object>();
	
	/**
	 * nullを表す{@link OGE}。
	 */
	public static final OGE NULL = new OGE("null");
	
	private final String expression;
	private final Object tree;

	private static Object registerToMap(final String expression) throws OgnlException{
		Object myTree = cache.get(expression);
		if(myTree == null){
			myTree = Ognl.parseExpression(expression);
			cache.put(expression, myTree);
		}
		return myTree;
	}
	
	/**
	 * コンストラクタ。
	 * @param expression OGNLの文字列表現。
	 * @throws NullPointerException 引数が null の場合。
	 * @throws RuntimeException ({@link OgnlException})OGNLの解析に失敗した場合。
	 */
	public OGE(final String expression){
		// TODO TypeConverterを引数指定する。
		expression.getClass();
		this.expression = expression;
		try {
			this.tree = registerToMap(expression);
		} catch (final OgnlException exception) {
			throw new RuntimeException(exception);
		}
	}

	/**
	 * {@link Ognl#getValue(java.lang.Object, java.util.Map, java.lang.Object)}を実行する。
	 * その際に{@link CollectionTypeConverter}を作用させる。
	 * @param root 基点。
	 * @return 実行結果の値。
	 * @throws RuntimeException ({@link OgnlException})OGNLの解析に失敗した場合。
	 */
	public Object getValue(final Object root) {
		final Map context = Ognl.createDefaultContext(root);
		final TypeConverter defaultTypeConverter = Ognl.getTypeConverter(context);
		final TypeConverter addTypeConverter = new CollectionTypeConverter(defaultTypeConverter);
		Ognl.setTypeConverter(context, addTypeConverter);
		try {
			return Ognl.getValue(tree, context, root);
		} catch (final OgnlException exception) {
			throw new RuntimeException(exception);
		}finally{
			Ognl.setTypeConverter(context, defaultTypeConverter);
		}
	}

	/**
	 * {@link Ognl#getValue(java.lang.Object, java.util.Map, java.lang.Object, java.lang.Class)}を変換後クラスをStringとして実行する。
	 * その際に{@link CollectionTypeConverter}を作用させる。
	 * @param root 基点。
	 * @return 実行結果の値。
	 * @throws RuntimeException ({@link OgnlException})OGNLの解析に失敗した場合。
	 */
	public String getString(final Object root) {
		final Map context = Ognl.createDefaultContext(root);
		final TypeConverter defaultTypeConverter = Ognl.getTypeConverter(context);
		final TypeConverter addTypeConverter = new CollectionTypeConverter(defaultTypeConverter);
		Ognl.setTypeConverter(context, addTypeConverter);
		try {
			return (String)Ognl.getValue(tree, context, root, String.class);
		} catch (final OgnlException exception) {
			throw new RuntimeException(exception);
		}finally{
			Ognl.setTypeConverter(context, defaultTypeConverter);
		}
	}

	/**
	 * {@link Ognl#setValue(java.lang.Object, java.util.Map, java.lang.Object, java.lang.Object)}を実行する。
	 * その際に{@link CollectionTypeConverter}を作用させる。
	 * @param root 基点。
	 * @param value 設定する値。
	 * @throws RuntimeException ({@link OgnlException})OGNLの解析に失敗した場合。
	 */
	public void setValue(final Object root, final Object value) {
		final Map context = Ognl.createDefaultContext(root);
		final TypeConverter defaultTypeConverter = Ognl.getTypeConverter(context);
		final TypeConverter addTypeConverter = new CollectionTypeConverter(defaultTypeConverter);
		Ognl.setTypeConverter(context, addTypeConverter);
		try {
			Ognl.setValue(tree, context, root, value);
		} catch (final OgnlException exception) {
			throw new RuntimeException(exception);
		}finally{
			Ognl.setTypeConverter(context, defaultTypeConverter);
		}
	}

	/**
	 * {@link Ognl#setValue(java.lang.Object, java.util.Map, java.lang.Object, java.lang.Object)}を実行する。
	 * その際に{@link CollectionTypeConverter}を作用させる。
	 * @param root 基点。
	 * @param valueMap 設定する値の{@link Map}。
	 * @throws RuntimeException ({@link OgnlException})OGNLの解析に失敗した場合。
	 */
	public void setValues(final Object root, final Map valueMap) {
		final Map context = Ognl.createDefaultContext(root);
		final TypeConverter defaultTypeConverter = Ognl.getTypeConverter(context);
		final TypeConverter addTypeConverter = new CollectionTypeConverter(defaultTypeConverter);
		Ognl.setTypeConverter(context, addTypeConverter);
		try {
			final Object root2;
			try{
				root2 = Ognl.getValue(tree, context, root);
			} catch (final OgnlException exception) {
				throw new RuntimeException(exception);
			}
			for(final Iterator it = valueMap.keySet().iterator(); it.hasNext();){
				final String paramName = (String)it.next();
				final Object paramValue = valueMap.get(paramName);
				try{
					final Object paramTree = registerToMap(paramName);
					Ognl.setValue(paramTree, context, root2, paramValue);
				} catch (final OgnlException exception) {
					// TODO 無視以外の方法を考える
					continue;
				}
			}
		}finally{
			Ognl.setTypeConverter(context, defaultTypeConverter);
		}
	}

	/**
	 * {@link Ognl#getValue(java.lang.Object, java.util.Map, java.lang.Object)}を実行する。
	 * その際に{@link CollectionTypeConverter}を作用させる。
	 * OGNLのAPIからthrowされた{@link MethodFailedException}はそのまま呼出元に投げる。
	 * @param root 基点。
	 * @return 実行結果の値。
	 * @throws MethodFailedException メソッドが投げた例外をラップしたもの。
	 * @throws RuntimeException ({@link OgnlException})OGNLの解析に失敗した場合。
	 */
	public Object invoke(final Object root) throws MethodFailedException {
		final Map context = Ognl.createDefaultContext(root);
		final TypeConverter defaultTypeConverter = Ognl.getTypeConverter(context);
		final TypeConverter addTypeConverter = new CollectionTypeConverter(defaultTypeConverter);
		Ognl.setTypeConverter(context, addTypeConverter);
		try {
			return Ognl.getValue(tree, context, root);
		}catch(final MethodFailedException exception){
			if(exception.getReason() instanceof NoSuchMethodException){
				throw new RuntimeException(exception);
			}
			throw exception;
		} catch (final OgnlException exception) {
			throw new RuntimeException(exception);
		}finally{
			Ognl.setTypeConverter(context, defaultTypeConverter);
		}
	}
	
	@Override public int hashCode(){
		return expression.hashCode();
	}
	
	@Override public boolean equals(final Object o){
		if(o instanceof OGE){
			return expression.equals(((OGE)o).expression);
		}
		return false;
	}
	
	@Override public String toString(){
		return expression;
	}
}
