/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.dom;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import junit.framework.TestCase;
import woolpack.TestUtils;
import woolpack.utils.OGE;

public class BranchByAttrValueTest extends TestCase {

	public void testConstructor(){
		try{
			final Map<String,DomExpression> branch = new HashMap<String,DomExpression>();
			branch.put("id0", DomConstants.NULL);
			new BranchByAttrValue(null, branch, DomConstants.NULL);
			fail();
		}catch(final NullPointerException expected){
		}
		try{
			new BranchByAttrValue(Arrays.asList("action", "href"), null, DomConstants.NULL);
			fail();
		}catch(final NullPointerException expected){
		}
		try{
			final Map<String,DomExpression> branch = new HashMap<String,DomExpression>();
			branch.put("id0", DomConstants.NULL);
			new BranchByAttrValue(Arrays.asList("action", "href"), branch, null);
			fail();
		}catch(final NullPointerException expected){
		}
	}

	public void testNormal(){
		final Map<String,DomExpression> branch = new HashMap<String,DomExpression>();
		branch.put("page0.html", new UpdateAttrValue("href", new OGE("\"page0.do\"")));
		branch.put("page1.html", new UpdateAttrValue("href", new OGE("\"page1.do\"")));
		branch.put("page2.html", new UpdateAttrValue("href", new OGE("\"page2.do\"")));
		final DomExpression expression = new Serial(
				TestUtils.getToNodeBody(
						"<A href=\"page0.html\" >page0</A>"+
						"<A href=\"page1.html\" >page1</A>"+
						"<A href=\"page2.html\" >page2</A>"+
						"<A href=\"page3.html\" >page3</A>"),
				new XPath("//A", new BranchByAttrValue(
						Arrays.asList("href"), 
						branch, 
						new UpdateAttrValue("href", new OGE("\"pagee.do\""))))
		);
		final DomContext context = new DomContext();
		expression.interpret(context);
		
		assertTrue(TestUtils.equals(context, 
				"<A href=\"page0.do\" >page0</A>"+
				"<A href=\"page1.do\" >page1</A>"+
				"<A href=\"page2.do\" >page2</A>"+
				"<A href=\"pagee.do\" >page3</A>"));
	}

	public void testNotOther(){
		final Map<String,DomExpression> branch = new HashMap<String,DomExpression>();
		branch.put("page0.html", new UpdateAttrValue("href", new OGE("\"page0.do\"")));
		branch.put("page1.html", new UpdateAttrValue("href", new OGE("\"page1.do\"")));
		branch.put("page2.html", new UpdateAttrValue("href", new OGE("\"page2.do\"")));
		final DomExpression expression = new Serial(
				TestUtils.getToNodeBody(
						"<A href=\"page0.html\" >page0</A>"+
						"<A href=\"page1.html\" >page1</A>"+
						"<A href=\"page2.html\" >page2</A>"+
						"<A href=\"page3.html\" >page3</A>"),
				new XPath("//A", new BranchByAttrValue(
						Arrays.asList("href"), 
						branch))
		);
		final DomContext context = new DomContext();
		expression.interpret(context);

		assertTrue(TestUtils.equals(context, 
				"<A href=\"page0.do\" >page0</A>"+
				"<A href=\"page1.do\" >page1</A>"+
				"<A href=\"page2.do\" >page2</A>"+
				"<A href=\"page3.html\" >page3</A>"));
	}
}
