/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.samples.aa;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import woolpack.action.ActionConstants;
import woolpack.action.ActionDef;
import woolpack.action.ActionInvoker;
import woolpack.action.ForwardDef;
import woolpack.action.ForwardMatcher;
import woolpack.container.ComponentDef;
import woolpack.container.ComponentScope;
import woolpack.container.ScopeContainer;
import woolpack.dom.BranchById;
import woolpack.dom.ResetId;
import woolpack.dom.DelegateDomExpression;
import woolpack.dom.DomConstants;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.dom.Exec;
import woolpack.dom.FormatAttrValue;
import woolpack.dom.FormatId;
import woolpack.dom.Serial;
import woolpack.dom.XPath;
import woolpack.ee.ActionBuilder;
import woolpack.ee.HttpSessionMap;
import woolpack.ee.ServletContextMap;
import woolpack.ee.ServletInputStreamFactory;
import woolpack.ee.ServletRequestAttributeMap;
import woolpack.ee.ValidatorRuntimeException;
import woolpack.html.ToNode;
import woolpack.samples.ActionDefMaker;
import woolpack.text.FixFormat;
import woolpack.text.RegExpFormat;
import woolpack.utils.InputStreamReaderFactory;
import woolpack.utils.MapBuilder;
import woolpack.utils.OGE;
import woolpack.utils.UtilsConstants;

/**
 * 認証認可(Authentication and Authorization)を制御するサンプルウェブアプリケーション。
 * @author nakamura
 *
 */
public class AAServlet extends HttpServlet {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	public final Map<String,ComponentDef> componentDefMap;
	public final DelegateDomExpression toNode;
	public final DomExpression domExpression;
	
	public AAServlet(){
		componentDefMap = new HashMap<String,ComponentDef>();
		componentDefMap.put("aaComponent", new ComponentDef(ComponentScope.SESSION){
			@Override
			protected Object newInstance() {
				return new AAComponent();
			}
		});

		final ActionDefMaker maker = new ActionDefMaker();
		
		maker.putForward("simple_loginInput");
		// ログイン処理を呼び出す定義。ログイン失敗(false)の場合は遷移する画面を切り替える。
		{
			final String id = "common_top";
			maker.put(id, new ActionDef(
					new OGE("container.aaComponent"), 
					new OGE("container.aaComponent.login(),container.aaComponent.valid"), 
					new ForwardDef(id, new ForwardMatcher(Boolean.TRUE)), 
					new ForwardDef("simple_errorLogin")));
		}
		maker.putForward("evenRole_result");
		maker.putForward("oddRole_result");
		maker.putForward("simple_logout");
		
		final ActionBuilder actionBuilder = new ActionBuilder(
				new ActionInvoker(
						maker.getMap(), 
						new ForwardDef("simple_error", new OGE("local"), ActionConstants.ANY)), 
				Arrays.asList("name", "id"));
		
		toNode = new DelegateDomExpression();
		
		// id と role の対応を定義。
		final Map<String,String> idRoleMap = new MapBuilder<String,String>()
			.put("evenRole_result", "even")
			.put("oddRole_result", "odd")
			.getMap();
		
		{
			// id と role の対応に対する id 存在検証。定義(製造)のミスを発見するための仕掛け。
			final Collection<String> retain =  new HashSet<String>(idRoleMap.keySet());
			retain.removeAll(maker.getMap().keySet());
			if(!retain.isEmpty()){
				throw new RuntimeException("retain target id: " + retain);
			}
		}

		final DomExpression actionExpression = new Exec(
				idRoleMap,
				new OGE("resource[context.id] == null || context.container.aaComponent.roleCollection.contains(resource[context.id])"),
				// 認可結果が true の場合はアクションに処理をさせる。
				actionBuilder.getActionExpression(), 
				// 認可結果が false の場合は遷移する画面を切り替える。
				new FormatId(new FixFormat("simple_errorForbidden")));

		domExpression = new Serial(
				new FormatId(new RegExpFormat("^.*/([^/]+)$", "$1")),
				new BranchById(UtilsConstants.defaultValueMap(
						// ログイン入力画面・認証処理・ログアウト処理がリクエストされた場合はアクションに処理をさせる。
						new MapBuilder<String,DomExpression>()
						.put("simple_loginInput", actionExpression)
						.put("common_top")
						.put("simple_logout").getMap(),
						// その他の URI が指定された場合は認証済であることを検証し、認証済でない場合は遷移する画面を切り替える。
						new Exec(
								new OGE("context.container.aaComponent.valid"), 
								actionExpression, 
								new FormatId(new FixFormat("simple_errorForbidden"))))
				),
				toNode,
				new XPath("//A[@href]", new FormatAttrValue("href", new RegExpFormat("^([^\\.]+)\\.[^\\.]+$", "$1"))),
				new XPath("//FORM[@action]", new FormatAttrValue("action", new RegExpFormat("^([^\\.]+)\\.[^\\.]+$", "$1"))),
				actionBuilder.getAutoUpdateExpression(),
				new BranchById(
						// ログアウト・認証失敗・認可失敗・不正アクセスの場合はセッションスコープをクリアしログアウト状態とする。
						new MapBuilder<String,DomExpression>()
						.put("simple_logout", DomConstants.CLEAR_SESSION)
						.put("simple_errorLogin")
						.put("simple_errorForbidden").getMap())
		);
	}


	@Override public void init(final ServletConfig servletConfig) throws ServletException {
		super.init(servletConfig);
		toNode.setExpression(new ResetId(new Serial(
				new FormatId(new RegExpFormat("^(.*)$", "/html/sample/aa/$1.html")),
				new ToNode(new InputStreamReaderFactory(new ServletInputStreamFactory(servletConfig.getServletContext()), "UTF-8")))));
	}

	@Override protected void service(final HttpServletRequest request, final HttpServletResponse response) throws ServletException, IOException{
		final DomContext domContext = new DomContext();
		domContext.setId(request.getRequestURI());
		domContext.setInput(request.getParameterMap());
		domContext.setRequest(new ServletRequestAttributeMap(request));
		domContext.setSession(new HttpSessionMap(request.getSession()));
		domContext.setApplication(new ServletContextMap(request.getSession().getServletContext()));
		domContext.setContainer(new ScopeContainer(domContext.getRequest(), domContext.getSession(), domContext.getApplication(), componentDefMap));
		try{
			domExpression.interpret(domContext);
		}catch(final ValidatorRuntimeException e){
			final StringBuilder sb = new StringBuilder();
			e.getContext().appendTo(sb);
			throw new RuntimeException(sb.toString(), e);
		}catch(final RuntimeException e){
			final StringBuilder sb = new StringBuilder();
			domContext.appendTo(sb);
			throw new RuntimeException(sb.toString(), e);
		}
		final Writer w = new BufferedWriter(new OutputStreamWriter(response.getOutputStream(), "UTF-8"));
		try{
			DomConstants.write(domContext.getNode(), w);
		}finally{
			w.close();
		}
	}
}
