/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.action;

import java.util.Arrays;

import woolpack.el.ELConstants;
import woolpack.el.GettingEL;



/**
 * ひとつのアクション定義を表す。
 * @author nakamura
 *
 */
public class ActionDef {
	
	private final GettingEL componentEL;
	private final GettingEL methodEL;
	private final Iterable<ForwardDef> forwardDefIterable;
	
	/**
	 * コンストラクタ。
	 * @param componentEL コンポーネントへの参照。コンテキスト役に対して副作用が発生すべきではない。
	 * @param methodEL 実行するメソッドへの参照。コンテキスト役に対して副作用が発生してもよい。
	 * @param forwardDefIterable 遷移先定義の一覧。本クラスはこの引数の状態を変化させない。ステートレスであるべき。
	 * @throws 引数のいずれかが null の場合。
	 */
	public ActionDef(final GettingEL componentEL, final GettingEL methodEL, final Iterable<ForwardDef> forwardDefIterable){
		componentEL.getClass();
		methodEL.getClass();
		forwardDefIterable.getClass();
		
		this.componentEL = componentEL;
		this.methodEL = methodEL;
		this.forwardDefIterable = forwardDefIterable;
	}

	/**
	 * コンストラクタ。
	 * コンポーネントの取得先を{@link ELConstants#NULL}とする。
	 * @param methodEL 実行するメソッドへの参照。コンテキスト役に対して副作用が発生してもよい。
	 * @param forwardDefIterable 遷移先定義の一覧。本クラスはこの引数の状態を変化させない。ステートレスであるべき。
	 * @throws 引数のいずれかが null の場合。
	 */
	public ActionDef(final GettingEL methodEL, final Iterable<ForwardDef> forwardDefIterable){
		this(ELConstants.NULL, methodEL, forwardDefIterable);
	}
	
	/**
	 * コンストラクタ。
	 * @param componentEL コンポーネントへの参照。コンテキスト役に対して副作用が発生すべきではない。
	 * @param methodEL 実行するメソッドへの参照。コンテキスト役に対して副作用が発生してもよい。
	 * @param forwardDefArray 遷移先定義の一覧。本クラスはこの引数の状態を変化させない。ステートレスであるべき。
	 * @throws 引数のいずれかが null の場合。
	 */
	public ActionDef(final GettingEL componentEL, final GettingEL methodEL, final ForwardDef... forwardDefArray){
		this(componentEL, methodEL, Arrays.asList(forwardDefArray));
	}

	/**
	 * コンストラクタ。
	 * コンポーネントの取得先を{@link ELConstants#NULL}とする。
	 * @param methodEL 実行するメソッドへの参照。コンテキスト役に対して副作用が発生してもよい。
	 * @param forwardDefArray 遷移先定義の一覧。本クラスはこの引数の状態を変化させない。ステートレスであるべき。
	 * @throws 引数のいずれかが null の場合。
	 */
	public ActionDef(final GettingEL methodEL, final ForwardDef... forwardDefArray){
		this(ELConstants.NULL, methodEL, forwardDefArray);
	}
	
	/**
	 * コンポーネントへの参照を返す。
	 * @return コンポーネントへの参照。
	 */
	public GettingEL getComponentEL() {
		return componentEL;
	}

	/**
	 * メソッドへの参照を返す。
	 * @return メソッドの位置。
	 */
	public GettingEL getMethodEL() {
		return methodEL;
	}

	/**
	 * 遷移先定義の一覧を返す。
	 * @return 遷移先定義の一覧。
	 */
	public Iterable<ForwardDef> getForwardDefIterable() {
		return forwardDefIterable;
	}
}
