/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.el.convert;

import java.text.DateFormat;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;

import woolpack.text.BooleanFormat;
import woolpack.validator.ValidatorExpression;

/**
 * 定数の集まり。
 * @author nakamura
 *
 */
public class ConvertConstants {
	private static final NumberFormat NUMBER_FORMAT = new DecimalFormat();
	private static final DateFormat DATE_FORMAT = new SimpleDateFormat("yyyyMMdd");
	private static final BooleanFormat BOOLEAN_FORMAT = new BooleanFormat();

	/**
	 * 変換前の値をそのまま返す{@link Convertable}。
	 */
	public static final Convertable ECHO = new Convertable(){
		public Object convert(final Object beforeValue, final Class toType, final String propertyName) {
			return beforeValue;
		}
	};
	
	/**
	 * 単純型変換器のデフォルト値。
	 */
	public static final Convertable SIMPLE_CONVERTER = getSimpleConverter(
			NUMBER_FORMAT, DATE_FORMAT, BOOLEAN_FORMAT);
	
	/**
	 * 一覧型変換器のデフォルト値。
	 */
	public static final Convertable COLLECTION_CONVERTER = getCollectionConverter(
			NUMBER_FORMAT, DATE_FORMAT, BOOLEAN_FORMAT);

	private ConvertConstants(){}// カバレージがここを通過してはいけない
	
	/**
	 * 単純型変換器を返す。
	 * @param numberFormat 数値フォーマット。
	 * @param dateFormat 日付フォーマット。
	 * @param booleanFormat 真偽値フォーマット。
	 * @return 単純型変換器。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public static Convertable getSimpleConverter(
			final NumberFormat numberFormat, 
			final DateFormat dateFormat, 
			final BooleanFormat booleanFormat){
		return new DelegationIfNecessityConverter(new ToTypeToWapperConverter(new SimpleConverter(
				numberFormat, dateFormat, booleanFormat
		)));
	}
	
	/**
	 * 単純型変換器を返す。
	 * @param expression プロパティ名による変換処理の委譲先。
	 * @param numberFormat 数値フォーマット。
	 * @param dateFormat 日付フォーマット。
	 * @param booleanFormat 真偽値フォーマット。
	 * @return 単純型変換器。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public static Convertable getSimpleConverter(
			final ValidatorExpression expression, 
			final NumberFormat numberFormat, 
			final DateFormat dateFormat, 
			final BooleanFormat booleanFormat){
		return new DelegationIfNecessityConverter(
				new SerialConverter(
						new ValidatorConverter(expression),
						new ToTypeToWapperConverter(
								new SimpleConverter(
										numberFormat, 
										dateFormat, 
										booleanFormat
								)
						)
				)
		);
	}
	
	/**
	 * 一覧型変換器を返す。
	 * @param numberFormat 数値フォーマット。
	 * @param dateFormat 日付フォーマット。
	 * @param booleanFormat 真偽値フォーマット。
	 * @return 一覧型変換器。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public static Convertable getCollectionConverter(
			final NumberFormat numberFormat, 
			final DateFormat dateFormat, 
			final BooleanFormat booleanFormat){
		return new DelegationIfNecessityConverter(new CollectionConverter(
				getSimpleConverter(numberFormat, dateFormat, booleanFormat)
		));
	}
	
	/**
	 * 一覧型変換器を返す。
	 * @param expression プロパティ名による変換処理の委譲先。
	 * @param numberFormat 数値フォーマット。
	 * @param dateFormat 日付フォーマット。
	 * @param booleanFormat 真偽値フォーマット。
	 * @return 一覧型変換器。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public static Convertable getCollectionConverter(
			final ValidatorExpression expression, 
			final NumberFormat numberFormat, 
			final DateFormat dateFormat, 
			final BooleanFormat booleanFormat){
		return new DelegationIfNecessityConverter(new CollectionConverter(
				getSimpleConverter(expression, numberFormat, dateFormat, booleanFormat)
		));
	}
}
