/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.html;

import java.util.Collection;
import java.util.Set;

import org.w3c.dom.Element;
import org.w3c.dom.Node;

import woolpack.dom.DomConstants;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.dom.Loop;
import woolpack.dom.NodeLoop;
import woolpack.el.ArrayPathEL;
import woolpack.el.EL;
import woolpack.el.ELConstants;
import woolpack.el.GettingEL;
import woolpack.el.MapEL;
import woolpack.el.PropertyEL;
import woolpack.utils.NodeFindable;
import woolpack.utils.NodeFindableFactory;
import woolpack.utils.UtilsConstants;


/**
 * DOM エレメントの属性値をプロパティ名として、コンポーネントから値を取得し、 DOM ノードに自動設定する{@link DomExpression}。
 * @author nakamura
 *
 */
public class AutoUpdate2 extends NodeLoop {
	private static final EL COMPONENT_EL = new ArrayPathEL(DomConstants.LOCAL_EL, new MapEL("autoUpdateComponent"));
	
	private final Iterable<String> attrNames;
	private final GettingEL componentEL;
	private final boolean selectFlag;
	private final Set<Class> atomSet;
	private final GettingEL errorEL;
	private final NodeFindableFactory factory;
	
	AutoUpdate2(
			final Iterable<String> attrNames, 
			final GettingEL componentEL, 
			final boolean selectMode,
			final Set<Class> atomSet,
			final GettingEL errorEL,
			final NodeFindableFactory factory){
		attrNames.getClass();
		componentEL.getClass();
		atomSet.getClass();
		errorEL.getClass();
		factory.getClass();
		
		this.attrNames = attrNames;
		this.componentEL = componentEL;
		this.selectFlag = selectMode;
		this.atomSet = atomSet;
		this.errorEL = errorEL;
		this.factory = factory;
	}
	
	/**
	 * コンストラクタ。
	 * @param attrNames 属性名の一覧。本クラスはこの引数の状態を変化させない。
	 * @param componentEL コンポーネントへの参照。コンテキスト役に対して副作用が発生すべきではない。
	 * @param atomSet 値の個数に関して原子的であるクラスの一覧。
	 * @param errorEL {@link DomContext}からの値取得に失敗した場合の値の取得先。
	 * @param factory {@link NodeFindable}のファクトリ。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public AutoUpdate2(
			final Iterable<String> attrNames, 
			final GettingEL componentEL,
			final Set<Class> atomSet,
			final GettingEL errorEL,
			final NodeFindableFactory factory){
		this(attrNames, componentEL, true, atomSet, errorEL, factory);
	}
	
	/**
	 * コンストラクタ。DomContextからの値取得に失敗した場合は何もしない。
	 * @param attrNames 属性名の一覧。本クラスはこの引数の状態を変化させない。
	 * @param componentEL コンポーネントへの参照。コンテキスト役に対して副作用が発生すべきではない。
	 * @param factory {@link NodeFindable}のファクトリ。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public AutoUpdate2(
			final Iterable<String> attrNames, 
			final GettingEL componentEL,
			final NodeFindableFactory factory){
		this(attrNames, componentEL, UtilsConstants.ATOM_SET, ELConstants.NULL, factory);
	}

	/**
	 * @throws NullPointerException 引数または{@link DomContext#getNode()}が null の場合。
	 * @throws ClassCastException {@link DomContext#getNode()}が{@link Element}でない場合。
	 */
	@Override public void interpret(final DomContext context) {
		if(context.getNode().getNodeType() == Node.ELEMENT_NODE){
			final Element e = (Element)context.getNode();
			final String attrName = getAttrName(e);
			if(attrName != null){
				final String attrValue = e.getAttribute(attrName);
				final GettingEL valueEL = new ArrayPathEL(componentEL, new PropertyEL(attrValue));
				final Object value = getValue(context, valueEL);
				if(value != null){
					if(isBeanCollection(value, atomSet)){
						new Loop(valueEL, COMPONENT_EL, 
								new AutoUpdate2(attrNames, COMPONENT_EL, false, atomSet, errorEL, factory)).interpret(context);
					}else{
						new UpdateValue2(
								valueEL, 
								new ArrayPathEL(DomConstants.CONFIG_EL, new MapEL(attrValue)), 
								selectFlag, factory).interpret(context);
					}
					return;
				}
			}
		}
		super.interpret(context);
	}
	
	private Object getValue(final DomContext context, final GettingEL valueEL){
		try{
			return valueEL.getValue(context);
		}catch(final RuntimeException exception){
			return errorEL.getValue(context);
		}
	}
	
	private String getAttrName(final Element e){
		for(final String attrName:attrNames){
			if(e.hasAttribute(attrName)){
				return attrName;
			}
		}
		return null;
	}
	
	private static boolean isBeanCollection(final Object value, final Set<Class> atomSet){
		if(value instanceof Collection){
			final Collection c = (Collection)value;
			if(c.size() == 0){
				return true;
			}
			return !atomSet.contains(c.iterator().next().getClass());
		}
		if(value.getClass().isArray()){
			return !atomSet.contains(value.getClass().getComponentType());
		}
		return false;
	}
}
