/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.validator;

import java.util.Arrays;

/**
 * 委譲先の一覧を順次呼び出す{@link ValidatorExpression}。
 * {@link ValidatorContext}に対して状態を変更する操作を行わない。
 * @author nakamura
 *
 */
public class SerialValidator implements ValidatorExpression {
	private final ValidatorIterable validatorIterable;
	private final Iterable<? extends ValidatorExpression> iterable;
	
	/**
	 * コンストラクタ。
	 * @param validatorIterable 値検証一覧の呼び出し方。
	 * @param iterable 委譲先の一覧。本クラスはこの引数の状態を変化させない。ステートレスであるべき。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public SerialValidator(final ValidatorIterable validatorIterable, final Iterable<? extends ValidatorExpression> iterable){
		validatorIterable.getClass();
		iterable.getClass();
		
		this.validatorIterable = validatorIterable;
		this.iterable = iterable;
	}
	
	/**
	 * コンストラクタ。
	 * 値検証一覧の呼び出し方に{@link ValidatorConstants#ANDAND}を使用する。
	 * @param iterable 委譲先の一覧。本クラスはこの引数の状態を変化させない。ステートレスであるべき。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public SerialValidator(final Iterable<? extends ValidatorExpression> iterable){
		this(ValidatorConstants.ANDAND, iterable);
	}
	
	/**
	 * コンストラクタ。
	 * @param validatorIterable 値検証一覧の呼び出し方。
	 * @param expressions 委譲先の一覧。本クラスはこの引数の状態を変化させない。ステートレスであるべき。
	 * @throws NullPointerException validatorIterable が null の場合。
	 */
	public SerialValidator(final ValidatorIterable validatorIterable, final ValidatorExpression... expressions){
		this(validatorIterable, Arrays.asList(expressions));
	}
	
	/**
	 * コンストラクタ。
	 * 値検証一覧の呼び出し方に{@link ValidatorConstants#ANDAND}を使用する。
	 * @param expressions 委譲先の一覧。本クラスはこの引数の状態を変化させない。ステートレスであるべき。
	 */
	public SerialValidator(final ValidatorExpression... expressions){
		this(ValidatorConstants.ANDAND, Arrays.asList(expressions));
	}

	public boolean interpret(ValidatorContext context){
		return validatorIterable.interpret(context, iterable);
	}

	public Iterable<? extends ValidatorExpression> getIterable() {
		return iterable;
	}

	public ValidatorIterable getValidatorIterable() {
		return validatorIterable;
	}
}
