/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.visitor.impl;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;

import woolpack.text.FixFormat;
import woolpack.text.LimitedValueFormat;
import woolpack.text.RegExpFormat;
import woolpack.text.SwitchableFormat;
import woolpack.text.ToBigDecimalFormat;
import woolpack.text.ToBigIntegerFormat;
import woolpack.text.ToByteFormat;
import woolpack.text.ToCharacterFormat;
import woolpack.text.ToDoubleFormat;
import woolpack.text.ToFloatFormat;
import woolpack.text.ToIntegerFormat;
import woolpack.text.ToLongFormat;
import woolpack.text.ToSQLDateFormat;
import woolpack.text.ToShortFormat;
import woolpack.text.ToStringFormat;
import woolpack.text.TrysDateFormat;
import woolpack.utils.MapBuilder;
import woolpack.visitor.Acceptable;
import woolpack.visitor.Visitor;
import woolpack.visitor.VisitorConstants;

/**
 * コンストラクタツリーを日本語 HTML ドキュメントに変換するための定数の集まり。
 * @author nakamura
 *
 */
public class DocTextConstants {
	
	/**
	 * {@link wookpack.text}パッケージで定義されているクラスに関する
	 * 委譲先{@link Acceptable}を分岐する定義。
	 */
	public static final Map<Object,Acceptable> BASE = Collections.unmodifiableMap(MapBuilder.get(new LinkedHashMap<Object,Acceptable>())
			.put(ToCharacterFormat.class, VisitorConstants.visitAcceptable("文字に変換するフォーマット"))
			.put(ToStringFormat.class, VisitorConstants.visitAcceptable("文字列に変換するフォーマット"))
			.put(FixFormat.class, new Acceptable<FixFormat>(){
				public void accept(final Visitor visitor, final FixFormat v) {
					visitor.visit(v.getValue());
					visitor.visit("に変換するフォーマット");
				}
			})
			.put(LimitedValueFormat.class, new Acceptable<LimitedValueFormat>(){
				public void accept(final Visitor visitor, final LimitedValueFormat v) {
					visitor.visit("次の定義で変換するフォーマット");
					visitor.visit("、フォーマットとパースの対応：");
					visitor.visit(v.getParseMap());
					visitor.visit("、デフォルトのフォーマット値：");
					visitor.visit(v.getDefaultFormattedValue());
					visitor.visit("、デフォルトのパース値：");
					visitor.visit(v.getDefaultParsedValue());
				}
			})
			.put(RegExpFormat.class, new Acceptable<RegExpFormat>(){
				public void accept(final Visitor visitor, final RegExpFormat v) {
					visitor.visit("正規表現\"");
					visitor.visit(v.getPattern());
					visitor.visit("\"にマッチした文字列を\"");
					visitor.visit(v.getReplacement());
					visitor.visit("\"のパターンに変換するフォーマット");
				}
			})
			.put(SwitchableFormat.class, new Acceptable<SwitchableFormat>(){
				public void accept(final Visitor visitor, final SwitchableFormat v) {
					visitor.visit("次の対応で変換するフォーマット");
					visitor.visit(v.getSwitchable());
				}
			})
			.put(ToBigDecimalFormat.class, new Acceptable<ToBigDecimalFormat>(){
				public void accept(final Visitor visitor, final ToBigDecimalFormat v) {
					visitor.visit(v.getFormat());
					visitor.visit("による変換結果を BigDecimal に変換するフォーマット");
				}
			})
			.put(ToBigIntegerFormat.class, new Acceptable<ToBigIntegerFormat>(){
				public void accept(final Visitor visitor, final ToBigIntegerFormat v) {
					visitor.visit(v.getFormat());
					visitor.visit("による変換結果を BigInteger に変換するフォーマット");
				}
			})
			.put(ToByteFormat.class, new Acceptable<ToByteFormat>(){
				public void accept(final Visitor visitor, final ToByteFormat v) {
					visitor.visit(v.getFormat());
					visitor.visit("による変換結果を Byte に変換するフォーマット");
				}
			})
			.put(ToDoubleFormat.class, new Acceptable<ToDoubleFormat>(){
				public void accept(final Visitor visitor, final ToDoubleFormat v) {
					visitor.visit(v.getFormat());
					visitor.visit("による変換結果を Double に変換するフォーマット");
				}
			})
			.put(ToFloatFormat.class, new Acceptable<ToFloatFormat>(){
				public void accept(final Visitor visitor, final ToFloatFormat v) {
					visitor.visit(v.getFormat());
					visitor.visit("による変換結果を Float に変換するフォーマット");
				}
			})
			.put(ToIntegerFormat.class, new Acceptable<ToIntegerFormat>(){
				public void accept(final Visitor visitor, final ToIntegerFormat v) {
					visitor.visit(v.getFormat());
					visitor.visit("による変換結果を Integer に変換するフォーマット");
				}
			})
			.put(ToLongFormat.class, new Acceptable<ToLongFormat>(){
				public void accept(final Visitor visitor, final ToLongFormat v) {
					visitor.visit(v.getFormat());
					visitor.visit("による変換結果を Long に変換するフォーマット");
				}
			})
			.put(ToShortFormat.class, new Acceptable<ToShortFormat>(){
				public void accept(final Visitor visitor, final ToShortFormat v) {
					visitor.visit(v.getFormat());
					visitor.visit("による変換結果を Short に変換するフォーマット");
				}
			})
			.put(ToSQLDateFormat.class, new Acceptable<ToSQLDateFormat>(){
				public void accept(final Visitor visitor, final ToSQLDateFormat v) {
					visitor.visit(v.getFormat());
					visitor.visit("による変換結果を SQL 用の日付型(java.sql.Date)に変換するフォーマット");
				}
			})
			.put(TrysDateFormat.class, new Acceptable<TrysDateFormat>(){
				public void accept(final Visitor visitor, final TrysDateFormat v) {
					visitor.visit("次の一覧を順にパースし、最初に成功した変換値を適用するフォーマット, ");
					visitor.visit(v.getIterable());
				}
			})
			.get());
	
	private DocTextConstants(){}// カバレージがここを通過してはいけない
}
