/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.utils;

import java.util.HashMap;
import java.util.Map;

import junit.framework.TestCase;

public class EmbeddingResolverTest extends TestCase {
	
	public void testResolveValue(){
		final Map<String,Object> map = new HashMap<String,Object>();
		map.put("baseKey0", "baseValue0");
		map.put("baseKey1", "baseValue1");
		map.put("abstractKey0", "a{0}b{1}c");
		
		final EmbeddingResolver expression = new EmbeddingResolver();
		assertEquals("abaseValue0bbaseValue1c", expression.resolve(map, "#{abstractKey0}{baseKey0}{baseKey1}"));
		assertEquals("xbaseValue0ybaseValue1z", expression.resolve(map, "x{baseKey0}y{baseKey1}z"));
		assertEquals("baseValue0", expression.resolve(map, "baseKey0"));
		
		final Map<String,Object> expected = new HashMap<String,Object>();
		expected.put("baseKey0", "baseValue0");
		expected.put("baseKey1", "baseValue1");
		expected.put("abstractKey0", "a{0}b{1}c");
		assertEquals(expected, map);
	}
	
	public void testResolve(){
		final Map<String,Object> map = new HashMap<String,Object>();
		map.put("baseKey0", "baseValue0");
		map.put("baseKey1", "baseValue1");
		map.put("complexKey0", "x{baseKey0}y{baseKey1}z");
		map.put("abstractKey0", "a{0}b{1}c");
		map.put("complexKey1", "#{abstractKey0}{baseKey0}{baseKey1}");
		
		final EmbeddingResolver expression = new EmbeddingResolver();
		expression.resolve(map);
		
		final Map<String,Object> expected = new HashMap<String,Object>();
		expected.put("baseKey0", "baseValue0");
		expected.put("baseKey1", "baseValue1");
		expected.put("complexKey0", "xbaseValue0ybaseValue1z");
		expected.put("abstractKey0", "a{0}b{1}c");
		expected.put("complexKey1", "abaseValue0bbaseValue1c");
		assertEquals(expected, map);
	}
	
	public void testResolveNotFound(){
		final Map<String,Object> map = new HashMap<String,Object>();
		map.put("baseKey0", "baseValue0");
		map.put("abstractKey0", "a{0}b{1}c");
		map.put("complexKey1", "#{abstractKey0}{baseKey0}{baseKey1}");
		map.put("complexKey2", "#{abstractKey1}{baseKey0}");
		map.put("complexKey3", "#{hoge");
		
		final EmbeddingResolver expression = new EmbeddingResolver();
		expression.resolve(map);
		
		final Map<String,Object> expected = new HashMap<String,Object>();
		expected.put("baseKey0", "baseValue0");
		expected.put("abstractKey0", "a{0}b{1}c");
		expected.put("complexKey1", "#{abstractKey0}{baseKey0}{baseKey1}");
		expected.put("complexKey2", "#{abstractKey1}{baseKey0}");
		expected.put("complexKey3", "#{hoge");
		assertEquals(expected, map);
	}
}
