/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.utils.concurrent;

import java.util.concurrent.Semaphore;

import junit.framework.TestCase;

public class AcquirableChainImplTest extends TestCase {

	public void testConstructor(){
		try{
			new AcquirableChain((Acquirable[])null);
			fail();
		}catch(final NullPointerException e){
		}
	}
	
	public void testNormal() throws InterruptedException{
		final Semaphore semaphore0 = new Semaphore(1);
		final Semaphore semaphore1 = new Semaphore(1);
		final Acquirable acquirable = new AcquirableChain(
				new TrySemaphore(semaphore0),
				new TrySemaphore(semaphore1)
				);
		assertTrue(acquirable.acquire());
		assertEquals(0, semaphore0.availablePermits());
		assertEquals(0, semaphore1.availablePermits());

		acquirable.release();
		assertEquals(1, semaphore0.availablePermits());
		assertEquals(1, semaphore1.availablePermits());
		
		semaphore1.acquire();

		assertFalse(acquirable.acquire());
		assertEquals(1, semaphore0.availablePermits());
		assertEquals(0, semaphore1.availablePermits());
		
		semaphore1.release();
		semaphore0.acquire();

		assertFalse(acquirable.acquire());
		assertEquals(0, semaphore0.availablePermits());
		assertEquals(1, semaphore1.availablePermits());
	}
	
	public void testException() throws InterruptedException{
		final Semaphore semaphore0 = new Semaphore(1);
		final Acquirable acquirable = new AcquirableChain(
				new TrySemaphore(semaphore0),
				AcquirableConstants.INTERRUPTED_EXCEPTION
				);
		try{
			acquirable.acquire();
			fail();
		}catch(final RuntimeException e){
			assertTrue(e.getCause() instanceof InterruptedException);
			assertTrue(Thread.currentThread().isInterrupted());
		}
		
		assertEquals(1, semaphore0.availablePermits());
		
		Thread.interrupted();
		semaphore0.acquire();
		
		assertFalse(acquirable.acquire());
	}
}
