/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.samples.aa;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import woolpack.action.ActionConstants;
import woolpack.action.ActionDef;
import woolpack.action.ActionInvoker;
import woolpack.action.ForwardDef;
import woolpack.action.ForwardMatcher;
import woolpack.adapter.JXP;
import woolpack.adapter.JXPFactory;
import woolpack.container.AbstractComponentDef;
import woolpack.container.ComponentScope;
import woolpack.container.ScopeContainer;
import woolpack.dom.Branch;
import woolpack.dom.DelegateDomExpression;
import woolpack.dom.DomConstants;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.dom.DumpIfCatch;
import woolpack.dom.EvalEL;
import woolpack.dom.FormatAttrValue;
import woolpack.dom.FormatId;
import woolpack.dom.If;
import woolpack.dom.ResetId;
import woolpack.dom.Serial;
import woolpack.dom.XPath;
import woolpack.dom.XmlToNode;
import woolpack.ee.ActionBuilder;
import woolpack.ee.HttpSessionMap;
import woolpack.ee.ServletContextMap;
import woolpack.ee.ServletInputStreamFactory;
import woolpack.ee.ServletRequestAttributeMap;
import woolpack.el.AbstractToELTargetExceptionEL;
import woolpack.el.PathEL;
import woolpack.html.HtmlConstants;
import woolpack.samples.ActionDefMaker;
import woolpack.text.FixFormat;
import woolpack.text.RegExpFormat;
import woolpack.utils.InputStreamReaderFactory;
import woolpack.utils.MapBuilder;
import woolpack.utils.SwitchBuilder;
import woolpack.utils.Switchable;
import woolpack.utils.UtilsConstants;

/**
 * 認証認可(Authentication and Authorization)を制御するサンプルウェブアプリケーション。
 * @author nakamura
 *
 */
public class AAServlet extends HttpServlet {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	public final transient Map<String, AbstractComponentDef> componentDefMap;
	public final transient DelegateDomExpression toNode;
	public final transient DomExpression domExpression;
	
	public AAServlet() {
		super();
		componentDefMap = new HashMap<String, AbstractComponentDef>();
		componentDefMap.put("aaComponent", new AbstractComponentDef(ComponentScope.SESSION) {
			@Override
			protected Object newInstance() {
				return new AAComponent();
			}
		});

		final Switchable<String, ActionDef> actionDefSwitchable;
		{
			final ActionDefMaker maker = new ActionDefMaker();
			
			maker.putForward("simple_loginInput");
			// ログイン処理を呼び出す定義。ログイン失敗(false)の場合は遷移する画面を切り替える。
			{
				final String id = "common_top";
				maker.put(id, new ActionDef(
						new PathEL("container.aaComponent"),
						new AbstractToELTargetExceptionEL() {
							@Override public Object execute(final Object root, final Object value) {
								// new OGE("container.aaComponent.login(),container.aaComponent.valid")
								final AAComponent component = (AAComponent) ((Map) ((DomContext) root).getContainer()).get("aaComponent");
								component.login();
								return component.isValid();
							}
						},
						new ForwardDef(id, new ForwardMatcher(Boolean.TRUE)),
						new ForwardDef("simple_errorLogin")));
			}
			maker.putForward("evenRole_result");
			maker.putForward("oddRole_result");
			maker.putForward("simple_logout");
			actionDefSwitchable = maker.get();
		}
		
		final ActionBuilder actionBuilder = new ActionBuilder(
				new ActionInvoker(
						actionDefSwitchable,
						new ForwardDef("simple_error", DomConstants.LOCAL_EL, ActionConstants.ANY)),
				Arrays.asList("name", "id"), new JXPFactory());
		
		toNode = new DelegateDomExpression();
		
		// id と role の対応を定義。
		final Map<String, String> idRoleMap = MapBuilder.get(new HashMap<String, String>())
			.put("evenRole_result", "even")
			.put("oddRole_result", "odd")
			.get();
		
		{
			// id と role の対応に対する id 存在検証。定義(製造)のミスを発見するための仕掛け。
			final Collection<String> retain =  new HashSet<String>(idRoleMap.keySet());
			retain.removeAll(actionDefSwitchable.keys());
			if (!retain.isEmpty()) {
				throw new IllegalStateException("retain target id: " + retain);
			}
		}

		final DomExpression actionExpression = new If(
				new EvalEL(idRoleMap, new AbstractToELTargetExceptionEL() {
					@Override public Object execute(final Object root, final Object value) {
						// new OGE("resource[context.id] == null || context.container.aaComponent.roleCollection.contains(resource[context.id])")
						final Map rootMap = (Map) root;
						final DomContext context = (DomContext) rootMap.get("context");
						final Object role = ((Map) rootMap.get("resource")).get(context.getId());
						return role == null || ((AAComponent) ((Map) context.getContainer()).get("aaComponent")).getRoleCollection().contains(role);
					}
				}),
				// 認可結果が true の場合はアクションに処理をさせる。
				actionBuilder.getActionExpression(),
				// 認可結果が false の場合は遷移する画面を切り替える。
				new FormatId(new FixFormat("simple_errorForbidden")));

		domExpression = new DumpIfCatch(new Serial(
				new FormatId(new RegExpFormat("^.*/([^/]+)$", "$1")),
				new Branch<String>(DomConstants.EVAL_ID,
						// ログイン入力画面・認証処理・ログアウト処理がリクエストされた場合はアクションに処理をさせる。
						new SwitchBuilder<String, DomExpression>()
						.put("simple_loginInput", actionExpression)
						.put("common_top")
						.put("simple_logout")
						// その他の URI が指定された場合は認証済であることを検証し、認証済でない場合は遷移する画面を切り替える。
						.get(new If(
								new EvalEL(null, new PathEL("context.container.aaComponent.valid")),
								actionExpression,
								new FormatId(new FixFormat("simple_errorForbidden"))))
						
				),
				toNode,
				HtmlConstants.NORMALIZE_CASE,
				new XPath(new JXP("//A[@href]"), new FormatAttrValue("href", new RegExpFormat("^([^\\.]+)\\.[^\\.]+$", "$1"))),
				new XPath(new JXP("//FORM[@action]"), new FormatAttrValue("action", new RegExpFormat("^([^\\.]+)\\.[^\\.]+$", "$1"))),
				actionBuilder.getAutoUpdateExpression(),
				new Branch<String>(DomConstants.EVAL_ID,
						// ログアウト・認証失敗・認可失敗・不正アクセスの場合はセッションスコープをクリアしログアウト状態とする。
						new SwitchBuilder<String, DomExpression>()
						.put("simple_logout", DomConstants.CLEAR_SESSION)
						.put("simple_errorLogin")
						.put("simple_errorForbidden").get())
		));
	}


	@Override public void init(final ServletConfig servletConfig) throws ServletException {
		super.init(servletConfig);
		toNode.setExpression(new ResetId(new Serial(
				new FormatId(new RegExpFormat("^(.*)$", "/html/sample/aa/$1.html")),
				new XmlToNode(new InputStreamReaderFactory(new ServletInputStreamFactory(servletConfig.getServletContext()), "UTF-8")))));
	}

	@Override protected void service(final HttpServletRequest request, final HttpServletResponse response) throws ServletException, IOException {
		final DomContext domContext = new DomContext();
		domContext.setId(request.getRequestURI());
		domContext.setInput(request.getParameterMap());
		domContext.setRequest(new ServletRequestAttributeMap(request));
		domContext.setSession(UtilsConstants.concurrentMap(new HttpSessionMap(request.getSession()), request.getSession()));
		domContext.setApplication(UtilsConstants.concurrentMap(new ServletContextMap(request.getSession().getServletContext()), request.getSession().getServletContext()));
		domContext.setContainer(new ScopeContainer(domContext.getRequest(), domContext.getSession(), domContext.getApplication(), componentDefMap));
		domExpression.interpret(domContext);
		final Writer w = new BufferedWriter(new OutputStreamWriter(response.getOutputStream(), "UTF-8"));
		try {
			UtilsConstants.TRANSFORMER_FACTORY.newInstance().transform(
					new DOMSource(domContext.getNode()),
					new StreamResult(w)
			);
		} finally {
			w.close();
		}
	}
}
