/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.ee;

import woolpack.action.ActionInvoker;
import woolpack.action.ActionResult;
import woolpack.config.ConfigUtils;
import woolpack.el.ArrayPathEL;
import woolpack.el.EL;
import woolpack.el.GettingEL;
import woolpack.el.MapEL;
import woolpack.fn.Fn;
import woolpack.html.HtmlUtils;
import woolpack.utils.Utils;

/**
 * アクションを実行する{@link Fn}と
 * アクションの実行結果を DOM ノードに自動生成する{@link Fn}のビルダです。
 * <br/>適用しているデザインパターン：Builder。
 * 
 * @author nakamura
 * 
 */
public class ActionBuilder {

	/**
	 * 遷移先で使用するコンポーネントの取得先の{@link EL}のデフォルト値です。
	 */
	public static final EL DEFAULT_FORWARD_COMPONENT_EL_EL = new ArrayPathEL(
			EEUtils.LOCAL_EL, new MapEL("woolpack.ee.FORWARD_COMPONENT"));

	/**
	 * 返却値の取得先のデフォルト値です。
	 */
	public static final EL DEFAULT_RETURN_EL = new ArrayPathEL(
			EEUtils.LOCAL_EL, new MapEL("returnMessages"));

	private ActionInvoker actionDefs;
	private Iterable<String> attrNames;
	private EL forwardComponentELEL;
	private EL returnEL;

	/**
	 * @param actionDefs アクション定義の一覧と遷移先定義の一覧の集合。
	 * @param attrNames 属性名の一覧。本クラスはこの引数の状態を変化させません。
	 * @param forwardComponentELEL 遷移先で使用するコンポーネントの設定先の{@link EL}への参照。
	 * @param returnEL 返却値の設定先への参照。
	 */
	public ActionBuilder(
			final ActionInvoker actionDefs,
			final Iterable<String> attrNames,
			final EL forwardComponentELEL,
			final EL returnEL) {
		this.actionDefs = actionDefs;
		this.attrNames = attrNames;
		this.forwardComponentELEL = forwardComponentELEL;
		this.returnEL = returnEL;
	}

	/**
	 * @param actionDefs アクション定義の一覧と遷移先定義の一覧の集合。
	 * @param attrNames 属性名の一覧。本クラスはこの引数の状態を変化させません。
	 */
	public ActionBuilder(
			final ActionInvoker actionDefs,
			final Iterable<String> attrNames) {
		this(actionDefs, attrNames, DEFAULT_FORWARD_COMPONENT_EL_EL, DEFAULT_RETURN_EL);
	}

	/**
	 * アクションを実行し実行結果を{@link #getReturnEL()}の位置に設定する{@link Fn}を返します。
	 * このクラスは LSP(The Liskov Substitution Principle) を満たしません。
	 * 
	 * @return アクションを実行する{@link Fn}。
	 */
	public Fn<EEContext, Void> getActionExpression() {
		return new Fn<EEContext, Void>() {
			public Void exec(final EEContext context) {
				final ActionResult actionResult = actionDefs.invoke(context, context.getId(), context.getInput());
				context.setId(actionResult.getForwardDef().getId());
				forwardComponentELEL.setValue(context, actionResult.getForwardDef().getComponentEL());
				if (actionResult.getReturnedObject() instanceof Throwable) {
					returnEL.setValue(context, Utils.toMessageList((Throwable) actionResult.getReturnedObject()));
				} else {
					returnEL.setValue(context, actionResult.getReturnedObject());
				}
				return null;
			}
		};
	}

	/**
	 * コンポーネントの取得先をアクションの実行結果から取得し、
	 * 属性値をプロパティ名としてコンポーネントから値を取得し
	 * DOM ノードに自動設定する{@link Fn}を返します。
	 * 
	 * @return アクションの実行結果を DOM ノードに自動生成する{@link Fn}。
	 */
	public Fn<EEContext, Void> getAutoUpdateExpression() {
		return new Fn<EEContext, Void>() {
			public Void exec(final EEContext context) {
				final GettingEL o = (GettingEL) forwardComponentELEL.getValue(context);
				if (o != null) {
					HtmlUtils.updateAuto(attrNames, o, ConfigUtils.CONFIG_EL).exec(context);
				}
				return null;
			}
		};
	}

	public ActionInvoker getActionDefs() {
		return actionDefs;
	}
	public void setActionDefs(final ActionInvoker actionDefs) {
		this.actionDefs = actionDefs;
	}
	public Iterable<String> getAttrNames() {
		return attrNames;
	}
	public void setAttrNames(final Iterable<String> attrNames) {
		this.attrNames = attrNames;
	}
	public EL getForwardComponentELEL() {
		return forwardComponentELEL;
	}
	public void setForwardComponentELEL(final EL forwardComponentELEL) {
		this.forwardComponentELEL = forwardComponentELEL;
	}
	public EL getReturnEL() {
		return returnEL;
	}
	public void setReturnEL(final EL returnEL) {
		this.returnEL = returnEL;
	}
}
