/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.el;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import woolpack.fn.Fn;
import woolpack.typeconvert.ConvertContext;

/**
 * リフレクションを使用した{@link PropertyELFactory}です。
 * getter/setterのオーバロードには対応していません。
 * 
 * @author nakamura
 * 
 */
public class ReflectionPropertyELFactory implements PropertyELFactory {
	private Fn<ConvertContext, Void> fn;

	/**
	 * @param fn 変換器。
	 */
	public ReflectionPropertyELFactory(final Fn<ConvertContext, Void> fn) {
		this.fn = fn;
	}

	private BeanInfo generateBeanInfo(final Class clazz) {
		try {
			return Introspector.getBeanInfo(clazz);
		} catch (final IntrospectionException e) {
			throw new IllegalStateException(e);
		}
	}

	public GettingEL newGetter(final Class clazz, final String propertyName) {
		final BeanInfo beanInfo = generateBeanInfo(clazz);
		for (final PropertyDescriptor p : beanInfo.getPropertyDescriptors()) {
			if (!p.getName().equals(propertyName)) {
				continue;
			}
			final Method m = p.getReadMethod();
			if (m == null) {
				break;
			}
			return new AbstractGettingEL() {
				@Override
				public Object getValue(final Object root, final Class toType) {
					try {
						final Object result = m.invoke(root, new Object[0]);
						final ConvertContext context = new ConvertContext();
						context.setPropertyName(propertyName);
						context.setToType(toType);
						context.setValue(result);
						fn.exec(context);
						return context.getValue();
					} catch (final IllegalAccessException e) {
						throw new IllegalStateException(e);
					} catch (final InvocationTargetException e) {
						throw new ELTargetRuntimeException(e);
					}
				}
			};
		}
		throw new IllegalArgumentException("property not found:" + clazz + ':'
				+ propertyName);
	}

	public EL newSetter(final Class clazz, final String propertyName) {
		final BeanInfo beanInfo = generateBeanInfo(clazz);
		for (final PropertyDescriptor p : beanInfo.getPropertyDescriptors()) {
			if (!p.getName().equals(propertyName)) {
				continue;
			}
			final Method m = p.getWriteMethod();
			if (m == null) {
				break;
			}
			return new AbstractEL() {
				@Override
				public void setValue(final Object root, final Object value) {
					try {
						final ConvertContext context = new ConvertContext();
						context.setPropertyName(propertyName);
						context.setToType(m.getParameterTypes()[0]);
						context.setValue(value);
						fn.exec(context);
						m.invoke(root, new Object[] { context.getValue() });
					} catch (final IllegalAccessException e) {
						throw new IllegalStateException(e);
					} catch (final InvocationTargetException e) {
						throw new ELTargetRuntimeException(e);
					}
				}

				@Override
				public Object getValue(final Object root, final Class clazz) {
					return null;
				}
			};
		}
		throw new IllegalArgumentException("property not found:" + clazz + ':'
				+ propertyName);
	}

	public Fn<ConvertContext, Void> getFn() {
		return fn;
	}
	public void setFn(final Fn<ConvertContext, Void> fn) {
		this.fn = fn;
	}
}
