/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.html;

import junit.framework.TestCase;
import woolpack.adapter.JXP;
import woolpack.el.FixEL;
import woolpack.el.PathEL;
import woolpack.fn.Fn;
import woolpack.utils.Utils;
import woolpack.xml.MapNodeContext;
import woolpack.xml.NodeContext;
import woolpack.xml.SimpleNodeContext;
import woolpack.xml.XmlUtils;

public class ValueUpdaterTest extends TestCase {
	public void testOptionSelect() {
		final Fn<NodeContext, Void> expression = XmlUtils.findNode(
				new JXP("id", "name0"),
				HtmlUtils.updateValue(new FixEL("value1")));
		final NodeContext context = new SimpleNodeContext();
		context.setNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<OPTION id=\"name0\" selected=\"true\" value=\"value0\" >label0</OPTION>"
				+ "<OPTION id=\"name0\" value=\"value1\" >label1</OPTION>"
				+ "</BODY></HTML>"));
		expression.exec(context);
		assertTrue(XmlUtils.equalsNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<OPTION id=\"name0\" value=\"value0\" >label0</OPTION>"
				+ "<OPTION id=\"name0\" value=\"value1\" selected=\"true\" >label1</OPTION>"
				+ "</BODY></HTML>"), context.getNode()));
	}

	public void testSelectSelect() {
		final Fn<NodeContext, Void> expression = XmlUtils.findNode(
				new JXP("name", "name0"),
				new ValueUpdater(new FixEL("value1")));
		final NodeContext context = new SimpleNodeContext();
		context.setNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<SELECT name=\"name0\">"
				+ "<OPTION selected=\"true\" value=\"value0\" >label0</OPTION>"
				+ "<OPTION value=\"value1\" >label1</OPTION>"
				+ "</SELECT>"
				+ "</BODY></HTML>"));
		expression.exec(context);
		assertTrue(XmlUtils.equalsNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<SELECT name=\"name0\">"
				+ "<OPTION value=\"value0\" >label0</OPTION>"
				+ "<OPTION value=\"value1\" selected=\"true\" >label1</OPTION>"
				+ "</SELECT>"
				+ "</BODY></HTML>"), context.getNode()));
	}

	public void testOptionUpdate() {
		final Fn<NodeContext, Void> expression = XmlUtils.findNode(
				new JXP("id", "name0"),
				HtmlUtils.updateValue(new FixEL("value1"), null, false));
		final NodeContext context = new SimpleNodeContext();
		context.setNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<OPTION id=\"name0\" value=\"value0\" >label0</OPTION>"
				+ "</BODY></HTML>"));
		expression.exec(context);
		assertTrue(XmlUtils.equalsNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<OPTION id=\"name0\" value=\"value1\" >label0</OPTION>"
				+ "</BODY></HTML>"), context.getNode()));
	}

	public void testRadioSelect() {
		final Fn<NodeContext, Void> expression = XmlUtils.findNode(
				new JXP("name", "name0"),
				HtmlUtils.updateValue(new FixEL("value1")));
		final NodeContext context = new SimpleNodeContext();
		context.setNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<INPUT name=\"name0\" type=\"radio\" value=\"value0\" checked=\"true\" />"
				+ "<INPUT name=\"name0\" type=\"radio\" value=\"value1\" />"
				+ "</BODY></HTML>"));
		expression.exec(context);
		assertTrue(XmlUtils.equalsNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<INPUT name=\"name0\" type=\"radio\" value=\"value0\" />"
				+ "<INPUT name=\"name0\" type=\"radio\" value=\"value1\" checked=\"true\" />"
				+ "</BODY></HTML>"), context.getNode()));
	}

	public void testRadioUpdate() {
		final Fn<NodeContext, Void> expression = XmlUtils.findNode(
				new JXP("name", "name0"),
				HtmlUtils.updateValue(new FixEL("value1"), null, false));
		final NodeContext context = new SimpleNodeContext();
		context.setNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<INPUT name=\"name0\" type=\"radio\" value=\"value0\" />"
				+ "</BODY></HTML>"));
		expression.exec(context);
		assertTrue(XmlUtils.equalsNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<INPUT name=\"name0\" type=\"radio\" value=\"value1\" />"
				+ "</BODY></HTML>"), context.getNode()));
	}

	public void testInput() {
		final Fn<NodeContext, Void> expression = XmlUtils.findNode(
				new JXP("name", "name0"),
				HtmlUtils.updateValue(new FixEL("value1")));
		final NodeContext context = new SimpleNodeContext();
		context.setNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<INPUT name=\"name0\" value=\"value0\" />"
				+ "</BODY></HTML>"));
		expression.exec(context);
		assertTrue(XmlUtils.equalsNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<INPUT name=\"name0\" value=\"value1\" />"
				+ "</BODY></HTML>"), context.getNode()));
	}

	public void testButton() {
		final Fn<NodeContext, Void> expression = XmlUtils.findNode(
				new JXP("name", "name0"),
				HtmlUtils.updateValue(new FixEL("value1")));
		final NodeContext context = new SimpleNodeContext();
		context.setNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<BUTTON name=\"name0\" value=\"value0\" />"
				+ "</BODY></HTML>"));
		expression.exec(context);
		assertTrue(XmlUtils.equalsNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<BUTTON name=\"name0\" value=\"value1\" />"
				+ "</BODY></HTML>"), context.getNode()));
	}

	public void testTextArea() {
		final Fn<NodeContext, Void> expression = XmlUtils.findNode(
				new JXP("name", "name0"),
				HtmlUtils.updateValue(new FixEL("value1")));
		final NodeContext context = new SimpleNodeContext();
		context.setNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<TEXTAREA name=\"name0\" >value0</TEXTAREA>"
				+ "</BODY></HTML>"));
		expression.exec(context);
		assertTrue(XmlUtils.equalsNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<TEXTAREA name=\"name0\" >value1</TEXTAREA>"
				+ "</BODY></HTML>"), context.getNode()));
	}

	public void testDiv() {
		final Fn<NodeContext, Void> expression = XmlUtils.findNode(
				new JXP("id", "id0"),
				HtmlUtils.updateValue(new FixEL("value1")));
		final NodeContext context = new SimpleNodeContext();
		context.setNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<DIV id=\"id0\" >value0</DIV>"
				+ "</BODY></HTML>"));
		expression.exec(context);
		assertTrue(XmlUtils.equalsNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<DIV id=\"id0\" >value1</DIV>"
				+ "</BODY></HTML>"), context.getNode()));
	}

	public void testSpan() {
		final Fn<NodeContext, Void> expression = XmlUtils.findNode(
				new JXP("id", "id0"),
				HtmlUtils.updateValue(new FixEL("value1")));
		final NodeContext context = new SimpleNodeContext();
		context.setNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<SPAN id=\"id0\" >value0</SPAN>"
				+ "</BODY></HTML>"));
		expression.exec(context);
		assertTrue(XmlUtils.equalsNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<SPAN id=\"id0\" >value1</SPAN>"
				+ "</BODY></HTML>"), context.getNode()));
	}

	public void testSpanSelect() {
		final Fn<NodeContext, Void> expression = XmlUtils.findNode(
				new JXP("id", "id0"),
				HtmlUtils.updateValue(new FixEL("1"), new PathEL("map.list0")));
		final MapNodeContext context = new MapNodeContext();
		context.setNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<SPAN id=\"id0\" >value0</SPAN>"
				+ "</BODY></HTML>"));
		context.setMap(Utils.map("list0", Utils.map("0", "a").map("1", "b").map("2", "c")));
		expression.exec(context);
		assertTrue(XmlUtils.equalsNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<SPAN id=\"id0\" >b</SPAN>"
				+ "</BODY></HTML>"), context.getNode()));
	}

	public void testFileSelect() {
		final Fn<NodeContext, Void> expression = XmlUtils.findNode(
				new JXP("name", "name0"),
				HtmlUtils.updateValue(new FixEL("value1")));
		final NodeContext context = new SimpleNodeContext();
		context.setNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<INPUT type=\"file\" name=\"name0\" />"
				+ "</BODY></HTML>"));
		expression.exec(context);
		assertTrue(XmlUtils.equalsNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<INPUT type=\"file\" name=\"name0\" />"
				+ "</BODY></HTML>"), context.getNode()));
	}

	public void testFileUpdate() {
		final Fn<NodeContext, Void> expression = XmlUtils.findNode(new JXP("name", "name0"),
				HtmlUtils.updateValue(new FixEL("value1"), null, false));
		final NodeContext context = new SimpleNodeContext();
		context.setNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<INPUT type=\"file\" name=\"name0\" />"
				+ "</BODY></HTML>"));
		expression.exec(context);
		assertTrue(XmlUtils.equalsNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<INPUT type=\"file\" name=\"name0\" />"
				+ "</BODY></HTML>"), context.getNode()));
	}

	public void testImageSelect() {
		final Fn<NodeContext, Void> expression = XmlUtils.findNode(
				new JXP("name", "name0"),
				HtmlUtils.updateValue(new FixEL("value1")));
		final NodeContext context = new SimpleNodeContext();
		context.setNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<INPUT type=\"image\" name=\"name0\" />"
				+ "</BODY></HTML>"));
		expression.exec(context);
		assertTrue(XmlUtils.equalsNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<INPUT type=\"image\" name=\"name0\" />"
				+ "</BODY></HTML>"), context.getNode()));
	}

	public void testImageUpdate() {
		final Fn<NodeContext, Void> expression = XmlUtils.findNode(
				new JXP("name", "name0"),
				HtmlUtils.updateValue(new FixEL("value1"), null, false));
		final NodeContext context = new SimpleNodeContext();
		context.setNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<INPUT type=\"image\" name=\"name0\" />"
				+ "</BODY></HTML>"));
		expression.exec(context);
		assertTrue(XmlUtils.equalsNode(XmlUtils.convert(
				"<HTML><BODY>"
				+ "<INPUT type=\"image\" name=\"name0\" />"
				+ "</BODY></HTML>"), context.getNode()));
	}
}
