/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.xml;

import junit.framework.TestCase;

import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import woolpack.bool.BoolUtils;
import woolpack.fn.FnUtils;
import woolpack.fn.ThrowFn;

public class NodeFindableChildNodesImplTest extends TestCase {

	private Node generateSampleNode() {
		return XmlUtils.convert("<HTML><BODY>" +
				"<SPAN id=\"id0\"/>" +
				"<SPAN id=\"id1\"><SPAN id=\"id11\"/></SPAN>" +
				"</BODY></HTML>");
	}
	
	public void testEvaluateOneException() {
		final Node node = generateSampleNode();
		final NodeFindable findable = new NodeFindableChildNodesImpl(
				new ThrowFn<Node, Boolean>(new IllegalStateException()), true);
		try {
			findable.evaluateOne(node);
			fail();
		} catch (final IllegalStateException e) {
		}
	}
	
	public void testEvaluateOneNotFound() {
		final Node node = generateSampleNode();
		final NodeFindable findable = new NodeFindableChildNodesImpl(
				FnUtils.join(XmlUtils.GET_NODE_NAME, BoolUtils.checkEquals("DIV")), true);
		final Node result = findable.evaluateOne(node);
		assertNull(result);
	}
	
	public void testEvaluateOneFound() {
		final Node node = generateSampleNode();
		final NodeFindable findable = new NodeFindableChildNodesImpl(
				FnUtils.join(XmlUtils.GET_NODE_NAME, BoolUtils.checkEquals("SPAN")), true);
		final Node result = findable.evaluateOne(node);
		assertEquals("id0", ((Element) result).getAttribute("id"));
	}
	
	public void testEvaluateListNotFound() {
		final Node node = generateSampleNode();
		final NodeFindable findable = new NodeFindableChildNodesImpl(
				FnUtils.join(XmlUtils.GET_NODE_NAME, BoolUtils.checkEquals("DIV")), true);
		final NodeList result = findable.evaluateList(node);
		assertEquals(0, result.getLength());
	}
	
	public void testEvaluateListShallow() {
		final Node node = generateSampleNode();
		final NodeFindable findable = new NodeFindableChildNodesImpl(
				FnUtils.join(XmlUtils.GET_NODE_NAME, BoolUtils.checkEquals("SPAN")), false);
		final NodeList result = findable.evaluateList(node);
		assertEquals(2, result.getLength());
		assertEquals("id0", ((Element) result.item(0)).getAttribute("id"));
		assertEquals("id1", ((Element) result.item(1)).getAttribute("id"));
	}
	
	public void testEvaluateListDeep() {
		final Node node = generateSampleNode();
		final NodeFindable findable = new NodeFindableChildNodesImpl(
				FnUtils.join(XmlUtils.GET_NODE_NAME, BoolUtils.checkEquals("SPAN")), true);
		final NodeList result = findable.evaluateList(node);
		assertEquals(3, result.getLength());
		assertEquals("id0", ((Element) result.item(0)).getAttribute("id"));
		assertEquals("id1", ((Element) result.item(1)).getAttribute("id"));
		assertEquals("id11", ((Element) result.item(2)).getAttribute("id"));
	}
}
