/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.sql.fn;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;

import woolpack.fn.Fn;

/**
 * 検索結果を一行だけ取得して返す{@link Fn}です。
 * @author nakamura
 *
 * @param <R>
 */
public class OneLineResult<R> implements Fn<Statement, R> {
	private Fn<? super ResultSet, ? extends R> rowFn;
	private Fn<? super Statement, ? extends R> zeroFn;
	
	/**
	 * 
	 * @param rowFn 一行を返却値に変換する委譲先。
	 * @param zeroFn 行が存在しない場合の委譲先。
	 */
	public OneLineResult(
			final Fn<? super ResultSet, ? extends R> rowFn,
			final Fn<? super Statement, ? extends R> zeroFn) {
		this.rowFn = rowFn;
		this.zeroFn = zeroFn;
	}
	
	/**
	 * 行が存在しない場合は{@link IllegalStateException}を投げます。
	 * @param rowFn
	 */
	public OneLineResult(
			final Fn<? super ResultSet, ? extends R> rowFn) {
		this(rowFn, new Fn<Statement, R>() {
			public R exec(final Statement c) {
				throw new IllegalStateException("empty row.");
			}
		});
	}

	public R exec(final Statement c) {
		try {
			final ResultSet resultSet = c.getResultSet();
			try {
				if (resultSet.next()) {
					return rowFn.exec(resultSet);
				} else {
					return zeroFn.exec(c);
				}
			} finally {
				resultSet.close();
			}
		} catch (final SQLException e) {
			throw new IllegalStateException(e);
		}
	}

	public Fn<? super ResultSet, ? extends R> getRowFn() {
		return rowFn;
	}
	public void setRowFn(Fn<? super ResultSet, ? extends R> fn) {
		this.rowFn = fn;
	}
	public Fn<? super PreparedStatement, ? extends R> getZeroFn() {
		return zeroFn;
	}
	public void setZeroFn(final Fn<? super Statement, ? extends R> zeroFn) {
		this.zeroFn = zeroFn;
	}
}
