/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.sql.fn;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.sql.DataSource;

import woolpack.fn.Fn;

/**
 * ユーティリティです。
 * 型推論で表記を簡略するためのスタティックメソッドを含みます。
 * 
 * @author nakamura
 *
 */
public class SqlFnUtils {
	private static final Pattern BIND_COMMENT_SQL_PATTERN = Pattern.compile("[^\\s]*/\\*([^\\*\\s]+)\\s*\\*/[^\\s]*");
	private static final Pattern COMMENT_SQL_PATTERN = Pattern.compile("/\\*\\s[^\\*]*\\*/");
	
	public static final CountResult GET_COUNT = new CountResult();
	public static final SingleResult GET_SINGLE = new SingleResult();
	
	private SqlFnUtils() {
	}
	
	public static <R> BeanResult<R> getBeanResult(final Class<R> clazz) {
		return new BeanResult<R>(clazz);
	}
	
	public static MapResult getMapResult(final Fn<? super Integer, ? extends Map<String, Object>> mapFactory) {
		return new MapResult(mapFactory);
	}
	
	public static MapResult getMapResult() {
		return new MapResult();
	}

	public static <R> ListResult<R> getList(final Fn<? super ResultSet, ? extends R> fn) {
		return new ListResult<R>(fn);
	}

	public static <R> ListResult<R> getList(final Fn<? super ResultSet, ? extends R> fn, final int max) {
		return new ListResult<R>(fn, max);
	}

	public static <R> MapInput<R> inputMap(
			final DataSource dataSource,
			final Fn<? super Map<String, Object>, ? extends PreparedStatementInfo> queryFactory,
			final Fn<? super PreparedStatement, ? extends R> converter) {
		return new MapInput<R>(dataSource, queryFactory, converter);
	}

	public static <R> MapInput<R> inputMap(
			final DataSource dataSource,
			final PreparedStatementInfo info,
			final Fn<? super PreparedStatement, ? extends R> converter) {
		return new MapInput<R>(dataSource, info, converter);
	}

	public static <R> OneLineResult<R> getOne(
			final Fn<? super ResultSet, ? extends R> rowFn,
			final Fn<? super Statement, ? extends R> zeroFn) {
		return new OneLineResult<R>(rowFn, zeroFn);
	}

	public static <R> OneLineResult<R> getOne(
			final Fn<? super ResultSet, ? extends R> rowFn) {
		return new OneLineResult<R>(rowFn);
	}

	public static <C, R> SingleInput<C, R> inputSingle(
			final DataSource dataSource,
			final Fn<? super C, ? extends String> queryFactory,
			final Fn<? super PreparedStatement, ? extends R> converter) {
		return new SingleInput<C, R>(dataSource, queryFactory, converter);
	}

	public static <C, R> SingleInput<C, R> inputSingle(
			final DataSource dataSource,
			final String query,
			final Fn<? super PreparedStatement, ? extends R> converter) {
		return new SingleInput<C, R>(dataSource, query, converter);
	}

	public static <C, R> StatementInput<C, R> inputStatement(
			final DataSource dataSource,
			final Fn<? super C, ? extends String> fn,
			final Fn<? super Statement, ? extends R> converter) {
		return new StatementInput<C, R>(dataSource, fn, converter);
	}

	public static <C, R> StatementInput<C, R> inputStatement(
			final DataSource dataSource,
			final String query,
			final Fn<? super Statement, ? extends R> converter) {
		return new StatementInput<C, R>(dataSource, query, converter);
	}
	
	public static <C, R> RollbackFn<C, R> rollback(final DataSource dataSource) {
		return new RollbackFn<C, R>(dataSource);
	}
	
	public static <C, R> CommitFn<C, R> commit(final DataSource dataSource) {
		return new CommitFn<C, R>(dataSource);
	}

	/**
	 * バインド変数コメント付きのSQLを簡単な正規表現で解析して
	 * {@link PreparedStatementInfo}に変換します。
	 * 正規表現「[^\s]*<span/>/\*([^\*\s]+)\s*\*<span/>/[^\s]*」
	 * で検索して$1を属性名として取得し、マッチ箇所全体を「?」に置き換えます。
	 * そのあと正規表現「/\*\s[^\*]*\*<span/>/」のマッチ箇所を通常コメントとして削除します。
	 * @param bindCommentSQL
	 * @return バインド変数コメント付きのSQLを解析した結果。
	 */
	public static PreparedStatementInfo toPreparedStatementInfo(final String bindCommentSQL) {
		final PreparedStatementInfo info = new PreparedStatementInfo();
		final List<String> list = new ArrayList<String>();
		info.setList(list);
		final Matcher m = BIND_COMMENT_SQL_PATTERN.matcher(bindCommentSQL);
		final StringBuffer sb = new StringBuffer(bindCommentSQL.length());
		while (m.find()) {
			list.add(m.group(1));
			m.appendReplacement(sb, "?");
		}
		m.appendTail(sb);
		info.setQuery(COMMENT_SQL_PATTERN.matcher(sb.toString()).replaceAll(""));
		return info;
	}
}
