/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.web;

import java.util.Map;
import java.util.concurrent.ConcurrentMap;

import woolpack.container.ContainerContext;
import woolpack.container.ContainerUtils;
import woolpack.el.AbstractGettingEL;
import woolpack.el.GettingEL;
import woolpack.factory.ConcurrentMapCache;
import woolpack.factory.FactoryUtils;
import woolpack.factory.MapCache;
import woolpack.fn.Fn;
import woolpack.fn.FnUtils;

/**
 * 各スコープを操作するユーティリティです。
 * @author nakamura
 *
 */
public final class WebUtils {
	/**
	 * リクエストスコープを返す{@link Fn}です。
	 */
	public static final Fn<WebContext, Map<String, Object>> REQUEST_FN =
		new Fn<WebContext, Map<String, Object>>() {
			public Map<String, Object> exec(final WebContext c) {
				return c.getRequest();
			}
	};
	
	/**
	 * セッションスコープを返す{@link Fn}です。
	 */
	public static final Fn<WebContext, ConcurrentMap<String, Object>> SESSION_FN =
		new Fn<WebContext, ConcurrentMap<String, Object>>() {
			public ConcurrentMap<String, Object> exec(final WebContext c) {
				return c.getSession();
			}
	};
	
	/**
	 * アプリケーションスコープを返す{@link Fn}です。
	 */
	public static final Fn<WebContext, ConcurrentMap<String, Object>> APPLICATION_FN =
		new Fn<WebContext, ConcurrentMap<String, Object>>() {
			public ConcurrentMap<String, Object> exec(final WebContext c) {
				return c.getApplication();
			}
	};

	/**
	 * {@link WebContext}を基点とみなして
	 * リクエストスコープにアクセスする{@link GettingEL}です。
	 */
	public static final GettingEL REQUEST_EL = new AbstractGettingEL() {
		@Override
		public Object getValue(final Object root, final Class clazz) {
			return ((WebContext) root).getRequest();
		}
	};
	
	/**
	 * {@link WebContext}を基点とみなして
	 * セッションスコープにアクセスする{@link GettingEL}です。
	 */
	public static final GettingEL SESSION_EL = new AbstractGettingEL() {
		@Override
		public Object getValue(final Object root, final Class clazz) {
			return ((WebContext) root).getSession();
		}
	};

	/**
	 * {@link WebContext}を基点とみなして
	 * アプリケーションスコープにアクセスする{@link GettingEL}です。
	 */
	public static final GettingEL APPLICATION_EL = new AbstractGettingEL() {
		@Override
		public Object getValue(final Object root, final Class clazz) {
			return ((WebContext) root).getApplication();
		}
	};
	
	/**
	 * セッションスコープをクリアする{@link Fn}です。
	 */
	public static final Fn<WebContext, Void> CLEAR_SESSION = new Fn<WebContext, Void>() {
		public Void exec(final WebContext context) {
			context.getSession().clear();
			return null;
		}
	};
	
	private static final Fn<ContainerContext<WebContext>, String> GET_KEY =
		new Fn<ContainerContext<WebContext>, String>() {
		public String exec(final ContainerContext<WebContext> c) {
			return "woolpack.web.WebUtils." + c.getKey();
		}
	};
	private static final Fn<ContainerContext<WebContext>, WebContext> SUB_CONTEXT = ContainerUtils.getSubContext();
	private static final Fn<ContainerContext<WebContext>, Map<String, Object>> REQUEST_MAP = FnUtils.join(SUB_CONTEXT, REQUEST_FN);
	private static final Fn<ContainerContext<WebContext>, ConcurrentMap<String, Object>> SESSION_MAP = FnUtils.join(SUB_CONTEXT, SESSION_FN);
	private static final Fn<ContainerContext<WebContext>, ConcurrentMap<String, Object>> APPLICATION_MAP = FnUtils.join(SUB_CONTEXT, APPLICATION_FN);
	
	
	private WebUtils() {
	}

	/**
	 * リクエストスコープでキャッシュする{@link Fn}を返します。
	 * @param <C>
	 * @param fn
	 * @return リクエストスコープでキャッシュする{@link Fn}。
	 */
	public static <C extends ContainerContext<WebContext>> MapCache<C, String> request(
			final Fn<? super C, ?> fn) {
		return FactoryUtils.cache(REQUEST_MAP, GET_KEY, fn);
	}
	
	/**
	 * セッションスコープでキャッシュする{@link Fn}を返します。
	 * @param <C>
	 * @param fn
	 * @return セッションスコープでキャッシュする{@link Fn}。
	 */
	public static <C extends ContainerContext<WebContext>> ConcurrentMapCache<C, String> session(
			final Fn<? super C, ?> fn) {
		return FactoryUtils.concurrentCache(SESSION_MAP, GET_KEY, fn);
	}
	
	/**
	 * アプリケーションスコープでキャッシュする{@link Fn}を返します。
	 * @param <C>
	 * @param fn
	 * @return アプリケーションスコープでキャッシュする{@link Fn}。
	 */
	public static <C extends ContainerContext<WebContext>> ConcurrentMapCache<C, String> application(
			final Fn<? super C, ?> fn) {
		return FactoryUtils.concurrentCache(APPLICATION_MAP, GET_KEY, fn);
	}
}
