/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.action;

import java.util.Arrays;

import woolpack.el.ELUtils;
import woolpack.el.GettingEL;

/**
 * ひとつのアクション定義を表します。
 * 
 * @author nakamura
 * 
 */
public class ActionDef {
	private GettingEL componentEL;
	private GettingEL methodEL;
	private Iterable<ForwardDef> forwardDefIterable;

	/**
	 * @param componentEL コンポーネントへの参照。コンテキスト役に対して副作用が発生すべきではありません。
	 * @param methodEL 実行するメソッドへの参照。コンテキスト役に対して副作用が発生してもよいです。
	 * @param forwardDefIterable 遷移先定義の一覧。本クラスはこの引数の状態を変化させません。ステートレスである必要があります。
	 */
	public ActionDef(
			final GettingEL componentEL,
			final GettingEL methodEL,
			final Iterable<ForwardDef> forwardDefIterable) {
		this.componentEL = componentEL;
		this.methodEL = methodEL;
		this.forwardDefIterable = forwardDefIterable;
	}

	/**
	 * コンポーネントの取得先を{@link ELUtils#NULL}とします。
	 * 
	 * @param methodEL 実行するメソッドへの参照。コンテキスト役に対して副作用が発生してもよいです。
	 * @param forwardDefIterable 遷移先定義の一覧。本クラスはこの引数の状態を変化させません。ステートレスである必要があります。
	 */
	public ActionDef(
			final GettingEL methodEL,
			final Iterable<ForwardDef> forwardDefIterable) {
		this(ELUtils.NULL, methodEL, forwardDefIterable);
	}

	/**
	 * @param componentEL コンポーネントへの参照。コンテキスト役に対して副作用が発生すべきではありません。
	 * @param methodEL 実行するメソッドへの参照。コンテキスト役に対して副作用が発生してもよいです。
	 * @param forwardDefArray 遷移先定義の一覧。本クラスはこの引数の状態を変化させません。ステートレスである必要があります。
	 */
	public ActionDef(
			final GettingEL componentEL,
			final GettingEL methodEL,
			final ForwardDef... forwardDefArray) {
		this(componentEL, methodEL, Arrays.asList(forwardDefArray));
	}

	/**
	 * コンポーネントの取得先を{@link ELUtils#NULL}とします。
	 * 
	 * @param methodEL 実行するメソッドへの参照。コンテキスト役に対して副作用が発生してもよいです。
	 * @param forwardDefArray 遷移先定義の一覧。本クラスはこの引数の状態を変化させません。ステートレスである必要があります。
	 */
	public ActionDef(
			final GettingEL methodEL,
			final ForwardDef... forwardDefArray) {
		this(ELUtils.NULL, methodEL, forwardDefArray);
	}

	public GettingEL getComponentEL() {
		return componentEL;
	}
	public void setComponentEL(final GettingEL componentEL) {
		this.componentEL = componentEL;
	}
	public Iterable<ForwardDef> getForwardDefIterable() {
		return forwardDefIterable;
	}
	public void setForwardDefIterable(final Iterable<ForwardDef> forwardDefIterable) {
		this.forwardDefIterable = forwardDefIterable;
	}
	public GettingEL getMethodEL() {
		return methodEL;
	}
	public void setMethodEL(final GettingEL methodEL) {
		this.methodEL = methodEL;
	}
}
