/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.action;

import woolpack.bool.BoolUtils;
import woolpack.el.ELUtils;
import woolpack.el.GettingEL;
import woolpack.fn.Fn;

/**
 * ひとつの遷移先定義を表します。
 * 
 * @author nakamura
 * 
 */
public class ForwardDef {
	private String id;
	private GettingEL componentEL;
	private Fn<Object, Boolean, RuntimeException> matcher;

	/**
	 * @param id 遷移先 id。
	 * @param componentEL 遷移先で使用するコンポーネントへの参照。
	 * @param matcher マッチ操作。
	 */
	public ForwardDef(
			final String id,
			final GettingEL componentEL,
			final Fn<Object, Boolean, RuntimeException> matcher) {
		this.id = id;
		this.componentEL = componentEL;
		this.matcher = matcher;
	}

	/**
	 * 遷移先で使用するコンポーネントへの参照を{@link ELUtils#NULL}とします。
	 * 
	 * @param id 遷移先 id。
	 * @param matcher マッチ操作。
	 */
	public ForwardDef(final String id, final Fn<Object, Boolean, RuntimeException> matcher) {
		this(id, ELUtils.NULL, matcher);
	}

	/**
	 * マッチ操作を{@link BoolUtils#NOT_THROWABLE}とします。
	 * 
	 * @param id 遷移先 id。
	 * @param componentEL 遷移先で使用するコンポーネントへの参照。コンテキスト役に対して副作用が発生すべきではありません。
	 */
	public ForwardDef(final String id, final GettingEL componentEL) {
		this(id, componentEL, BoolUtils.NOT_THROWABLE);
	}

	/**
	 * 遷移先で使用するコンポーネントの取得先を{@link ELUtils#NULL}とします。
	 * マッチ操作を{@link BoolUtils#NOT_THROWABLE}とします。
	 * 
	 * @param id 遷移先 id。
	 */
	public ForwardDef(final String id) {
		this(id, ELUtils.NULL, BoolUtils.NOT_THROWABLE);
	}

	public GettingEL getComponentEL() {
		return componentEL;
	}
	public void setComponentEL(final GettingEL componentEL) {
		this.componentEL = componentEL;
	}
	public String getId() {
		return id;
	}
	public void setId(final String id) {
		this.id = id;
	}
	public Fn<Object, Boolean, RuntimeException> getMatcher() {
		return matcher;
	}
	public void setMatcher(final Fn<Object, Boolean, RuntimeException> matching) {
		this.matcher = matching;
	}
}
