/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.sql.convert;

import java.util.regex.Pattern;

/**
 * SQLの変数バインドオペレータを表します。
 * @author nakamura
 *
 */
public enum SqlOperator {
	/**
	 * 演算子「=」を表します。
	 */
	EQ("EQ", toRegExpCol("@s*@=@s*@?")),
	
	/**
	 * 演算子「&lt;」を表します。
	 */
	LT("LT", toRegExpCol("@s*@<@s*@?")),
	
	/**
	 * 演算子「&gt;」を表します。
	 */
	GT("GT", toRegExpCol("@s*@>@s*@?")),
	
	/**
	 * 演算子「&lt;=」を表します。
	 */
	LE("LE", toRegExpCol("@s*@<@=@s*@?")),
	
	/**
	 * 演算子「&gt;=」を表します。
	 */
	GE("GE", toRegExpCol("@s*@>@=@s*@?")),
	
	/**
	 * 演算子「IN」を表します。
	 */
	IN("IN", toRegExpCol("@s+in@s+@(@?(@s*,@s*@?)*@s*@)")),
	
	/**
	 * 演算子「LIKE」を表します。
	 */
	LIKE("LIKE", toRegExpCol("@s+like@s+@?")),
	
	/**
	 * 演算子「BETWEEN」を表します。
	 */
	BETWEEN("BETWEEN", toRegExpCol("@s+between@s+@?@s+and@s+@?"));
	
	/**
	 * {@link #getPattern()}でマッチした際の、
	 * {@link java.util.regex.Matcher#group(int)}におけるカラム名のインデックスです。
	 */
	public static final int COL_GROUP = 1;
	
	/**
	 * {@link #getPattern()}でマッチした際の、
	 * {@link java.util.regex.Matcher#group(int)}におけるバインド文のインデックスです。
	 */
	public static final int BIND_GROUP = 2;

	private final String label;
	private final Pattern pattern;

	private static final Pattern toRegExp(final String s) {
		return Pattern.compile(
				s.replaceAll("@", "\\\\"),
				Pattern.CASE_INSENSITIVE);
	}
	private static final Pattern toRegExpCol(final String operator) {
		return toRegExp("([^,@s@(@)@<@>@=]+)(" + operator + ")");
	}
	
	private SqlOperator(
			final String label,
			final Pattern pattern) {
		this.label = label;
		this.pattern = pattern;
	}

	/**
	 * パラメータ名にオペレータ名を付ける際のラベルを返します。
	 * @return パラメータ名にオペレータ名を付ける際のラベル。
	 */
	public String getLabel() {
		return label;
	}
	
	/**
	 * SQLからバインド文を検索するためのパターンを返します。
	 * @return SQLからバインド文を検索するためのパターン。
	 */
	public Pattern getPattern() {
		return pattern;
	}
}
