/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.sql.fn;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;

import woolpack.fn.Fn;

public class ListResult<R> implements Fn<Statement, List<R>, Exception> {
	private Fn<? super ResultSet, ? extends R, ? extends Exception> rowFn;
	private int max;
	private Fn<? super SQLException, ?, ? extends Exception> errorFn;
	
	public ListResult(
			final Fn<? super ResultSet, ? extends R, ? extends Exception> fn,
			final int max,
			final Fn<? super SQLException, ?, ? extends Exception> errorFn) {
		this.rowFn = fn;
		this.max = max;
		this.errorFn = errorFn;
	}

	public List<R> exec(final Statement c) throws Exception {
		SQLException e0 = null;
		try {
			final ResultSet resultSet = c.getResultSet();
			try {
				final List<R> list = new ArrayList<R>();
				int i = 0;
				while (resultSet.next()) {
					i++;
					if (i > max) {
						break;
					}
					list.add(rowFn.exec(resultSet));
				}
				return list;
			} catch (final SQLException e1) {
				errorFn.exec(e1);
				e0 = e1;
			} finally {
				resultSet.close();
			}
		} catch (final SQLException e1) {
			errorFn.exec(e1);
			if (e0 == null) {
				e0 = e1;
			}
		}
		// 委譲先でSQLException以外の例外が発生した場合は以下のブロックは実行されない。
		if (e0 != null) {
			throw e0;
		} else {
			return null;
		}
	}

	public Fn<? super ResultSet, ? extends R, ? extends Exception> getRowFn() {
		return rowFn;
	}
	public void setRowFn(Fn<? super ResultSet, ? extends R, ? extends Exception> fn) {
		this.rowFn = fn;
	}
	public int getMax() {
		return max;
	}
	public void setMax(final int max) {
		this.max = max;
	}
	public Fn<? super SQLException, ?, ? extends Exception> getErrorFn() {
		return errorFn;
	}
	public void setErrorFn(final Fn<? super SQLException, ?, ? extends Exception> errorFn) {
		this.errorFn = errorFn;
	}
}
