/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.adapter;

import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.WeakHashMap;

import junit.framework.TestCase;
import ognl.OgnlException;
import woolpack.el.ELTargetRuntimeException;
import woolpack.test.TestBean;

public class OGETest extends TestCase {
	static class OGETestMap extends LinkedHashMap<String, Object> {
		/**
		 * 
		 */
		private static final long serialVersionUID = 1L;

		@Override
		protected boolean removeEldestEntry(final Map.Entry<String, Object> eldest) {
			return this.size() > 2;
		}
	}
	
	public void testException() {
		try {
			new OGE("a.");
			fail();
		} catch (final IllegalArgumentException e) {
			assertTrue(e.getCause() instanceof OgnlException);
		}
	}

	public void testChangeCache() {
		try {
			OGE.changeCache(null);
			fail();
		} catch (final NullPointerException e) {
		}
		final LinkedHashMap<String, Object> map = new OGETestMap();
		OGE.changeCache(map);
		try {
			map.clear();
			new OGE("a");
			assertEquals(1, map.size());
			new OGE("b");
			assertEquals(2, map.size());
			new OGE("c");
			assertEquals(2, map.size());
		} finally {
			OGE.changeCache(new WeakHashMap<String, Object>());
		}
	}
	
	public void testSetValueGetValue() {
		final Map<String, String> map = new HashMap<String, String>();
		map.put("param0", "value0");
		assertEquals("value0", new OGE("param0").getValue(map));
		assertEquals("value0", new OGE("param0").getValue(map, String.class));

		assertTrue(new OGE("param1").setValue(map, "value1"));
		assertEquals("value1", new OGE("param1").getValue(map));

		assertNull(new OGE("param2").getValue(map));

		assertTrue(new OGE("param3").setValue(map, null));
		assertNull(new OGE("param3").getValue(map));
	}

	public void testString() {
		final Map<String, Object> map = new HashMap<String, Object>();
		map.put("param0", Integer.valueOf(7));
		assertEquals(Integer.valueOf(7), new OGE("param0").getValue(map));
		assertTrue(new OGE("param0").getValue(map, String.class) instanceof String);
		assertEquals("7", new OGE("param0").getValue(map, String.class));

		assertTrue(new OGE("param1").setValue(map, Integer.valueOf(11)));
		assertEquals(Integer.valueOf(11), new OGE("param1").getValue(map));
		assertEquals("11", new OGE("param1").getValue(map, String.class));
	}

	public void testCollection() {
		assertEquals(Arrays.asList(13, 17), new OGE("#this").getValue(new int[] { 13, 17 }, List.class));
	}

	public void testGetValueException() {
		final Map map = new HashMap();
		try {
			new OGE("param0.param1").getValue(map);
			fail();
		} catch (final IllegalStateException expected) {
			assertTrue(expected.getCause() instanceof OgnlException);
		}
	}

	public void testGetStringException() {
		final Map map = new HashMap();
		try {
			new OGE("param0.param1").getValue(map, String.class);
			fail();
		} catch (final IllegalStateException expected) {
			assertTrue(expected.getCause() instanceof OgnlException);
		}
	}

	public void testSetValueException() {
		final Map map = new HashMap();
		try {
			new OGE("param0.param1").setValue(map, "a");
			fail();
		} catch (final IllegalStateException expected) {
			assertTrue(expected.getCause() instanceof OgnlException);
		}
	}

	public void testInvoke() {
		final Map<String, Object> root = new HashMap<String, Object>();
		root.put("component0", new TestBean());
		{
			final Object result = new OGE("component0.executeVoid()")
					.getValue(root);
			assertNull(result);
		}
		{
			Object result = new OGE("component0.executeInt()").getValue(root);
			assertEquals(Integer.valueOf(1), result);

			result = new OGE("component0.executeInt()").getValue(root);
			assertEquals(Integer.valueOf(2), result);
		}
		try {
			new OGE("component0.executeException()").getValue(root);
			fail();
		} catch (final ELTargetRuntimeException expected) {
			assertTrue(expected.getCause() instanceof IndexOutOfBoundsException);
		}
	}

	public void testInvokeMethodNotFound() {
		final Map<String, Object> root = new HashMap<String, Object>();
		root.put("component0", new TestBean());
		try {
			new OGE("component0.executeNotFound()").getValue(root);
			fail();
		} catch (final UnsupportedOperationException expected) {
			assertTrue(expected.getCause() instanceof OgnlException);
		}
	}

	public void testInvokeException() {
		final Map<String, Object> root = new HashMap<String, Object>();
		root.put("component0", new TestBean());
		try {
			new OGE("component0.notFoundProperty").getValue(root);
			fail();
		} catch (final IllegalStateException expected) {
			assertTrue(expected.getCause() instanceof OgnlException);
		}
	}
}
