/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.xml;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.xml.transform.TransformerException;

import junit.framework.TestCase;

import org.w3c.dom.Document;
import org.w3c.dom.Node;

import woolpack.adapter.JXPUtils;
import woolpack.convert.ConvertUtils;
import woolpack.el.PathEL;
import woolpack.factory.FactoryUtils;
import woolpack.fn.Fn;
import woolpack.fn.FnUtils;
import woolpack.utils.Utils;

public class XmlUtilsTest extends TestCase {
	
	public void testNodeEL() throws TransformerException {
		final Node node = XmlTransformerUtils.convert("<HTML><BODY></BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		assertTrue(XmlUtils.NODE_EL.setValue(context, node));
		assertSame(node, XmlUtils.NODE_EL.getValue(context));
	}
	
	public void testBranchByAttrValue() throws TransformerException {
		
		final NodeContext context = new SimpleNodeContext();
		context.setNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<A href=\"page0.html\" >page0</A>"
				+ "<A href=\"page1.html\" >page1</A>"
				+ "<A href=\"page2.html\" >page2</A>"
				+ "<A href=\"page3.html\" >page3</A>"
				+ "</BODY></HTML>"));
		
		XmlUtils.findNode(
				JXPUtils.list("//A"), 
				XmlUtils.branchByAttrValue(Arrays.asList("href"), FnUtils.switching(
						Utils
						.map("page0.html", XmlUtils.updateAttrValue("href", FnUtils.fix("page0.do")))
						.map("page1.html", XmlUtils.updateAttrValue("href", FnUtils.fix("page1.do")))
						.map("page2.html", XmlUtils.updateAttrValue("href", FnUtils.fix("page2.do"))),
						FnUtils.fix(XmlUtils.updateAttrValue("href", FnUtils.fix("pagee.do")))
						))).exec(context);

		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<A href=\"page0.do\" >page0</A>"
				+ "<A href=\"page1.do\" >page1</A>"
				+ "<A href=\"page2.do\" >page2</A>"
				+ "<A href=\"pagee.do\" >page3</A>"
				+ "</BODY></HTML>"), context.getNode()));
	}
	
	public void testCloneNode() throws TransformerException {
		final NodeContext context = new SimpleNodeContext();
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "a"
				+ "</BODY></HTML>");
		context.setNode(node);
		
		XmlUtils.CLONE_NODE.exec(context);
		
		assertNotSame(node, context.getNode());
		assertTrue(XmlUtils.equalsNode(node, context.getNode()));
	}

	public void testBranchByAttrValueNotOther() throws TransformerException {
		final NodeContext context = new SimpleNodeContext();
		context.setNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<A href=\"page0.html\" >page0</A>"
				+ "<A href=\"page1.html\" >page1</A>"
				+ "<A href=\"page2.html\" >page2</A>"
				+ "<A href=\"page3.html\" >page3</A>"
				+ "</BODY></HTML>"));
		
		XmlUtils.findNode(
				JXPUtils.list("//A"), 
				XmlUtils.branchByAttrValue(Arrays.asList("href"), FnUtils.switching(
						Utils
						.map("page0.html", XmlUtils.updateAttrValue("href", FnUtils.fix("page0.do")))
						.map("page1.html", XmlUtils.updateAttrValue("href", FnUtils.fix("page1.do")))
						.map("page2.html", XmlUtils.updateAttrValue("href", FnUtils.fix("page2.do")))
						))).exec(context);

		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<A href=\"page0.do\" >page0</A>"
				+ "<A href=\"page1.do\" >page1</A>"
				+ "<A href=\"page2.do\" >page2</A>"
				+ "<A href=\"page3.html\" >page3</A>"
				+ "</BODY></HTML>"), context.getNode()));
	}
	
	public void testFindNode() throws TransformerException {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<span id=\"id0\" />"
				+ "<span id=\"id0\" />"
				+ "<span id=\"id0\" />"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node);
		
		XmlUtils.findNode(
				JXPUtils.list("id", "id0"),
				XmlUtils.replaceTextToChild(FnUtils.fix("value0")),
				XmlUtils.replaceTextToChild(FnUtils.fix("value1")))
		.exec(context);
		
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<span id=\"id0\" >value0</span>"
				+ "<span id=\"id0\" >value1</span>"
				+ "<span id=\"id0\" >value1</span>"
				+ "</BODY></HTML>"), node));
	}
	
	public void testGetAttrValue() throws TransformerException {
		final NodeContext context = new SimpleNodeContext();
		context.setNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<div id=\"hoge\">0</div>"
				+ "<span>1</span>"
				+ "</BODY></HTML>"));
		context.setNode(context.getNode().getFirstChild().getFirstChild().getFirstChild());
		
		assertEquals("hoge", XmlUtils.getAttrValue("id").exec(context));
	}
	
	public void testNodeName() throws TransformerException {
		final NodeContext context = new SimpleNodeContext();
		context.setNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<div>0</div>"
				+ "<span>1</span>"
				+ "</BODY></HTML>"));
		context.setNode(context.getNode().getFirstChild().getFirstChild().getFirstChild());
		
		assertEquals("div", FnUtils.join(XmlUtils.GET_NODE, XmlUtils.GET_NODE_NAME).exec(context));
	}
	
	public void testInsertElementToChild() throws TransformerException {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<SPAN id=\"id0\" >hoge0<SPAN id=\"id1\" >hoge1<SPAN id=\"id2\" >value0</SPAN>piyo1</SPAN>piyo0</SPAN>"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild());
		
		XmlUtils.insertElementToChild("FORM", XmlUtils.updateAttrValue("action", FnUtils.fix("action0")))
		.exec(context);
		
		node.normalize();
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<SPAN id=\"id0\" ><FORM action=\"action0\">"
				+ "hoge0<SPAN id=\"id1\" >hoge1<SPAN id=\"id2\" >value0</SPAN>piyo1</SPAN>piyo0</FORM></SPAN>"
				+ "</BODY></HTML>"), node));
	}

	public void testInsertElementToChildNotExpression() throws TransformerException {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<SPAN id=\"id0\" >hoge0<SPAN id=\"id1\" >hoge1<SPAN id=\"id2\" >value0</SPAN>piyo1</SPAN>piyo0</SPAN>"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild());
		
		XmlUtils.insertElementToChild("FORM").exec(context);
		
		node.normalize();
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<SPAN id=\"id0\" ><FORM>"
				+ "hoge0<SPAN id=\"id1\" >hoge1<SPAN id=\"id2\" >value0</SPAN>piyo1</SPAN>piyo0</FORM></SPAN>"
				+ "</BODY></HTML>"), node));
	}
	
	public void testInsertElementToParent() throws TransformerException {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<SPAN id=\"id0\" >hoge0<SPAN id=\"id1\" >hoge1<SPAN id=\"id2\" >value0</SPAN>piyo1</SPAN>piyo0</SPAN>"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild().getFirstChild().getNextSibling());
		
		XmlUtils.insertElementToParent("FORM", XmlUtils.updateAttrValue("action", FnUtils.fix("action0")))
		.exec(context);
		
		node.normalize();
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<SPAN id=\"id0\" >hoge0<FORM action=\"action0\"><SPAN id=\"id1\" >hoge1<SPAN id=\"id2\" >value0</SPAN>piyo1</SPAN></FORM>piyo0</SPAN>"
				+ "</BODY></HTML>"), node));
	}

	public void testInsertElementToParentNotExpression() throws TransformerException {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<SPAN id=\"id0\" >hoge0<SPAN id=\"id1\" >hoge1<SPAN id=\"id2\" >value0</SPAN>piyo1</SPAN>piyo0</SPAN>"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild().getFirstChild().getNextSibling());
		
		XmlUtils.insertElementToParent("FORM").exec(context);
		
		node.normalize();
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<SPAN id=\"id0\" >hoge0<FORM><SPAN id=\"id1\" >hoge1<SPAN id=\"id2\" >value0</SPAN>piyo1</SPAN></FORM>piyo0</SPAN>"
				+ "</BODY></HTML>"), node));
	}

	
	private void scenarioLoop(final Object collection) throws TransformerException {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<TABLE>"
				+ "<TR id=\"iterate0\"><TD><SPAN id=\"name0\">dummy</SPAN></TD></TR>"
				+ "</TABLE>"
				+ "</BODY></HTML>");
		final MapNodeContext context = new MapNodeContext();
		context.setMap(Utils.map("list0", collection));
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild().getFirstChild());
		
		XmlUtils.copyTemplate(
				new PathEL("map.list0"),
				new PathEL("map.bean"),
				XmlUtils.findNode(
						JXPUtils.list("id", "name0"),
						XmlUtils.replaceTextToChild(
								FnUtils.join(
										FactoryUtils.doEL(new PathEL("map.bean")),
										ConvertUtils.TO_STRING))))
										.exec(context);
		
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<TABLE>"
				+ "<TR id=\"iterate0\"><TD><SPAN id=\"name0\">value0</SPAN></TD></TR>"
				+ "<TR id=\"iterate0\"><TD><SPAN id=\"name0\">value1</SPAN></TD></TR>"
				+ "<TR id=\"iterate0\"><TD><SPAN id=\"name0\">value2</SPAN></TD></TR>"
				+ "</TABLE>"
				+ "</BODY></HTML>"), node));
	}
	
	public void testLoopList() throws TransformerException {
		scenarioLoop(Arrays.asList("value0", "value1", "value2"));
	}
	
	public void testLoopArray() throws TransformerException {
		scenarioLoop(new String[]{"value0", "value1", "value2"});
	}

	public void testNodeFactory() throws Exception {
		final Node node = XmlUtils.nodeFactory(
				FactoryUtils.stringReaderFactory("<HTML><BODY>myxml</BODY></HTML>"),
				XmlTransformerUtils.TRANSFORMER
		).exec("a");
		assertEquals("myxml", node.getFirstChild().getFirstChild().getFirstChild().getNodeValue());
	}

	public void testNormalize() throws TransformerException {
		final NodeContext context = new SimpleNodeContext();
		context.setNode(XmlTransformerUtils.convert("<HTML><BODY>a</BODY></HTML>"));
		final Node n1 = JXPUtils.one("/HTML/BODY").exec(context.getNode());
		n1.appendChild(((Document) context.getNode()).createTextNode("b"));
		n1.appendChild(((Document) context.getNode()).createTextNode("c"));
		assertEquals(3, n1.getChildNodes().getLength());
		
		XmlUtils.NORMALIZE_NODE.exec(context);
		
		assertEquals(1, n1.getChildNodes().getLength());
	}
	
	public void testRemoveAttr() throws TransformerException {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<div id=\"hoge\">0</div>"
				+ "<span>1</span>"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild());
		
		XmlUtils.removeAttr("id").exec(context);
		
		node.normalize();
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<div>0</div>"
				+ "<span>1</span>"
				+ "</BODY></HTML>"), node));
	}

	public void testRemove() throws TransformerException {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<div>0</div>"
				+ "<span>1</span>"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild());
		
		XmlUtils.REMOVE_THIS.exec(context);
		
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<span>1</span>"
				+ "</BODY></HTML>"), node));
	}

	public void testRemoveChildren() throws TransformerException {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<div>0</div>"
				+ "<span>1</span>"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild());
		
		XmlUtils.REMOVE_CHILDREN.exec(context);
		
		node.normalize();
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<div></div>"
				+ "<span>1</span>"
				+ "</BODY></HTML>"), node));
	}
	
	public void testReplaceText() throws TransformerException {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "hoge<TEXTAREA name=\"name0\" >value0</TEXTAREA>piyo"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild().getNextSibling());
		
		XmlUtils.replaceText(FnUtils.fix("value1")).exec(context);
		
		node.normalize();
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "hogevalue1piyo"
				+ "</BODY></HTML>"), node));
	}

	public void testReplaceTextNull() throws TransformerException {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "hoge<TEXTAREA name=\"name0\" >value0</TEXTAREA>piyo"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild().getNextSibling());
		
		XmlUtils.replaceText(FnUtils.fix((String) null)).exec(context);
		
		node.normalize();
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "hoge<TEXTAREA name=\"name0\" >value0</TEXTAREA>piyo"
				+ "</BODY></HTML>"), node));
	}

	public void testReplaceTextToChild() throws TransformerException {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<TEXTAREA name=\"name0\" >value0</TEXTAREA>"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild());
		
		XmlUtils.replaceTextToChild(FnUtils.fix("value1")).exec(context);
		
		node.normalize();
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<TEXTAREA name=\"name0\" >value1</TEXTAREA>"
				+ "</BODY></HTML>"), node));
	}

	public void testReplaceTextToChildNull() throws TransformerException {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<TEXTAREA name=\"name0\" >value0</TEXTAREA>"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild());
		
		XmlUtils.replaceTextToChild(FnUtils.fix((String) null)).exec(context);
		
		node.normalize();
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<TEXTAREA name=\"name0\" >value0</TEXTAREA>"
				+ "</BODY></HTML>"), node));
	}

	public void testRetainChildren() throws TransformerException {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<div>0</div>"
				+ "<span>1</span>"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild());
		
		XmlUtils.RETAIN_CHILDREN.exec(context);
		
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "0"
				+ "<span>1</span>"
				+ "</BODY></HTML>"), node));
	}

	public void testSeekNode() throws TransformerException {
		final List<String> list = new ArrayList<String>();
		final NodeContext context = new SimpleNodeContext();
		context.setNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<SPAN><SPAN>a</SPAN><SPAN>b</SPAN></SPAN><SPAN>c</SPAN>"
				+ "</BODY></HTML>"));
		
		XmlUtils.seekNode(new Fn<NodeContext, Void, RuntimeException>() {
			public Void exec(final NodeContext context) {
				list.add(context.getNode().getNodeName());
				return null;
			}
		}).exec(context);

		final List<String> expected = new ArrayList<String>();
		expected.add("#document");
		expected.add("HTML");
		expected.add("BODY");
		expected.add("SPAN");
		expected.add("SPAN");
		expected.add("#text");
		expected.add("SPAN");
		expected.add("#text");
		expected.add("SPAN");
		expected.add("#text");
		assertEquals(expected, list);
	}
	
	public void testSetNode() throws TransformerException {
		final Node node = XmlTransformerUtils.convert("<HTML><BODY></BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		
		XmlUtils.setNode(FnUtils.<NodeContext, Node>fix(node)).exec(context);
		
		assertSame(node, context.getNode());
	}

	public void testUpdateAttrValue() throws TransformerException {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<div id=\"hoge\">0</div>"
				+ "<span>1</span>"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild());
		
		XmlUtils.updateAttrValue("id", FnUtils.fix("piyo")).exec(context);
		
		node.normalize();
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<div id=\"piyo\">0</div>"
				+ "<span>1</span>"
				+ "</BODY></HTML>"), node));
	}
	
	public void testUpdateAttrValueNull() throws TransformerException {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<div id=\"hoge\">0</div>"
				+ "<span>1</span>"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild());
		
		XmlUtils.updateAttrValue("id", FnUtils.fix((String) null)).exec(context);
		
		node.normalize();
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<div id=\"hoge\">0</div>"
				+ "<span>1</span>"
				+ "</BODY></HTML>"), node));
	}


	private void scenarioEqualsNode(final boolean expected, final String s0, final String s1) throws TransformerException {
		final Node node0 = XmlTransformerUtils.convert("<HTML><BODY>" + s0 + "</BODY></HTML>");
		final Node node1 = XmlTransformerUtils.convert("<HTML><BODY>" + s1 + "</BODY></HTML>");
		assertEquals(expected, XmlUtils.equalsNode(node0, node1));
		assertTrue(node0 != node1);
	}

	public void testDefferentDocument() throws TransformerException {
		scenarioEqualsNode(true,
				"<!-- hoge -->" + "<TEXTAREA name=\"name0\" >value0</TEXTAREA>",
				"<!-- hoge -->" + "<TEXTAREA name=\"name0\" >value0</TEXTAREA>");
	}

	public void testAttrOrder() throws TransformerException {
		scenarioEqualsNode(true,
				"<!-- hoge -->" + "<TEXTAREA id=\"id0\" name=\"name0\" >value0</TEXTAREA>",
				"<!-- hoge -->" + "<TEXTAREA name=\"name0\" id=\"id0\" >value0</TEXTAREA>");
	}

	public void testDefferentAttrValue() throws TransformerException {
		scenarioEqualsNode(false,
				"<!-- hoge -->" + "<TEXTAREA name=\"name0\" >value0</TEXTAREA>",
				"<!-- hoge -->" + "<TEXTAREA name=\"name1\" >value0</TEXTAREA>");
	}

	public void testDefferentAttrRack() throws TransformerException {
		scenarioEqualsNode(false,
				"<!-- hoge -->" + "<TEXTAREA name=\"name0\" >value0</TEXTAREA>",
				"<!-- hoge -->" + "<TEXTAREA >value0</TEXTAREA>");
	}

	public void testDefferentOption() throws TransformerException {
		scenarioEqualsNode(true,
				"<!-- hoge -->" + "<SELECT name=\"name0\" ><OPTION value=\"value0\" >label0</OPTION></SELECT>",
				"<!-- hoge -->" + "<SELECT name=\"name0\" ><OPTION value=\"value0\" >label0</OPTION></SELECT>");
	}

	public void testDefferentChildRack() throws TransformerException {
		scenarioEqualsNode(false,
				"<!-- hoge -->" + "<SELECT name=\"name0\" ><OPTION value=\"value0\" >label0</OPTION></SELECT>",
				"<!-- hoge -->" + "<SELECT name=\"name0\" ></SELECT>");
	}

	public void testDefferentAttrName() throws TransformerException {
		scenarioEqualsNode(false, "<!-- hoge -->"
				+ "<TEXTAREA name=\"name0\" >value0</TEXTAREA>",
				"<!-- hoge -->" + "<TEXTAREA namf=\"name0\" >value0</TEXTAREA>");
	}

	public void testDefferentText() throws TransformerException {
		scenarioEqualsNode(false, 
				"<!-- hoge -->" + "<TEXTAREA name=\"name0\" >value0</TEXTAREA>",
				"<!-- hoge -->" + "<TEXTAREA name=\"name0\" >value1</TEXTAREA>");
	}

	public void testDefferentComment() throws TransformerException {
		scenarioEqualsNode(false,
				"<!-- hoge -->" + "<TEXTAREA name=\"name0\" >value0</TEXTAREA>",
				"<!-- piyo -->" + "<TEXTAREA name=\"name0\" >value0</TEXTAREA>");
	}

	public void testNull0() throws TransformerException {
		final Node node = XmlTransformerUtils.convert("<!-- hoge --><TEXTAREA name=\"name0\" >value0</TEXTAREA>");
		assertTrue(XmlUtils.equalsNode(null, null));
		assertFalse(XmlUtils.equalsNode(null, node));
		assertFalse(XmlUtils.equalsNode(node, null));
	}

	public void testDefferentElementName() throws TransformerException {
		scenarioEqualsNode(false, 
				"<!-- hoge -->" + "<SPAN>hoge</SPAN>",
				"<!-- hoge -->" + "<DIV>hoge</DIV>");
	}

	public void testDefferentNodeType() throws TransformerException {
		scenarioEqualsNode(false, 
				"<!-- hoge -->" + "hoge",
				"<!-- hoge -->" + "<DIV>hoge</DIV>");
	}
}
