<?php
//  $Revision: 1.32 $                                                           //
//  --------------------------------------------------------------------------  //
//  XooNIps Xoops modules for Neuroinformatics Platforms                        //
//  Copyright (C) 2005 RIKEN, Japan. All rights reserved.                       //
//  http://sourceforge.jp/projects/xoonips/                                     //
//  --------------------------------------------------------------------------  //
//  This program is free software; you can redistribute it and/or               //
//  modify it under the terms of the GNU General Public License                 //
//  as published by the Free Software Foundation; either version 2              //
//  of the License, or (at your option) any later version.                      //
//                                                                              //
//  This program is distributed in the hope that it will be useful,             //
//  but WITHOUT ANY WARRANTY; without even the implied warranty of              //
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               //
//  GNU General Public License for more details.                                //
//                                                                              //
//  You should have received a copy of the GNU General Public License           //
//  along with this program; if not, write to the Free Software                 //
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. //
//  --------------------------------------------------------------------------  //

include_once XOOPS_ROOT_PATH.'/modules/xoonips/condefs.php';
include_once 'lib.php';

$search_modules = xnpGetFileSearchModules();

/**
 * 
 * Return xml with Basic information.
 * 
 * To write xml of item to specified folder.
 * 
 * In the case of writing attachment file,  at first create a files/ folder in the specified folder. 
 * next it write attachment file to the folder.
 * 
 * path of xml file is "${export_path}/${item_id}.xml"
 * path of attchment file is "${export_path}/files/${file_id}"
 * 
 * If unknown item_id or error in database exists, return false.
 * 
 * @param export_path folder that export file is written to.
 * @param export_xml name of export file(xml file)
 * @param item_id item id that is exported
 * @param attachment true if attachment files are exported, else false.
 * @return returns array(
 *     'path' => folder that export files are written to.
 *     'xml' => file path of xml(relative path of 'path')
 *     'attachments' => arary( file path of attachment 1, file path of attachment2, .... ) ) (relative path of 'path')
 *     
 *     return false if failed.
 */
function xnpExportItem( $export_path, $item_id, $attachment = false, $is_absolute, $base_index_id = false )
{
    $filename = "${export_path}/${item_id}.xml";
    $tmpfile = tempnam( "/tmp", "XooNIps" );

    $fhdl = fopen($tmpfile, "w");
    if( !$fhdl ){
        xoonips_error( "can't open file '${tmpfile}' for write." );
        return false;
    }
    
	$xnpsid = $_SESSION['XNPSID'];
    $item = array();
    $itemtypes = array();

    $res = xnp_get_item( $xnpsid, $item_id, $item );
    if( $res != RES_OK ){
        return false;
    }
    
    $res = xnp_get_item_types( $itemtypes );
    if( $res != RES_OK ){
        return false;
    }else{
        foreach( $itemtypes as $i){
            if( $i['item_type_id'] == $item['item_type_id'] ){
                $itemtype = $i;
                break;
            }
        }
    }
    if( !isset( $itemtype ) ){ return false; }
    
    include_once XOOPS_ROOT_PATH . '/modules/' . $itemtype['viewphp'];
    
    if( !fwrite( $fhdl, "<item version=\"1.00\">\n" ) ) return false;
    
    // item type module doesn't support export
    $func = "${itemtype['name']}ExportItem";
    if( !function_exists( $func ) ){ return false; }
    
    if( !xnpExportBasic( $fhdl, $item_id, $is_absolute, $base_index_id ) ) return false;
    
    $attachment_files = $func( $export_path, $fhdl, $item_id, $attachment );
    if( !$attachment_files ) return false;
    
    if( !fwrite( $fhdl, "</item>\n" ) ) return false;
    
    fclose( $fhdl );
    
    // ݡȥեʸɤ򡤴Ķ˹碌Ѵ
    $fp_r = fopen($tmpfile, "r");
    $fp_w = fopen($filename, "w");
    if( !$fp_r || !$fp_w ){
        if( !$fp_r ) xoonips_error( "can't open file '${tmpfile}' for read." );
        if( !$fp_w ) xoonips_error( "can't open file '${filename}' for write." );
        unlink( $tmpfile );
        unlink( $filename );
        if( $fp_r ) fclose( $fp_r );
        if( $fp_w ) fclose( $fp_w );
        return false;
    }
    while( !feof( $fp_r ) ){
        fputs( $fp_w, encodeServer2Client( fgets( $fp_r, 131072 ) ) );
    }
    fclose( $fp_r );
    fclose( $fp_w );
    unlink( $tmpfile );
    
    
    return array( 'path' => $export_path,
                  'xml' => "${item_id}.xml",
                  'attachments' => $attachment_files['attachments'] );
}

/******************************************************************/
//Import 
$parser_hash = array();
function startElement($parser, $name, $attribs){
    global $currentTag, $currentAttribs; 
    global $parser_hash;
    $currentTag = $name; 
    
    //return if item_id is not in accept_id
    if( array_key_exists( 'basic', $parser_hash[$parser] )
        && array_key_exists( 'ID', $parser_hash[$parser]['basic'] )
        && !in_array( $parser_hash[$parser]['basic']['ID'],
                      $parser_hash[$parser]['accept_id'] ) ) return;
    
    $currentAttribs = $attribs; 
    
    array_push( $parser_hash[$parser]['tagstack'], $name );
    
    $tags = "/".implode( '/', $parser_hash[$parser]['tagstack'] );
    if( array_key_exists( $tags, $parser_hash[$parser]['handler'] ) ){
        array_push( $parser_hash[$parser]['handlerstack'], $parser_hash[$parser]['handler'][$tags] );
    }
    
    if( count( $parser_hash[$parser]['handlerstack'] ) > 0 ){
        $handler = end( $parser_hash[$parser]['handlerstack'] );
        //echo "call ${handler[0]}\n";
        if( function_exists( $handler[0] ) )
            $handler[0]($parser, $name, $attribs, $parser_hash[$parser] );
        return;
    }
//    print_r( $parser_hash[$parser] );
//    echo("<b>&lt$name&gt</b><br>"); 
    
} 

function endElement($parser, $name) { 
    global $currentTag; 
    global $parser_hash;
    
    //return if item_id is not in accept_id
    if( array_key_exists( 'basic', $parser_hash[$parser] )
        && array_key_exists( 'ID', $parser_hash[$parser]['basic'] )
        && !in_array( $parser_hash[$parser]['basic']['ID'],
                      $parser_hash[$parser]['accept_id'] ) ) return;
    
    if( count( $parser_hash[$parser]['handlerstack'] ) > 0 ){
        $handler = end( $parser_hash[$parser]['handlerstack'] );
        //echo "call ${handler[1]}($parser, $name)\n";
        if( function_exists( $handler[1] ) )
            $handler[1]($parser, $name, $parser_hash[$parser] );
        //TODO: compare with first value in 'handler' key.
        if( array_key_exists( "/".implode( '/', $parser_hash[$parser]['tagstack'] ), $parser_hash[$parser]['handler'] ) ){
            array_pop( $parser_hash[$parser]['handlerstack'] );
        }
        //print "/".implode( '/', $parser_hash[$parser]['tagstack'] )."\n";
        //print_r( $parser_hash[$parser]['handlerstack'] );
        array_pop( $parser_hash[$parser]['tagstack'] );
        return;
    }
    
//    echo("<br><b>&lt/$name&gt</b><br><br>"); 
    
    $currentTag = ""; 
    $currentAttribs = ""; 
    
    array_pop( $parser_hash[$parser]['tagstack'] );
} 
    
function characterData($parser, $data) { 
    global $currentTag, $currentAttribs; 
    global $parser_hash;
    
    //return if item_id is not in accept_id
    if( array_key_exists( 'basic', $parser_hash[$parser] )
        && array_key_exists( 'ID', $parser_hash[$parser]['basic'] )
        && !in_array( $parser_hash[$parser]['basic']['ID'],
                      $parser_hash[$parser]['accept_id'] ) ) return;
    
    $tags = "/".implode( '/', $parser_hash[$parser]['tagstack'] );
    
    if( count( $parser_hash[$parser]['handlerstack'] ) > 0 ){
        $handler = end( $parser_hash[$parser]['handlerstack'] );
//        echo "call ${handler[2]}\n";
        if( function_exists( $handler[2] ) )
            $handler[2]($parser, $data, $parser_hash[$parser] );
        return;
    }
//    echo("<font color='#ff0000'>".$data."</font>"); 
} 

function basicStartElement($parser, $name, $attribs, &$parser_hash )
{
//    echo("<b><font color=\"#007777\">&lt$name&gt</font></b><br>"); 
    
    $tags = "/".implode( '/', $parser_hash['tagstack'] );
    switch( $tags ){
    case '/ITEM/BASIC':
        $parser_hash['basic'] = $attribs;
        break;
    case "/ITEM/BASIC/TITLES": 
    case "/ITEM/BASIC/KEYWORDS": 
        $parser_hash['basic'][$name] = array();
        break;
    case "/ITEM/BASIC/ITEMTYPE":
    case "/ITEM/BASIC/CONTRIBUTOR": 
    case "/ITEM/BASIC/DESCRIPTION": 
    case "/ITEM/BASIC/DOI": 
    case "/ITEM/BASIC/LAST_UPDATE_DATE": 
    case "/ITEM/BASIC/CREATION_DATE": 
    case "/ITEM/BASIC/PUBLICATION_YEAR": 
    case "/ITEM/BASIC/PUBLICATION_MONTH": 
    case "/ITEM/BASIC/PUBLICATION_MDAY": 
    case "/ITEM/BASIC/URL": 
    case "/ITEM/BASIC/LANG": 
    case "/ITEM/BASIC/TITLES/TITLE": 
    case "/ITEM/BASIC/KEYWORDS/KEYWORD": 
        $parser_hash['basic'][$name] = '';
        break;
    }
}
function basicEndElement($parser, $name, &$parser_hash)
{
    global $xoopsDB;
    
    $tags = "/".implode( '/', $parser_hash['tagstack'] );
    switch ($tags) { 
    case "/ITEM/BASIC/TITLES/TITLE":
        $parser_hash['basic']['TITLES'][] = $parser_hash['basic']['TITLE'];
        $parser_hash['basic']['TITLE'] = '';
        break;
    case "/ITEM/BASIC/KEYWORDS/KEYWORD": 
        $parser_hash['basic']['KEYWORDS'][] = $parser_hash['basic']['KEYWORD'];
        $parser_hash['basic']['KEYWORD'] = '';
        break;
    case '/ITEM/BASIC':
        if( !in_array( $parser_hash['basic']['ID'], $parser_hash['accept_id'] ) ) break; //
        //print_r( $parser_hash['basic'] );
        $xnpsid = $_SESSION['XNPSID'];
        $uid = $_SESSION['xoopsUserId'];
        $myts =& MyTextsanitizer::getInstance();
        
        $sql = "select item_type_id from " . $xoopsDB->prefix('xoonips_item_type') . " where name='".$myts -> addslashes( $parser_hash['basic']['ITEMTYPE'] )."'";
        $result = $xoopsDB->query( $sql );
        if ( !$result ) break;
        list($item_type_id) = $xoopsDB->fetchRow($result);

        list( $Y, $M, $D, $h, $m, $s ) = preg_split( "/[-:T]/", $parser_hash['basic']['LAST_UPDATE_DATE'] );
        $last_update_date = gmmktime( $h, $m, $s, $M, $D, $Y );
        list( $Y, $M, $D, $h, $m, $s ) = preg_split( "/[-:T]/", $parser_hash['basic']['CREATION_DATE'] );
        $creation_date = gmmktime( $h, $m, $s, $M, $D, $Y );
        
        $item = array(
            'item_type_id'		=> $item_type_id,
            'uid'				=> $uid,
            'titles'			=> encodeClient2Server( $parser_hash['basic']['TITLES'] ),
            'keywords'			=> encodeClient2Server( $parser_hash['basic']['KEYWORDS'] ),
            'description'		=> encodeClient2Server( $parser_hash['basic']['DESCRIPTION'	] ),
            'doi'				=> encodeClient2Server( $parser_hash['basic']['DOI'			] ),
            'last_update_date'  => $last_update_date,
            'creation_date'     => $creation_date,
            'publication_year'	=> $parser_hash['basic']['PUBLICATION_YEAR' ],
            'publication_month'	=> $parser_hash['basic']['PUBLICATION_MONTH'],
            'publication_mday'	=> $parser_hash['basic']['PUBLICATION_MDAY'  ],
            'lang'				=> $parser_hash['basic']['LANG'  ]
            );
        xnpTrimColumn( $item, 'xoonips_item_basic', array( 'item_type_id'		,
                                                           'uid'				,
                                                           'description'		,
                                                           'doi'				,
                                                           'last_update_date'	,
                                                           'creation_date'		,
                                                           'publication_year'	,
                                                           'publication_month'	,
                                                           'publication_mday'	,
                                                           'lang' ),
                       'UTF-8' );

        //trim title
        $lengths = xnpGetColumnLengths( 'xoonips_item_title' );
        foreach( $item['titles'] as $k => $v ){
            list( $item['titles'][$k], $without ) = xnpTrimString( $v, $lengths['title'], 'UTF-8' );
        }
        //trim keyword
        $lengths = xnpGetColumnLengths( 'xoonips_item_keyword' );
        foreach( $item['keywords'] as $k => $v ){
            list( $item['keywords'][$k], $without ) = xnpTrimString( $v, $lengths['keyword'], 'UTF-8' );
        }

        $result = xnp_insert_item_direct( $xnpsid, $item, $item_id );
        if( $result == RES_OK ){
            $parser_hash['id_table'][$parser_hash['basic']['ID']] = $item_id;
            // remove item from private index
            $xids = array();
            if( xnp_get_index_id_by_item_id( $xnpsid, $item_id, $xids ) == RES_OK ){
                foreach( $xids as $i ) xnp_unregister_item( $xnpsid, $i, $item_id );
            }
        }else{
            $parser_hash['errmsg'] .= "can't insert item(xnp_insert_item_direct(item_id=${item_id}) returns ${result})\n";
            xoonips_error( "can't insert item(xnp_insert_item_direct(item_id=${item_id}) returns ${result})\n" );
        }
        break;
    }
//    echo("<br><b><font color=\"#007777\">&lt/$name&gt</font></b><br><br>"); 
}
function basicCharacterData($parser, $data, &$parser_hash )
{
    $tags = "/".implode( '/', $parser_hash['tagstack'] );
    switch( $tags ){
    case "/ITEM/BASIC/ITEMTYPE":
    case "/ITEM/BASIC/CONTRIBUTOR": 
    case "/ITEM/BASIC/DESCRIPTION": 
    case "/ITEM/BASIC/DOI": 
    case "/ITEM/BASIC/LAST_UPDATE_DATE": 
    case "/ITEM/BASIC/CREATION_DATE": 
    case "/ITEM/BASIC/PUBLICATION_YEAR": 
    case "/ITEM/BASIC/PUBLICATION_MONTH": 
    case "/ITEM/BASIC/PUBLICATION_MDAY": 
    case "/ITEM/BASIC/URL": 
    case "/ITEM/BASIC/LANG": 
    case "/ITEM/BASIC/TITLES/TITLE": 
    case "/ITEM/BASIC/KEYWORDS/KEYWORD": 
        $parser_hash['basic'][end($parser_hash['tagstack'])] .= $data;
        break; 
    } 
    
    switch ($tags) { 
    case "/ITEM/BASIC/ITEMTYPE":
        $itemtypes = array();
        $tmp = array();
        if( ( $res = xnp_get_item_types( $tmp ) ) != RES_OK ){
            return;
        }else{
            foreach( $tmp as $i){
                $itemtypes[$i['name']]=$i;
            }
        }
        // TODO array_key_exists( $data, $itemtypes )
        include_once XOOPS_ROOT_PATH . '/modules/' . $itemtypes[$data]['viewphp'];
        $func = $data."GetImportHandler";
        if( function_exists( $func ) ){
            $tmp_handler = $func( );
            $parser_hash['handler'] = array_merge( $tmp_handler, $parser_hash['handler'] );
        }else{
            echo "not exists '$func'\n";
        }
        //print_r( $parser_hash );
        break;
    default: 
//        echo("<font color='#ff0000'>".$data."</font>"); 
        break; 
    } 
}

function linkStartElement($parser, $name, $attribs, &$parser_hash )
{
//    echo("<b><font color=\"#007777\">&lt$name&gt</font></b><br>"); 
    
    $tags = "/".implode( '/', $parser_hash['tagstack'] );
    switch( $tags ){
    case "/ITEM/LINK": 
        $xnpsid = $_SESSION['XNPSID'];
        $index_id = $attribs['INDEX_ID'];
        $item_id = $attribs['ITEM_ID'];
        if( array_key_exists( $index_id, $parser_hash['id_table'] )
            && array_key_exists( $item_id, $parser_hash['id_table'] ) ){
            $result = xnp_register_item( $xnpsid, $parser_hash['id_table'][$index_id], $parser_hash['id_table'][$item_id] );
        }
        break;
    }
}
function linkEndElement($parser, $name, &$parser_hash){}
function linkCharacterData($parser, $data, &$parser_hash ){}


function fileStartElement($parser, $name, $attribs, &$parser_hash )
{
    $tags = "/".implode( '/', $parser_hash['tagstack'] );
    
    switch ($tags) { 
    case '/ITEM/DETAIL/FILE/CAPTION':
        $parser_hash['file']['caption'] = array();
        break; 
    case '/ITEM/DETAIL/FILE/THUMBNAIL':
        $parser_hash['file']['thumbnail'] = array();
        break;
    case '/ITEM/DETAIL/FILE':
        $parser_hash['file'] = $attribs;//array();
        break;
    }
//    echo("<b><font color=\"#ff00ff\">&lt$name&gt</font></b><br>"); 
    //print_r( $parser_hash );
}
function fileEndElement($parser, $name, &$parser_hash)
{
    global $xoopsDB;
    $tags = "/".implode( '/', $parser_hash['tagstack'] );
    
    switch ($tags) { 
    case '/ITEM/DETAIL/FILE':
        if( !array_key_exists( $parser_hash['file']['ITEM_ID'], $parser_hash['id_table'] ) ) break;

        //źեե̵ͭå¸ߤаե˽񤭽Ф
        //
        //եɤ߹fileơ֥ɲä
        //
        //ե(file_type_name=='preview')ĥͥ륿̤ʤ饵ͥ
        // print_r($parser_hash['file']['FILE_NAME']);
        // print_r($parser_hash['entriesmap'][$parser_hash['file']['FILE_NAME']]);
        if( !isset( $parser_hash['entriesmap'][$parser_hash['file']['FILE_NAME']] ) ){
            //<file>黲ȤϤΥե뤬ZIPե̵
            $parser_hash['errmsg'] .= "attachment file '".$parser_hash['file']['FILE_NAME']."' not exists in archive file.\n";
            xoonips_error( "attachment file '".$parser_hash['file']['FILE_NAME']."' not exists in archive file.\n" );
            break;
        }

        $myts =& MyTextSanitizer::getInstance();
        $sql = "select file_type_id from " . $xoopsDB->prefix('xoonips_file_type') . " where name='". addslashes( $parser_hash['file']['FILE_TYPE_NAME'] )."'";
        $result = $xoopsDB->query( $sql );
        if ( !$result ){
            $parser_hash['errmsg'] .= "Invalid query '".$xoopsDB->error()."'\n";
            xoonips_error( "Invalid query '".$xoopsDB->error()."'\n" );
            break;
        }
        list($file_type_id) = $xoopsDB->fetchRow($result);
        
        $ar = array(
            'original_file_name' => encodeClient2Server( $parser_hash['file']['ORIGINAL_FILE_NAME'] ),
            'file_name'          => encodeClient2Server( $parser_hash['file']['FILE_NAME'         ] ),
            'mime_type'          => encodeClient2Server( $parser_hash['file']['MIME_TYPE'         ] ),
        );
        xnpTrimColumn( $ar, 'xoonips_file', array_keys( $ar ), 'UTF-8' );
        
        $xnpsid = $_SESSION['XNPSID'];
        $sql = "insert into " . $xoopsDB->prefix('xoonips_file') . 
            " ( original_file_name, mime_type, file_size, item_id, file_type_id, caption, sess_id ) ".
            " values ( '".implode( "', '", 
                                   array( addslashes( $ar['original_file_name'] ),
                                          addslashes( $ar['mime_type'         ] ),
                                          $parser_hash['file']['FILE_SIZE'],
                                          $parser_hash['id_table'][$parser_hash['file']['ITEM_ID']],
                                          $file_type_id
                                          ) )
            ."', '". addslashes( encodeClient2Server( implode('', $parser_hash['file']['caption'] ) ) )."'"
            .", NULL"
            ." )";
        $result = $xoopsDB->queryF( $sql );
        if( !$result ){
            $parser_hash['errmsg'] .= "Invalid query '".$xoopsDB->error()."'\n";
            xoonips_error( "Invalid query '".$xoopsDB->error()."'\n" );
            break;
        }
        
        //get file_id from file table
        $fileID = $xoopsDB->getInsertId();
        //ե̾(original_file_nameȤʤĥҤĤ)
        $path = pathinfo( $parser_hash['file']['ORIGINAL_FILE_NAME'] );
        $temp = tempnam( "/tmp", "IMG" ).".".$path['extension'];
        //ե򥢡֤Ф̾$temp¸
        unzip_read_data_to_file( $parser_hash['entriesmap'][$parser_hash['file']['FILE_NAME']], $temp );
        //ե륿פ'preview'ʤ顤DBthmubnail_file򹹿
        $thumbnail = 'NULL';//ͥνͤNULL
        if( $parser_hash['file']['FILE_TYPE_NAME']=='preview' ){
            if( !array_key_exists( 'thumbnail', $parser_hash['file'] ) ){
                //<thumbnail>̵мư
                $file = array( 'name' => $parser_hash['file']['ORIGINAL_FILE_NAME'], 'tmp_name' => $temp, 'type' => $parser_hash['file']['MIME_TYPE'] );
                $result = xnpCreateThumbnail( $file );
                if( $result[0] ){
                    //Ǥͥ򥻥å
                    $thumbnail = "0x".bin2hex( $result[0] );
                }else{
                    //ͥ˼
                    $parser_hash['errmsg'] .= $result[1]."\n";
                    xoonips_error( $result[1]."\n" );
                    $parser_hash['errmsg'] .= "can't create a thumbnail of file(".$parser_hash['file']['ORIGINAL_FILE_NAME']." = ${temp}). Insert NULL into thumbnail.\n";
                    xoonips_error( "can't create a thumbnail of file(".$parser_hash['file']['ORIGINAL_FILE_NAME']." = ${temp}). Insert NULL into thumbnail.\n" );
                    $thumbnail = 'NULL';
                }
            }else{
                $thumbnail = "0x".bin2hex(base64_decode( preg_replace( "/[\r\n]/", '', implode( '', $parser_hash['file']['thumbnail'] ) ) ) );
            }
        }
        //ǽŪ¸ե̾
        $filePath = xnpGetUploadFilePath($fileID);
        //$temprenameƺǽŪʤΥե¸ذư
        rename( $temp, $filePath );
        //ͥǡfileơ֥˵Ͽ
        $sql =  "update " . $xoopsDB->prefix('xoonips_file') . " set thumbnail_file=${thumbnail} where file_id=$fileID";
        //echo $sql;
        if ( false == $xoopsDB->queryF( $sql ) ){
            $parser_hash['errmsg'] .= "Invalid query '".$xoopsDB->error()."'\n";
            xoonips_error( "Invalid query '".$xoopsDB->error()."'\n" );
            break;
        }
        //åץɤ줿եʸǥå
        global $search_modules;
        xnpExtractText( $fileID, $search_modules );
/*
        //եʸѥƥȥǥå
        $modules = xnpGetFileSearchModules();
        xnpExtractText( $fileID, $modules );
*/
        break;
    }
    
    $currentTag = ""; 
    $currentAttribs = ""; 
}
function fileCharacterData($parser, $data, &$parser_hash )
{
    $tags = "/".implode( '/', $parser_hash['tagstack'] );
    
    switch ($tags) { 
    case '/ITEM/DETAIL/FILE/CAPTION':
        array_push( $parser_hash['file']['caption'], $data );
        break; 
    case '/ITEM/DETAIL/FILE/DATA':
        array_push( $parser_hash['file']['data'], $data );
        break; 
    case '/ITEM/DETAIL/FILE/THUMBNAIL':
        array_push( $parser_hash['file']['thumbnail'], $data );
        break; 
    } 
}

function indexStartElement($parser, $name, $attribs, &$parser_hash )
{
	$xnpsid = $_SESSION['XNPSID'];

    $tags = "/".implode( '/', $parser_hash['tagstack'] );
    
    switch( $tags ){
    case "/INDEXES/INDEX/TITLE": 
        $parser_hash['indexes'][ count( $parser_hash['indexes'] ) - 1 ]['titles'][] = '';//prepare a new buffer of index title to read
        break;
    case "/INDEXES/INDEX": 
        $ar = array( 'parent_id' => $attribs['PARENT_ID'],
                     'index_id' => $attribs['ID'],
                     'titles' => array() );
        array_push( $parser_hash['indexes'], $ar );
        break;
    case "/INDEXES":
        break;
    }
//    echo("<b><font color=\"#007700\">&lt$name&gt</font></b><br>"); 
}
/**
 * 
 * Importing indexes to a index that is specified by $parent_index_id.
 * Associations of pseudo ID and Real index ID are sotred to $id_table.
 * 
 * @param parent_index_id index_id that indexes is imported to.
 * @param $indexes array of index information to be imported.
 * $indexes = array( 
 *                   array( 'titles' => array( TITLE1, TITLE2, ... )
 *                          'parent_id' => pseudo id of parent index
 *                          'item_id' => pseudo id of own index
 *                          'child' => array( [0] => array( 'titles' => ..., 'parent_id' => ..., 'child' => ....) 
 *                                            [1] => array( same above ),
 *                                            ....
 *                           )
 *                         ), 
 *                   array( 'titles' => array( TITLE1, TITLE2, ... )
 *                          same above ... ),
 *                   ...
 *                   );
 * @param id_table reference of associative array for output( [pseudo id] => [real index id] )
 * @return no return value.
 * 
 */
function import_index( $parent_index_id, &$indexes, &$id_table )
{
	$xnpsid = $_SESSION['XNPSID'];
    $lengths = xnpGetColumnLengths( 'xoonips_item_title' );
    foreach( $indexes as $index ){
        foreach( $index['titles'] as $k => $title ){
            list( $index['titles'][$k], $dummy ) = xnpTrimString( encodeClient2Server( $title ), $lengths['title'], 'UTF-8' );
        }
        $child = array();
		// numbers of same index name
        $cnt = 0;
        $index_id = 0;
        if( xnp_get_indexes( $xnpsid, $parent_index_id, array(), $child ) == RES_OK ){
            foreach( $child as $i ){
                $diff = array_diff( $i['titles'], $index['titles'] );
                if( empty( $diff ) ){
                    //true if $index have only same names of $i ( $i['titles'] == $index['titles'] )
                    $cnt++;
                    $index_id = $i['item_id'];
                }
            }
        }
        if( $cnt == 1 ){
            $id_table[$index['index_id']] = $index_id;
        }else{
            $insert_index = array();
            $insert_index['titles'] = $index['titles'];
            $insert_index['parent_index_id'] = $parent_index_id;
            $result = xnp_insert_index( $xnpsid, $insert_index, $index_id );
            if( $result != RES_OK ){
                break;
            }
            $id_table[$index['index_id']] = $index_id;
        }
        if( array_key_exists( 'child', $index ) )
            import_index( $index_id, $index['child'], $id_table );
    }
}


/**
 *
 * To construct tree structure of the index.
 * The root of the tree structue is specified by $root_id.
 * Return value of this function has information of this index and the childs of this index.
 * This function is called recursive by own.
 * 
 * @param p2c $p2c[ Parent ID of the index ] = array( ID of child index of the index[0], ...[1], ...[2], .... )
 * @param index_by_id assosiative array of index( index information that is associated by own index ID ) 
 *                    $index_by_id[ ID of the index 'A' ] = array( information of the index 'A' )
 * @param root_id Index ID of the root of the index tree structure that you want to get
 * @return assosiative array of index tree
 * 
 */
function constructIndexTree( $p2c, $index_by_id, $root_id )
{
    $ret = $index_by_id[ $root_id ];
    
    if( isset( $p2c[ $root_id ] ) ){//call constructIndexTree if index(id=$root_id) has child indexes
        foreach( $p2c[ $root_id ] as $child_id ){
            $ret['child'][] = constructIndexTree( $p2c, $index_by_id, $child_id );
        }
    }
    
    return $ret;
}

function indexEndElement($parser, $name, &$parser_hash)
{
//    echo("<b><font color=\"#000077\">&lt$name&gt</font></b><br>"); 
	$xnpsid = $_SESSION['XNPSID'];
    
    switch ($name) { 
    case "INDEXES":
        $indexes = $parser_hash['indexes'];

        // To construct tree structure from given indexes by $indexes
        $c2p = array( );//associative array (child ID -> parent ID)
        $p2c = array( );//associative array (parent ID -> array of child ID)
        $index_by_id = array(); // $index_by_id[ index_id ] => index array;
        foreach( $indexes as $i ){
            $c2p[ $i['index_id'] ] = $i['parent_id'];
            if( !isset( $p2c[ $i['parent_id'] ] ) ) $p2c[ $i['parent_id'] ] = array();
            $p2c[ $i['parent_id'] ][] = $i['index_id'];
            $index_by_id[ $i['index_id'] ] = $i;
        }

        $root_ids = array();//Index id of root of each index trees
        while( list( $child, $parent ) = each( $c2p ) ){
            while( array_key_exists( $parent, $c2p ) ){//track back to root
                unset( $c2p[$child] );
                $child = $parent;
                $parent = $c2p[$parent];
            }
        
            $parent = $child;

            $root_ids[] = $parent;
            unset( $c2p[ $parent ] );
        
            //remove all childs of the root($parent)
            if( isset( $p2c[ $parent ] ) ){
                $stack = $p2c[ $parent ];
                while( count( $stack ) > 0 ){
                    $id = array_pop( $stack );
                    unset( $c2p[ $id ] );
                    if( isset( $p2c[ $id ] ) ) $stack = array_merge( $stack, $p2c[ $id ] );
                }
            }
            reset( $c2p );
        }
        $tree = array();//structured index tree
        foreach( $root_ids as $root_id ){
            $tree[] = constructIndexTree( $p2c, $index_by_id, $root_id );
        }
        
        import_index( $parser_hash['parent_index_id'], $tree, $parser_hash['id_table'] );
        break;
    }
//    echo("<br><b><font color=\"#007700\">&lt/$name&gt</font></b><br><br>"); 
}
function indexCharacterData($parser, $data, &$parser_hash )
{
    $tags = "/".implode( '/', $parser_hash['tagstack'] );
    
    switch( $tags ){
    case "/INDEXES/INDEX/TITLE": 
        $index = array_pop( $parser_hash['indexes'] );
        $index['titles'][count($index['titles'])-1] .= $data;
        array_push( $parser_hash['indexes'], $index );
        break;
    }
//    echo $data."\n";
}

/**
 * 
 * import basic information that retrieved from XML.
 * store pseudo id and actual id(where each pseudo id is associated with actual id).
 * if a pseudo id is not appeared in $accept_id, item associated with the pseudo id is not imported.
 * 
 * @see unzip_all_entries
 * 
 * @param str XML charactor(UTF-8)
 * @param id_table array( 'pseudo ID' => 'actual ID', ... )
 * @param accept_id array( 'pseudo id of import item', ... )
 * @param entries result of unzip_all_entries
 * @param errmsg reference recieve error message
 * @return true success
 * @return false error. refer $errmsg.
 */
function xnpImportItem( $str, &$id_table, $accept_id = array(), $entries, &$errmsg )
{
    global $parser_hash;

    $errmsg = '';
	$xnpsid = $_SESSION['XNPSID'];
    $item = array();
    
	$str = xnpEntity2Utf8( $str );
    $parser = xml_parser_create( "UTF-8" );
    if( !$parser ){
        $errmsg .= "can't create parser\n";
        return false;
    }
    
    //$entriesmap(key=filename, vaule=information of compressed file(associated with filename) in a zip archive)
    $entriesmap = array();
    foreach( $entries as $ent ) $entriesmap[$ent['filename']] = $ent;
    $parser_hash[$parser] = array( 'tagstack' => array(),
                                   'id_table'  => $id_table,
                                   'accept_id' => $accept_id,
                                   'errmsg'  => null,
                                   'entries' => $entries,
                                   'entriesmap' => $entriesmap,
                                   'handler'  => array(),
                                   'handlerstack'  => array() );
    //XooNIps processes following tags.
    $parser_hash[$parser]['handler']['/ITEM/BASIC'] = array( "basicStartElement", "basicEndElement", "basicCharacterData" );
    $parser_hash[$parser]['handler']['/ITEM/DETAIL/FILE'] = array( "fileStartElement", "fileEndElement", "fileCharacterData" );
    
    xml_set_element_handler($parser, "startElement", "endElement"); 
    xml_set_character_data_handler($parser, "characterData"); 
    
    if( !xml_parse( $parser, $str, true ) ){
		$lines = preg_split( "/[\r\n]+/", $str );
        //die( xml_error_string( xml_get_error_code( $parser ) )." at column ".xml_get_current_column_number ( $parser )." of line ".htmlspecialchars($lines[xml_get_current_line_number( $parser )-1]) );
        $parser_hash[$parser]['errmsg'] .= xml_error_string( xml_get_error_code( $parser ) )." at column ".xml_get_current_column_number ( $parser )." of line ".htmlspecialchars($lines[xml_get_current_line_number( $parser )-1]);
        $errmsg = $parser_hash[$parser]['errmsg'];
        return false;
    }

    xml_parser_free( $parser );
    
    $id_table = $parser_hash[$parser]['id_table'];
    $parser_hash[$parser] = NULL;
    return true;
}

/**
 * 
 * import 'Related to' data of item from xml.
 * 
 * @param str XML characters (UTF-8)
 * @param id_table array( 'pseudo ID' => 'actual ID', ... )
 * @param accept_id array( 'pseudo id of import item', ... )
 * @param errmsg reference recieve error message
 * @return true success
 * @return false error. refer $errmsg.
 */
function xnpImportRelatedTo( $str, &$id_table, $accept_id = array(), &$errmsg )
{
	global $parser_hash;

	$errmsg = '';
	$xnpsid = $_SESSION['XNPSID'];
	$item = array();
	
	$str = xnpEntity2Utf8( $str );
	$parser = xml_parser_create( "UTF-8" );
	if( !$parser ){
		$errmsg .= "can't create parser\n";
		return false;
	}
	$parser_hash[$parser] = array( 'tagstack' => array(),
								   'id_table'  => $id_table,
								   'accept_id' => $accept_id,
								   'errmsg'  => $errmsg,
								   'handler'  => array(),
								   'handlerstack'  => array() );
//	print_r($parser_hash[$parser]);
	//XooNIps processes following tags.
	$parser_hash[$parser]['handler']['/ITEM/BASIC'] = array( "relatedToStartElement", "relatedToEndElement", "relatedToCharacterData" );
	
	xml_set_element_handler($parser, "startElement", "endElement"); 
	xml_set_character_data_handler($parser, "characterData"); 
	
	if( !xml_parse( $parser, $str, true ) ){
		$lines = preg_split( "/[\r\n]+/", $str );
		die( xml_error_string( xml_get_error_code( $parser ) )." at column ".xml_get_current_column_number ( $parser )." of line ".htmlspecialchars($lines[xml_get_current_line_number( $parser )-1]) );
	}

	xml_parser_free( $parser );
	
	$id_table = $parser_hash[$parser]['id_table'];
	$parser_hash[$parser] = NULL;
	return true;
}

/**
 * 
 * 
 * @param str XML characters (UTF-8)
 * @param parent_id index_id of import place
 * @return array( 'pseudo ID' => 'actual ID', ... )
 * @param errmsg reference recieve error message
 * @return true success
 * @return false error. refer $errmsg.
 */
function xnpImportIndex( $str, $parent_index_id, &$id_table, &$errmsg )
{
    global $parser_hash;
    
    $uid = $_SESSION['xoopsUserId'];
	$xnpsid = $_SESSION['XNPSID'];
    $item = array();
    
	$str = xnpEntity2Utf8( $str );
    $parser = xml_parser_create( "UTF-8" );
    if( !$parser ){
        $errmsg .= "can't create parser\n";
        return false;
    }
    $parser_hash[$parser] = array( 'tagstack' => array(),
                                   'id_table'  => $id_table,
                                   'errmsg'  => $errmsg,
                                   'handler'  => array(),
                                   'handlerstack'  => array(),
                                   'indexes' => array(),
                                   'parent_index_id' => $parent_index_id );
    
    $parser_hash[$parser]['handler']['/INDEXES'] = array( "indexStartElement", "indexEndElement", "indexCharacterData" );
    
    xml_set_element_handler($parser, "startElement", "endElement"); 
    xml_set_character_data_handler($parser, "characterData"); 
    
    if( !xml_parse( $parser, $str, true ) ){
		$lines = preg_split( "/[\r\n]+/", $str );
        die( xml_error_string( xml_get_error_code( $parser ) )." at column ".xml_get_current_column_number ( $parser )." of line ".htmlspecialchars($lines[xml_get_current_line_number( $parser )-1]) );
    }

    xml_parser_free( $parser );
    
    $id_table = $parser_hash[$parser]['id_table'];
    $parser_hash[$parser] = NULL;
    return true;
}


/**
 * 
 * import <link> of items's Basic Information from xml.
 * 
 * @param str XML characters (UTF-8)
 * @param reference of idtable array( 'pseudo ID' => 'actual ID', ... )
 * @param errmsg reference recieve error message
 * @return true success
 * @return false error refer $errmsg.
 */
function xnpImportItemLink( $str, &$idtable, &$errmsg )
{
    global $parser_hash;
    
	$xnpsid = $_SESSION['XNPSID'];
    $item = array();
    
	$str = xnpEntity2Utf8( $str );
    $parser = xml_parser_create( "UTF-8" );
    if( !$parser ){
        $errmsg .= "can't create parser\n";
        return false;
    }
    $parser_hash[$parser] = array( 'tagstack' => array(),
                                   'id_table'  => $idtable,
                                   'handler'  => array(),
                                   'handlerstack'  => array() );
//    print_r($parser_hash[$parser]);
    //XooNIps processes following tags.
    $parser_hash[$parser]['handler']['/ITEM/LINK'] = array( "linkStartElement", "linkEndElement", "linkCharacterData" );
    
    xml_set_element_handler($parser, "startElement", "endElement"); 
    xml_set_character_data_handler($parser, "characterData"); 
    
    if( !xml_parse( $parser, $str, true ) ){
		$lines = preg_split( "/[\r\n]+/", $str );
        die( xml_error_string( xml_get_error_code( $parser ) )." at column ".xml_get_current_column_number ( $parser )." of line ".htmlspecialchars($lines[xml_get_current_line_number( $parser )-1]) );
    }

    xml_parser_free( $parser );
    
    $idtable = $parser_hash[$parser]['id_table'];
    $parser_hash[$parser] = NULL;
    return true;
}


/**
 * 
 * @param fhdl file handle that indexes are exported to.
 * @param index_id id of index to display
 * @param recurse true:export recursively index that hangs under index_id.
 * @return true:success, false:failure
 */
function xnpExportIndex( $fhdl, $index_id, $recurse )
{
    if( !$fhdl ) return false;

	$xnpsid = $_SESSION['XNPSID'];
    $index = array();
    $child = array();
    
    $res = xnp_get_index( $xnpsid, $index_id, $index );
    if( $res != RES_OK ){
        return false;
    }
    
    $res = xnp_get_indexes( $xnpsid, $index_id, array(), $child );
    if( $res != RES_OK ){
        return false;
    }
    
    if( !fwrite( $fhdl, "<index parent_id=\"${index['parent_index_id']}\" id=\"${index_id}\">\n"
                 ."<title>".xnpHtmlspecialchars($index['titles'][DEFAULT_INDEX_TITLE_OFFSET], ENT_QUOTES)."</title>\n"
                 ."</index>\n" ) ) return false;
    
    $xml = array();
    if( $recurse ){
        $res = xnp_get_indexes( $xnpsid, $index_id, array( 'orders' => array( array( 'sort_number', 0 ) ) ), $child );
        if( $res == RES_OK ){
            foreach( $child as $i ){
                if( !xnpExportIndex( $fhdl, $i['item_id'], $recurse ) ) return false;
            }
        }
    }
    return true;
}

/**
 * 
 * exporting attachment files related to item.
 * 
 * @param fhdl file handle that items are exported to.
 * @param item_id id of item with attachment files to export.
 * @param export_path Exportե¸ե
 * @return array( 'path' => $export_path,
 *                   'attachments' => array( file path of attachment1, file path of attachment2, ... ) )
 *            (relative path of $export_filepath)
 *         returns false if it failed
 */
function xnpExportFile( $export_path, $fhdl, $item_id )
{
    $file = xnpGetFileInfo( "t_file.file_id, t_file_type.name, t_file.original_file_name, t_file.file_size, t_file.mime_type, t_file.thumbnail_file, t_file.caption", 
                            "item_id = ${item_id}", $item_id );
    
    if( !$fhdl ) return false;
    
    // files/եexport_pathľ˺
    $dir = $export_path."/files";
    if( !file_exists( $dir ) ){
        if( !mkdir( $dir ) ){
            xoonips_error( "can't make directory '${dir}'" );
            return false;
        }
    }
    
    $files = array(); //źեեХѥ¸
    foreach( $file as $f ){
        $file = array();
        list( $file['file_id'], $file['file_type_name'], $file['original_file_name'], $file['file_size'], $file['mime_type'], $file['thumbnail_file'], $file['caption'] ) = $f;
        
        // źեեμΤ${dir}${file['file_id']}ե̾ˤƥԡ
        // <file>$fhdl˽񤭽Ф
        $hdl = fopen( xnpGetUploadFilePath( $file['file_id'] ), "rb" );
        if( file_exists( xnpGetUploadFilePath( $file['file_id'] ) ) ){
            if( !copy( xnpGetUploadFilePath( $file['file_id'] ), $dir."/".$file['file_id'] ) ){
                xoonips_error( "can't write a file '".$dir."/".$file['file_id']."' of the item(ID=${item_id})" );
                return false;
            }
            if( !fwrite( $fhdl, "<file"
                         ." item_id=\"${item_id}\""
                         ." file_type_name=\"${file['file_type_name']}\""
                         ." original_file_name=\"${file['original_file_name']}\""
                         ." file_name=\"files/${file['file_id']}\""
                         ." file_size=\"${file['file_size']}\""
                         ." mime_type=\"${file['mime_type']}\""
                         .">\n"
                         .( isset( $file['thumbnail_file'] ) ? "<thumbnail>".base64_encode( $file['thumbnail_file'] )."</thumbnail>\n" : '' )
                         ."<caption>".$file['caption']."</caption>\n"
                         ."</file>\n" ) ){
                fclose( $hdl );
                xoonips_error( "can't export <file> of the item(ID=${item_id})" );
                return false;
            }
            $files[] = "files/${file['file_id']}";
        }
    }
    return true;

//    return array( 'path' => $export_path,
//                  'attachments' => $files );
}

/**
 * 
 * export Basic information of item
 * 
 * @param fhdl file handle that items are exported to.
 * @param item_id id of the item to change into XML.
 * @return true:success, false:failure
 */
function xnpExportBasic( $fhdl, $item_id, $is_absolute, $base_index_id = false )
{
    if( !$fhdl ) return false;
    
	$xnpsid = $_SESSION['XNPSID'];
    $item = array();
    $account = array();

    $res = xnp_get_item( $xnpsid, $item_id, $item );
    if( $res != RES_OK ){
        return false;
    }
    return xnpBasicInformation2XML( $fhdl, $item, $is_absolute, $base_index_id );
}

/**
 * 
 * export ChangeLog of item.
 * 
 * @param fhdl file handle that changelogs are exported to.
 * @param item id of the item to export.
 * @return true:success, false:failure
 */
function xnpExportChangeLog( $fhdl, $item_id )
{
    if( !$fhdl ) return false;
    
	$xnpsid = $_SESSION['XNPSID'];
    $xml = array();
    $logs = array();
    $res = xnp_get_change_logs( $xnpsid, $item_id, $logs );
    if( $res != RES_OK ) return false;
    if( !fwrite( $fhdl, "<changelogs>\n" ) ) return false;
    foreach( $logs as $l ){
        $log_date    = gmdate( 'Y-m-d\TH:i:s\Z', $l['log_date'] );
        if( !fwrite( $fhdl, "<changelog date='${log_date}'>".xnpHtmlspecialchars( $l['log'] )."</changelog>\n" ) ) return false;
    }
    if( !fwrite( $fhdl, "</changelogs>\n" ) ) return false;
    
    return true;
}


/**
 * 
 * export 'Related to' information of an item.
 * 
 * @param parent_id id of the item to export.
 * @return generated XML or NULL
 */
function xnpExportRelatedTo( $parent_id )
{
	$xnpsid = $_SESSION['XNPSID'];
	$xml = array();
	$item_id = array();
	//Export item: only accesible item
	//->export link information getted xnp_get_related_to.
	$res = xnp_get_related_to( $xnpsid, $parent_id, $item_id );
	if( $res != RES_OK ) return NULL;
	$xml = '';
	foreach( $item_id as $i ){
		$xml = $xml . "<related_to item_id='${i}'/>\n";
	}
	return $xml;
}




function titleStartElement($parser, $name, $attribs, &$parser_hash )
{
    $tags = "/".implode( '/', $parser_hash['tagstack'] );
    
    switch( $tags ){
    case "/ITEM/BASIC":
        $parser_hash['basic']['item_id'] = $attribs['ID'];
        break;
    case "/ITEM/BASIC/TITLES": 
        $parser_hash['basic']['titles'] = array();
        break;
    case "/ITEM/BASIC/TITLES/TITLE": 
        $parser_hash['basic']['titles'][] = ''; //<title></title>Ƥ¸ΰ
        break;
    }
}

function titleEndElement($parser, $name, &$parser_hash)
{
    $tags = "/".implode( '/', $parser_hash['tagstack'] );
    
    switch( $tags ){
    case "/ITEM/BASIC":
        $xnpsid = $_SESSION['XNPSID'];
        $iids = array( );
        $msg = '';
        $search_var = array();
        $dummy = false;
        $cond = array();
        // trim search keywords(=titles)
        foreach( $parser_hash['basic']['titles'] as $data ){
            list( $within, $without ) = xnpTrimString( encodeClient2Server( $data ), $parser_hash['max_title_length'], 'UTF-8' );
            if( !empty( $within ) ){//ignore the $within if it is empty
                // convert each titles to terms for query
                // 1. convert double-quote to white-space
                // 2. split the string by white-spcace
                // 3. each splitted strings are quoted by double-quote
                $repl = mb_ereg_replace( "\"", " ", $within );
                if( $repl ){
                    $cond[] = "\"" . implode( "\" \"", explode( " ", $repl ) ) . "\"";
                }
            }
        }
        //search items which have same titles
        if( xnpSearchExec( 'quicksearch', implode( ' OR ', $cond ), 'basic', false, $msg, $iids, $search_var, $dummy, $dummy ) ){
            $ret = array();
            foreach( $iids as $item_id ){
                $item = array();
                if( xnp_get_item( $xnpsid, $item_id, $item ) == RES_OK ){
                    $item_titles = array( );
                    // Here, using encodeClient2Server to convert special chars(like a latin-1) to numeric char reference.
                    // because these special chars are treated as numeric char references in a DB.
                    // ex: e; -> &#275; 
                    foreach( $parser_hash['basic']['titles'] as $t ){ $item_titles[] = encodeClient2Server( $t ); }
                    $diff = array_diff( $item_titles, $item['titles'] );

                    // true if $item_titles have only same titles of $item['titles'].
                    if( empty( $diff ) ) $ret[] = $item_id;
                }
            }
            $parser_hash['ids'] = $ret;
        }
        break;
    }
}

function titleCharacterData($parser, $data, &$parser_hash )
{
    $tags = "/".implode( '/', $parser_hash['tagstack'] );
    
    switch( $tags ){
    case "/ITEM/BASIC/ITEMTYPE":
        $parser_hash['basic']['itemtype'] = $data;
        break;
        
    case "/ITEM/BASIC/CONTRIBUTOR":
        $parser_hash['basic']['contributor'] = $data;
        break;
        
    case "/ITEM/BASIC/TITLES/TITLE":
        $parser_hash['basic']['titles'][count($parser_hash['basic']['titles'])-1] .= $data;
        break;
    } 
}
/**
 * 
 * check the existance of same name item in index to import, 
 * and return its results.
 * 
 * @param str XML characters (UTF-8)
 * @param basic Basic Information in XML
 * @param ids id of item with possibility of collision
 * @return 
 */
function xnpImportItemCheck( $str, &$basic, &$ids )
{
    global $parser_hash;
    
    if( !is_array( $basic ) ) $basic = array( );
    if( !is_array( $ids ) ) $ids = array( );
    
	$xnpsid = $_SESSION['XNPSID'];
    $item = array();
    
	$str = xnpEntity2Utf8( $str );
    $parser = xml_parser_create( "UTF-8" );
    if( !$parser ) return NULL;
    $parser_hash[$parser] = array(
        'tagstack' => array(),
        'basic' => array(),
        'ids'  => array(),
        'handler'  => array(),
        'handlerstack'  => array() );
    $parser_hash[$parser]['handler']['/ITEM/BASIC'] = array( "titleStartElement", "titleEndElement", "titleCharacterData" );
    
    $lengths = xnpGetColumnLengths( 'xoonips_item_title' );
    $parser_hash[$parser]['max_title_length'] = $lengths['title'];
    
    xml_set_element_handler($parser, "startElement", "endElement"); 
    xml_set_character_data_handler($parser, "characterData"); 
    
    if( !xml_parse( $parser, $str, true ) ){
		$lines = preg_split( "/[\r\n]+/", $str );
        die( xml_error_string( xml_get_error_code( $parser ) )." at column ".xml_get_current_column_number ( $parser )." of line ".htmlspecialchars($lines[xml_get_current_line_number( $parser )-1]) );
    }

    xml_parser_free( $parser );
    
    $ids = $parser_hash[$parser]['ids'];
    $basic = $parser_hash[$parser]['basic'];
    $parser_hash[$parser] = NULL;
    return true;
}


function indexcheckStartElement($parser, $name, $attribs, &$parser_hash )
{
//    echo("<b><font color=\"#007777\">&lt$name&gt</font></b><br>"); 
    $tags = "/".implode( '/', $parser_hash['tagstack'] );
    
    switch( $tags ){
    case "/INDEXES": 
        $parser_hash['indexlinks'] = array();
        break;
    case "/INDEXES/INDEX": 
        $ar = array( 'parent_id' => $attribs['PARENT_ID'],
                     'index_id' => $attribs['ID'],
                     'titles' => array() );
        array_push( $parser_hash['indexes'], $ar );
        break;
    case "/INDEXES/INDEX/TITLE": 
        $parser_hash['indexes'][ count( $parser_hash['indexes'] ) - 1 ]['titles'][] = '';
        break;
    }
}
function indexcheckEndElement($parser, $name, &$parser_hash){
    $tags = "/".implode( '/', $parser_hash['tagstack'] );
    
    switch( $tags ){
    case "/INDEXES/INDEX/TITLE":
        $title =& $parser_hash['indexes'][ count( $parser_hash['indexes'] ) - 1 ]['titles'][ count( $parser_hash['indexes'][ count( $parser_hash['indexes'] ) - 1 ]['titles'] ) - 1 ];
        $title = encodeClient2Server( $title );
        break;
    }
}
function indexcheckCharacterData($parser, $data, &$parser_hash )
{
    $tags = "/".implode( '/', $parser_hash['tagstack'] );
    
    switch( $tags ){
    case "/INDEXES/INDEX/TITLE":
        $parser_hash['indexes'][ count( $parser_hash['indexes'] ) - 1 ]['titles'][ count( $parser_hash['indexes'][ count( $parser_hash['indexes'] ) - 1 ]['titles'] ) - 1 ] .= $data;
        break;
    } 
}
function xnpImportIndexCheck( $str,&$indexes )
{
     global $parser_hash;

     $xnpsid = $_SESSION['XNPSID'];
     $item = array();

	 $str = xnpEntity2Utf8( $str );
     $parser = xml_parser_create( "UTF-8" );
     if( !$parser ) return NULL;
     $parser_hash[$parser] = array(
         'tagstack' => array(),
         'indexes' => array(),
         'handler'  => array(),
         'handlerstack'  => array() );
     //XooNIps processes following tags.
     $parser_hash[$parser]['handler']['/INDEXES'] = array( "indexcheckStartElement", "indexcheckEndElement", "indexcheckCharacterData" );
    
    xml_set_element_handler($parser, "startElement", "endElement"); 
    xml_set_character_data_handler($parser, "characterData"); 
    
    if( !xml_parse( $parser, $str, true ) ){
		$lines = preg_split( "/[\r\n]+/", $str );
        die( xml_error_string( xml_get_error_code( $parser ) )." at column ".xml_get_current_column_number ( $parser )." of line ".htmlspecialchars($lines[xml_get_current_line_number( $parser )-1]) );
    }

    xml_parser_free( $parser );
    
    $indexes = $parser_hash[$parser]['indexes'];
    $parser_hash[$parser] = NULL;
    return true;
}


function relatedToStartElement($parser, $name, $attribs, &$parser_hash )
{
	$xnpsid = $_SESSION['XNPSID'];
	$tags = "/".implode( '/', $parser_hash['tagstack'] );
	switch( $tags ){
	case '/ITEM/BASIC':
		$parser_hash['basic'] = $attribs;
		$parser_hash['related_to'] = array();
		$parser_hash['related_to']['parent_id'] = $attribs['ID']; // id of a original link
		$parser_hash['related_to']['child_id'] = 0; // id of a ahead link
		break;
	case '/ITEM/BASIC/RELATED_TO':
		if( isset( $attribs['ITEM_ID'] ) ) $parser_hash['related_to']['child_id'] = $attribs['ITEM_ID'];
		if( $parser_hash['related_to']['child_id'] != 0 ){
			//insert link among items into id_table,
			//if there is correspondence from pseudo id to the actual id in both original link and ahead link.
			if( array_key_exists( $parser_hash['related_to']['parent_id'], $parser_hash['id_table'] )
				&& array_key_exists( $parser_hash['related_to']['child_id'], $parser_hash['id_table'] ) ){
				$parent_id = $parser_hash['id_table'][ $parser_hash['related_to']['parent_id'] ];
				$child_id = $parser_hash['id_table'][ $parser_hash['related_to']['child_id'] ];
//  				echo "\nxnp_insert_related_to( $xnpsid, $parent_id, $child_id )=";
//  				echo xnp_insert_related_to( $xnpsid, $parent_id, $child_id );
			}
		}
		break;
	}
}
function relatedToEndElement($parser, $name, &$parser_hash)
{
}
function relatedToCharacterData($parser, $data, &$parser_hash )
{
}



/**
 * 
 * ƥˡDBΥॵۤ褦ʥǡФ֤
 * 
 * @param str XMLʸ(UTF-8)
 * @param exceeds_info ƥtitles, Ķ᤹륫̾Ȥ 
 *  array of array( 'titles' => array( title[0], title[1], .... ),
                    'exceeds' => array( array( 'column' => name of column of table,
                                               'within' => substring that is imported,
                                               'without' => substring that is pruned off,
                                               'name' => display name that corresponds to name of column of table,
                                               'value' => xnpWithinWithoutHtml( $within, $without ) ),
                                        array( same above ... ),
                                        ...
                                      )
                  )
 * @return 
 */
function xnpImportCheckItemLength( $str, $accept_id = array(), &$exceeds_info )
{
    global $parser_hash;
    
	$xnpsid = $_SESSION['XNPSID'];
    $item = array();
    
	$str = xnpEntity2Utf8( $str );
    $parser = xml_parser_create( "UTF-8" );
    if( !$parser ) return NULL;
    $parser_hash[$parser] = array(
        'tagstack' => array(),
        'basic' => array(),
        'accept_id' => $accept_id,
        'exceeds' => array(),
        'handler'  => array(),
        'handlerstack'  => array() );
    $parser_hash[$parser]['handler']['/ITEM/BASIC'] = array( "checkitemlengthStartElement", "checkitemlengthEndElement", "checkitemlengthCharacterData" );
    $parser_hash[$parser]['handler']['/ITEM/DETAIL/FILE'] = array( "fileStartElement", "checkfilelengthEndElement", "fileCharacterData" );
    
    xml_set_element_handler($parser, "startElement", "endElement"); 
    xml_set_character_data_handler($parser, "characterData"); 
    
    if( !xml_parse( $parser, $str, true ) ){
		$lines = preg_split( "/[\r\n]+/", $str );
        die( xml_error_string( xml_get_error_code( $parser ) )." at column ".xml_get_current_column_number ( $parser )." of line ".htmlspecialchars($lines[xml_get_current_line_number( $parser )-1]) );
    }

    xml_parser_free( $parser );
    
    if ( !empty( $parser_hash[$parser]['exceeds'] ) ){
        $titles = array();
        foreach( $parser_hash[$parser]['basic']['titles'] as $title ){
            $titles[] = encodeClient2Server( $title );
        }
        $exceeds_info = array(
            'titles' => $titles,
            'exceeds' => $parser_hash[$parser]['exceeds']
        );
    }
    else
        $exceeds_info = false;
    
    $parser_hash[$parser] = NULL;
    return true;
}

function checkitemlengthStartElement($parser,
 $name, $attribs, &$parser_hash )
{
    
    $tags = "/".implode( '/', $parser_hash['tagstack'] );
    switch( $tags ){
    case '/ITEM/BASIC':
        $parser_hash['basic'] = $attribs;
        break;
    case "/ITEM/BASIC/TITLES": 
        $parser_hash['basic']['titles'] = array();
        break;
    case "/ITEM/BASIC/TITLES/TITLE": 
        $parser_hash['basic']['titles'][] = '';
        break;
    case "/ITEM/BASIC/KEYWORDS": 
        $parser_hash['basic']['keywords'] = array();
        break;
    case "/ITEM/BASIC/KEYWORDS/KEYWORD": 
        $parser_hash['basic']['keywords'][] = '';
        break;
    case "/ITEM/BASIC/DESCRIPTION": 
    case "/ITEM/BASIC/DOI": 
    case "/ITEM/BASIC/LANG": 
        $parser_hash['basic'][$name] = '';
        break;
    }
}

function checkitemlengthEndElement($parser, $name, &$parser_hash)
{
    global $xoopsDB;
    
    $tags = "/".implode( '/', $parser_hash['tagstack'] );
    switch ($tags) { 
    case '/ITEM/BASIC':
        $xnpsid = $_SESSION['XNPSID'];
        $uid = $_SESSION['xoopsUserId'];
        $myts =& MyTextsanitizer::getInstance();
        
        $values = array(
            'description' => encodeClient2Server( $parser_hash['basic']['DESCRIPTION' ] ),
            'doi'         => encodeClient2Server( $parser_hash['basic']['DOI'         ] ),
            'lang'        => encodeClient2Server( $parser_hash['basic']['LANG'        ] ),
        );
        $names = array(
            'description' => _MD_XOONIPS_ITEM_DESCRIPTION_LABEL,
            'doi'         => _MD_XOONIPS_ITEM_DOI_LABEL        ,
            'lang'        => _MD_XOONIPS_ITEM_LANG_LABEL       ,
        );
        
        //test length of string of item_basic
        $lengths = xnpGetColumnLengths( 'xoonips_item_basic' );
        foreach ( $lengths as $key => $maxlen ){
            if ( isset( $values[ $key ] ) && strlen($values[$key]) > $maxlen ){
                list( $within, $without ) = xnpTrimString( $values[$key], $maxlen, 'UTF-8' );
                $parser_hash['exceeds'][$key] = array( 'column' => $key, 'within' => $within, 'without' => $without, 'name' => $names[$key], 'value' => xnpWithinWithoutHtml( $within, $without ) );
            }
        }

        //test length of string of item_title
        $lengths = xnpGetColumnLengths( 'xoonips_item_title' );
        $key = 'title';// see also array '$names'
        $maxlen = $lengths['title'];
        foreach ( $parser_hash['basic']['titles'] as $title ){
            if ( strlen($title) > $maxlen ){
                list( $within, $without ) = xnpTrimString( $title, $maxlen, 'UTF-8' );
                $parser_hash['exceeds']['title'] = array( 'column' => $key, 'within' => $within, 'without' => $without, 'name' => _MD_XOONIPS_ITEM_TITLE_LABEL, 'value' => xnpWithinWithoutHtml( $within, $without ) );
            }
        }

        //test length of string of item_keyword
        $lengths = xnpGetColumnLengths( 'xoonips_item_keyword' );
        $key = 'keyword';// see also array '$names'
        $maxlen = $lengths['keyword'];
        foreach ( $parser_hash['basic']['keywords'] as $keyword ){
            if ( strlen($keyword) > $maxlen ){
                list( $within, $without ) = xnpTrimString( $keyword, $maxlen, 'UTF-8' );
                $parser_hash['exceeds']['keyword'] = array( 'column' => $key, 'within' => $within, 'without' => $without, 'name' => _MD_XOONIPS_ITEM_KEYWORDS_LABEL, 'value' => xnpWithinWithoutHtml( $within, $without ) );
            }
        }
        break;
    }
}
function checkitemlengthCharacterData($parser, $data, &$parser_hash )
{
    $tags = "/".implode( '/', $parser_hash['tagstack'] );
    
    switch( $tags ){
    case "/ITEM/BASIC/TITLES/TITLE": 
        $parser_hash['basic']['titles'][ count( $parser_hash['basic']['titles'] ) - 1 ] .= $data;
        break;
    case "/ITEM/BASIC/KEYWORDS/KEYWORD": 
        $parser_hash['basic']['keywords'][ count( $parser_hash['basic']['keywords'] ) - 1 ] .= $data;
        break;
    case "/ITEM/BASIC/ITEMTYPE":
    case "/ITEM/BASIC/KEYWORDS/KEYWORD": 
    case "/ITEM/BASIC/DESCRIPTION": 
    case "/ITEM/BASIC/DOI": 
    case "/ITEM/BASIC/LANG": 
        if( !array_key_exists( end($parser_hash['tagstack']), $parser_hash['basic'] ) ){
            $parser_hash['basic'][end($parser_hash['tagstack'])] = $data;
        }else{
            $parser_hash['basic'][end($parser_hash['tagstack'])] .= $data;
        }
        break; 
    } 
    
    switch ($tags) { 
    case "/ITEM/BASIC/ITEMTYPE":
        $itemtypes = array();
        $tmp = array();
        if( ( $res = xnp_get_item_types( $tmp ) ) != RES_OK ){
            return;
        }else{
            foreach( $tmp as $i){
                $itemtypes[$i['name']]=$i;
            }
        }
        include_once XOOPS_ROOT_PATH . '/modules/' . $itemtypes[$data]['viewphp'];
        $func = $data."GetCheckItemLengthHandler";
        if( function_exists( $func ) ){
            $tmp_handler = $func( );
            $parser_hash['handler'] = array_merge( $tmp_handler, $parser_hash['handler'] );
        }else{
            echo "not exists '$func'\n";
        }
        //print_r( $parser_hash );
        break;
    default: 
        break; 
    } 
}
function checkfilelengthEndElement($parser, $name, &$parser_hash)
{
    global $xoopsDB;
    $tags = "/".implode( '/', $parser_hash['tagstack'] );
    
    switch ($tags) { 
    case '/ITEM/DETAIL/FILE':
        
        if( !in_array( $parser_hash['file']['ITEM_ID'], $parser_hash['accept_id'] ) ) break;
        
        $values = array(
            'original_file_name' => encodeClient2Server( $parser_hash['file']['ORIGINAL_FILE_NAME'] ),
            'mime_type'          => encodeClient2Server( $parser_hash['file']['MIME_TYPE'         ] ),
            'caption'            => encodeClient2Server( implode( '', $parser_hash['file']['caption'           ]) ),
        );
        $names = array(
            'original_file_name' => _MD_XOONIPS_ITEM_ORIGINAL_FILE_NAME_LABEL,
            'mime_type'          => _MD_XOONIPS_ITEM_MIME_TYPE_LABEL,
            'caption'            => _MD_XOONIPS_ITEM_CAPTION_LABEL,
        );
        
        $lengths = xnpGetColumnLengths( 'xoonips_file' );
        foreach ( $lengths as $key => $maxlen ){
            if ( isset($values[$key]) && strlen($values[$key]) > $maxlen ){
                list( $within, $without ) = xnpTrimString( $values[$key], $maxlen, 'UTF-8' );
                $parser_hash['exceeds'][$key] = array( 'within' => $within, 'without' => $without, 'name' => $names[$key], 'value' => xnpWithinWithoutHtml( $within, $without ) );
            }
        }
    }
    
    $currentTag = ""; 
    $currentAttribs = ""; 
}




/**
 * 
 * ǥåˡDBΥॵۤ褦ʥǡФ֤
 * 
 * @param str XMLʸ(UTF-8)
 * @param exceeds
 * @return 
 */
function xnpImportCheckIndexLength( $str, &$exceeds )
{
    global $parser_hash;
    
    if( !is_array( $exceeds ) ) $exceeds = array( );
    
	$xnpsid = $_SESSION['XNPSID'];
    $item = array();
    
	$str = xnpEntity2Utf8( $str );
    $parser = xml_parser_create( "UTF-8" );
    if( !$parser ) return NULL;
    $parser_hash[$parser] = array(
        'tagstack' => array(),
        'handler'  => array(),
        'handlerstack'  => array() );
    $parser_hash[$parser]['handler']['/ITEM/BASIC'] = array( "checkitemlengthStartElement", "checkitemlengthEndElement", "checkitemlengthCharacterData" );
    $parser_hash[$parser]['exceeds'] = array();
    
    xml_set_element_handler($parser, "startElement", "endElement"); 
    xml_set_character_data_handler($parser, "characterData"); 
    
    if( !xml_parse( $parser, $str, true ) ){
		$lines = preg_split( "/[\r\n]+/", $str );
        die( xml_error_string( xml_get_error_code( $parser ) )." at column ".xml_get_current_column_number ( $parser )." of line ".htmlspecialchars($lines[xml_get_current_line_number( $parser )-1]) );
    }

    xml_parser_free( $parser );
    
    $exceeds = $parser_hash[$parser]['exceeds'];
    
    $parser_hash[$parser] = NULL;
    return true;
}

function checkindexlengthStartElement($parser, $name, $attribs, &$parser_hash ){}
function checkindexlengthEndElement($parser, $name, &$parser_hash){}
function checkindexlengthCharacterData($parser, $data, &$parser_hash )
{
    $tags = "/".implode( '/', $parser_hash['tagstack'] );
    
    switch( $tags ){
    case "/INDEXES/INDEX/TITLE": 
        $lengths = xnpGetColumnLengths( 'xoonips_item_title' );
        list( $within, $without ) = xnpTrimString( encodeClient2Server($data), $lengths['title'], 'UTF-8' );
        $parser_hash[$parser]['exceeds'][] = array( 'within' => $within, 'without' => $without, 'name' => _MD_XOONIPS_ITEM_INDEX_LABEL, 'value' => xnpWithinWithoutHtml( $within, $without ) );
        break;
    }
}


/**
 * 
 * Ϳ줿ե̾ݡȵǽ줿ƥ
 * XMLե̾Ǥ뤫Ƚꤷޤ
 * 
 * @return true:ƥXMLեǤ롤false:ʳΥեǤ
 * 
 */
function isExportedItemFilename( $filename )
{
    return preg_match( "/^[0-9]+\.xml\$/i", $filename ) > 0;
}

?>
