/****************************************************************************
** $Id: qt/qthread_unix.cpp   3.1.1   edited Nov 7 16:55 $
**
** QThread class for Unix
**
** Created : 20000913
**
** Copyright (C) 1992-2000 Trolltech AS.  All rights reserved.
**
** This file is part of the kernel module of the Qt GUI Toolkit.
**
** This file may be distributed under the terms of the Q Public License
** as defined by Trolltech AS of Norway and appearing in the file
** LICENSE.QPL included in the packaging of this file.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** Licensees holding valid Qt Enterprise Edition or Qt Professional Edition
** licenses for Unix/X11 or for Qt/Embedded may use this file in accordance
** with the Qt Commercial License Agreement provided with the Software.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** See http://www.trolltech.com/pricing.html or email sales@trolltech.com for
**   information about Qt Commercial License Agreements.
** See http://www.trolltech.com/qpl/ for QPL licensing information.
** See http://www.trolltech.com/gpl/ for GPL licensing information.
**
** Contact info@trolltech.com if any conditions of this licensing are
** not clear to you.
**
**********************************************************************/

#if defined(QT_THREAD_SUPPORT)

#include "qplatformdefs.h"

// Solaris redefines connect -> __xnet_connect with _XOPEN_SOURCE_EXTENDED.
#if defined(connect)
#undef connect
#endif

#include "qthread.h"

#include "qmutex.h"
#include "qwaitcondition.h"
#include <private/qmutexpool_p.h>

#ifndef QT_H
#  include "qapplication.h"
#  include "qptrlist.h"
#endif // QT_H

#include <errno.h>

static QMutexPool *qt_thread_mutexpool = 0;


extern "C" { static void *start_thread( void *_arg ); }
extern "C" { static void finish_thread( void *arg ); }


class QThreadPrivate {
public:
    QWaitCondition thread_done;
    pthread_t thread_id;
    size_t stacksize;
    void *args[2];
    bool finished : 1;
    bool running  : 1;
    bool orphan   : 1;

    QThreadPrivate( size_t ss = 0);

    static inline void start( QThread *thread ) { thread->run(); }
};

inline QThreadPrivate::QThreadPrivate( size_t ss )
    : thread_id( 0 ), stacksize( ss ),
      finished( FALSE ), running( FALSE ), orphan( FALSE )
{
    // threads have not been initialized yet, do it now
    if ( ! qt_thread_mutexpool ) QThread::initialize();
}

extern "C" {
    static void *start_thread( void *_arg )
    {
	void **arg = (void **) _arg;

	pthread_cleanup_push( finish_thread, arg[1] );
	pthread_testcancel();

	QThreadPrivate::start( (QThread *) arg[0] );

	pthread_cleanup_pop( TRUE );

	return 0;
    }

    static void finish_thread( void *arg )
    {
	QThreadPrivate *d = (QThreadPrivate *) arg;

	if ( ! d ) {
#ifdef QT_CHECK_STATE
	    qWarning( "QThread: internal error: zero data for running thread." );
#endif // QT_CHECK_STATE
	    return;
	}

	QMutexLocker locker( qt_thread_mutexpool->get( d ) );
	d->running = FALSE;
	d->finished = TRUE;
	d->thread_id = 0;

	d->thread_done.wakeAll();

	d->args[0] = d->args[1] = 0;

	if ( d->orphan )
	    delete d;
    }
}


/**************************************************************************
 ** QThread
 *************************************************************************/

/*!
    \class QThread qthread.h
    \threadsafe
    \brief The QThread class provides platform-independent threads.

    \ingroup thread
    \ingroup environment

    A QThread represents a separate thread of control within the
    program; it shares data with all the other threads within the
    process but executes independently in the way that a separate
    program does on a multitasking operating system. Instead of
    starting in main(), QThreads begin executing in run(). You inherit
    run() to include your code. For example:

    \code
    class MyThread : public QThread {

    public:

	virtual void run();

    };

    void MyThread::run()
    {
	for( int count = 0; count < 20; count++ ) {
	    sleep( 1 );
	    qDebug( "Ping!" );
	}
    }

    int main()
    {
	MyThread a;
	MyThread b;
	a.start();
	b.start();
	a.wait();
	b.wait();
    }
    \endcode

    This will start two threads, each of which writes Ping! 20 times
    to the screen and exits. The wait() calls at the end of main() are
    necessary because exiting main() ends the program, unceremoniously
    killing all other threads. Each MyThread stops executing when it
    reaches the end of MyThread::run(), just as an application does
    when it leaves main().

    \sa \link threads.html Thread Support in Qt\endlink.
*/


/*!
    This returns the thread handle of the currently executing thread.

    \warning The handle returned by this function is used for internal
    purposes and should \e not be used in any application code. On
    Windows, the returned value is a pseudo handle for the current
    thread, and it cannot be used for numerical comparison.
*/
Qt::HANDLE QThread::currentThread()
{
    return (HANDLE) pthread_self();
}


/*! \internal
  Initializes the QThread system.
*/
void QThread::initialize()
{
    if ( ! qt_global_mutexpool )
	qt_global_mutexpool = new QMutexPool( TRUE );
    if ( ! qt_thread_mutexpool )
	qt_thread_mutexpool = new QMutexPool( FALSE );
}


/*! \internal
  Cleans up the QThread system.
*/
void QThread::cleanup()
{
    delete qt_global_mutexpool;
    delete qt_thread_mutexpool;
    qt_global_mutexpool = 0;
    qt_thread_mutexpool = 0;
}


/*! \obsolete

    Use QApplication::postEvent() instead.
*/
void QThread::postEvent( QObject * receiver, QEvent * event )
{
    QApplication::postEvent( receiver, event );
}


// helper function to do thread sleeps, since usleep()/nanosleep() aren't reliable
// enough (in terms of behavior and availability)
static void thread_sleep( struct timespec *ti )
{
    pthread_mutex_t mtx = PTHREAD_MUTEX_INITIALIZER;
    pthread_cond_t cnd = PTHREAD_COND_INITIALIZER;

    pthread_mutex_lock( &mtx );
    (void) pthread_cond_timedwait( &cnd, &mtx, ti );
    pthread_mutex_unlock( &mtx );

    pthread_cond_destroy( &cnd );
    pthread_mutex_destroy( &mtx );
}

/*!
    System independent sleep. This causes the current thread to sleep
    for \a secs seconds.
*/
void QThread::sleep( unsigned long secs )
{
    struct timeval tv;
    gettimeofday( &tv, 0 );
    struct timespec ti;
    ti.tv_sec = tv.tv_sec + secs;
    ti.tv_nsec = ( tv.tv_usec * 1000 );
    thread_sleep( &ti );
}

/*!
    System independent sleep. This causes the current thread to sleep
    for \a msecs milliseconds
*/
void QThread::msleep( unsigned long msecs )
{
    struct timeval tv;
    gettimeofday( &tv, 0 );
    struct timespec ti;
    ti.tv_nsec = ( tv.tv_usec * 1000 ) + ( msecs % 1000 ) * 1000000;
    ti.tv_sec = tv.tv_sec + ( msecs / 1000 ) + ( ti.tv_nsec / 1000000000 );
    ti.tv_nsec %= 1000000000;
    thread_sleep( &ti );
}

/*!
    System independent sleep. This causes the current thread to sleep
    for \a usecs microseconds
*/
void QThread::usleep( unsigned long usecs )
{
    struct timeval tv;
    gettimeofday( &tv, 0 );
    struct timespec ti;
    ti.tv_nsec = ( tv.tv_usec * 1000 ) + ( usecs % 1000000 ) * 1000;
    ti.tv_sec = tv.tv_sec + ( usecs / 1000000 ) + ( ti.tv_nsec / 1000000000 );
    ti.tv_nsec %= 1000000000;
    thread_sleep( &ti );
}

/*!
    Constructs a new thread. The thread does not begin executing until
    start() is called.
*/
QThread::QThread()
{
    d = new QThreadPrivate;
    Q_CHECK_PTR( d );
}

/*!
    Constructs a new thread. The thread does not begin executing until
    start() is called.

    If \a stackSize is greater than zero, the maximum stack size is
    set to \a stackSize bytes, otherwise the maximum stack size is
    automatically determined by the operating system.

    \warning Most operating systems place minimum and maximum limits
    on thread stack sizes. The thread will fail to start if the stack
    size is outside these limits.
*/
QThread::QThread( unsigned int stackSize )
{
    d = new QThreadPrivate( stackSize );
    Q_CHECK_PTR( d );
}

/*!
    QThread destructor.

    Note that deleting a QThread object will not stop the execution of
    the thread it represents. Deleting a running QThread (i.e.
    finished() returns FALSE) will probably result in a program crash.
    You can wait() on a thread to make sure that it has finished.
*/
QThread::~QThread()
{
    QMutexLocker locker( qt_thread_mutexpool->get( d ) );
    if ( d->running && !d->finished ) {
#ifdef QT_CHECK_STATE
	qWarning("QThread object destroyed while thread is still running.");
#endif

	d->orphan = TRUE;
	return;
    }

    delete d;
}

/*!
    Ends the execution of the calling thread and wakes up any threads
    waiting for its termination.
*/
void QThread::exit()
{
    pthread_exit( 0 );
}

/*!
    This begins the execution of the thread by calling run(), which
    should be reimplemented in a QThread subclass to contain your
    code. If you try to start a thread that is already running, this
    call will wait until the thread has finished, and then restart the
    thread.
*/
void QThread::start()
{
    QMutexLocker locker( qt_thread_mutexpool->get( d ) );

    if ( d->running ) {
#ifdef QT_CHECK_STATE
	qWarning( "Attempt to start a thread already running" );
#endif // QT_CHECK_STATE

	d->thread_done.wait( locker.mutex() );
    }

    d->running = TRUE;
    d->finished = FALSE;

    int ret;
    pthread_attr_t attr;
    pthread_attr_init( &attr );
    pthread_attr_setinheritsched( &attr, PTHREAD_INHERIT_SCHED );
    pthread_attr_setdetachstate( &attr, PTHREAD_CREATE_DETACHED );
    if ( d->stacksize > 0 ) {
	ret = pthread_attr_setstacksize( &attr, d->stacksize );
	if ( ret ) {
#ifdef QT_CHECK_STATE
	    qWarning( "QThread::start: thread stack size error: %s", strerror( ret ) ) ;
#endif // QT_CHECK_STATE

	    // we failed to set the stacksize, and as the documentation states,
	    // the thread will fail to run...
	    d->running = FALSE;
	    d->finished = FALSE;
	    return;
	}
    }
    d->args[0] = this;
    d->args[1] = d;
    ret = pthread_create( &d->thread_id, &attr, start_thread, d->args );
    pthread_attr_destroy( &attr );

    if ( ret ) {
#ifdef QT_CHECK_STATE
	qWarning( "QThread::start: thread creation error: %s", strerror( ret ) );
#endif // QT_CHECK_STATE

	d->running = FALSE;
	d->finished = FALSE;
    }
}

/*!
    This function terminates the execution of the thread. The thread
    may or may not be terminated immediately, depending on the
    operating systems scheduling policies. Use QThread::wait()
    after terminate() for synchronous termination.

    When the thread is terminated, all threads waiting for the
    the thread to finish will be woken up.

    \warning This function is dangerous, and its use is discouraged.
    The thread can be terminate at any point in its code path.  Threads
    can be terminated while modifying data.  There is no chance for
    the thread to cleanup after itself, unlock any held mutexes, etc.
    In short, use this function only if \e absolutely necessary.
*/
void QThread::terminate()
{
    QMutexLocker private_locker( qt_thread_mutexpool->get( d ) );
    if ( d->finished || !d->running )
	return;
    if ( ! d->thread_id )
	return;
    pthread_cancel( d->thread_id );
}

/*!
    This provides similar functionality to POSIX pthread_join. A thread
    calling this will block until either of these conditions is met:
    \list
    \i The thread associated with this QThread object has finished
	execution (i.e. when it returns from \l{run()}). This function
	will return TRUE if the thread has finished. It also returns
	TRUE if the thread has not been started yet.
    \i \a time milliseconds has elapsed. If \a time is ULONG_MAX (the
	default), then the wait will never timeout (the thread must
	return from \l{run()}). This function will return FALSE if the
	wait timed out.
    \endlist
*/
bool QThread::wait( unsigned long time )
{
    QMutexLocker locker( qt_thread_mutexpool->get( d ) );
    if ( d->finished || ! d->running )
	return TRUE;
    return d->thread_done.wait( locker.mutex(), time );
}

/*!
    Returns TRUE is the thread is finished; otherwise returns FALSE.
*/
bool QThread::finished() const
{
    QMutexLocker locker( qt_thread_mutexpool->get( d ) );
    return d->finished;
}

/*!
    Returns TRUE if the thread is running; otherwise returns FALSE.
*/
bool QThread::running() const
{
    QMutexLocker locker( qt_thread_mutexpool->get( d ) );
    return d->running;
}

/*!
    \fn void QThread::run()

    This method is pure virtual, and must be implemented in derived
    classes in order to do useful work. Returning from this method
    will end the execution of the thread.

    \sa wait()
*/

#endif // QT_THREAD_SUPPORT
