# vim fileencoding=utf-8
# vi:ts=4:et
# $Id: rsk.py,v 1.2 2004/07/27 19:14:45 etale Exp $
# =====================================================


import operator
import itertools
import bisect

import young

__all__ = [
    'RSK',
    ]

class RSK(object):
    """Class for RSK
    """
    # Quick and dirty implementation of RSK.
    # Most codes are borrowed from Young class.

    def __init__(self):
        """Test whether a given sequence is a sequence of sequence.
        """
        self.top_row = self.bottom_row = None

        self.P  = young.Young()
        self.Q  = young.Young()     # recording(or insertion) tableau

    def __repr__(self):
        return "P\n" + "\nQ\n".join((str(self.P), str(self.Q))) 

    __str__ = __repr__

    def set_array(self, top_row, bottom_row=None):
        """
        Set a two-rowed array.
        """
        # This array is used for the canonical procedure.

        if bottom_row is None:
            top_row, bottom_row = range(1, len(top_row) + 1), top_row
        else:
            # Check whether a two-rowed array is a permutation or not.
            array = zip(top_row, bottom_row)
            tmp_array = array[:]
            tmp_array.sort()

            if array != tmp_array:
                # if this expression is evaluated as True,
                # we need to sort an array in an increasing order.

                # Example(Fulton P.41):
                #
                # (1,2,2,1,2,1,1,1,2)
                # (1,1,1,2,2,3,3,3,3)
                # =>
                # (1,1,1,1,1,2,2,2,2)
                # (1,2,3,3,3,1,1,2,3)
                top_row, bottom_row = zip(*tmp_array)

        self.top_row, self.bottom_row = top_row, bottom_row

    def get_matrix(self):
        """
        create a matrix from a two-rowed array.
        """
        # See Fulton P.41 for more details.
        array = zip(self.top_row, self.bottom_row)

        m = max(self.top_row)
        n = max(self.bottom_row)

        #m = max(array[0])
        #n = max(array[1])
        
        # top_row    = {i | i in [m]}
        # bottom_row = {j | j in [n]}

        # return an m * n matrix A whose (i,j) entry is the number of
        # times (i j) occurs in the array.
        return [[array.count((i, j)) for j in range(1, n+1)] for i in range(1, m+1)] 


    def setup(self):
        """
        set up P and Q from a given two-rowed array.
        """
        # Each time setup method is called, 
        # Make sure that P and Q are initialized each time when setup method is called,

        self.P  = young.Young()
        self.Q  = young.Young()

        for p, q in itertools.izip(self.top_row, self.bottom_row):

            self.index = p
            self.single_bump(q, 0)

    def single_bump(self, item, row=0):
        # bump a single element in a tableau
        try:
            self.P[row]
            self.Q
        except IndexError, e:
            # We need a new row to insert a bumped box.
            # [1,3] <- [2]
            # =>
            # [1,2] 
            # [3]
            self.P.append([item])
            self.Q.append([self.index])
            return

        n = bisect.bisect_right(self.P[row], item)

        if n < len(self.P[row]):
            # [1,2,3] <- [2]
            # =>
            # [1,2,2]
            # [3]
            bumped, self.P[row][n] = self.P[row][n], item
            self.single_bump(bumped, row+1)
        else:
            # [1,2,2] <- [3]
            # =>
            # [1,2,2,3]
            self.P[row].append(item)
            self.Q[row].append(self.index)

