# vim fileencoding=utf-8
# vi:ts=4:et
# $Id: test_rsk.py,v 1.1 2004/07/23 05:31:56 etale Exp $
# =====================================================

"""
unittest for rsk.py
"""
import unittest
from test import test_support

from young import *

class TestRSKHotta(unittest.TestCase):
    def test_hotta1(self):
        # Hotta p.118

        w = Word((4,5,1,3,6,2))

        r = RSK()
        r.set_array(w)
        r.setup()

        self.assertEqual(r.P, Young([[1,2,6],[3,5],[4]]))
        self.assertEqual(r.Q, Young([[1,2,5],[3,4],[6]]))

    def test_hotta2(self):
        # Hotta p.118

        # implicitly specify two-rowed array
        top    = (1,2,3,4,5,6)
        bottom = (4,5,1,3,6,2)

        r = RSK()
        r.set_array(top, bottom)
        r.setup()

        self.assertEqual(r.P, Young([[1,2,6],[3,5],[4]]))
        self.assertEqual(r.Q, Young([[1,2,5],[3,4],[6]]))

    def test_hotta_symmetry(self):
        # Hotta p.118

        # test of symmetry theorem

        top    = (1,2,3,4,5,6)
        bottom = (4,5,1,3,6,2)

        r = RSK()
        r.set_array(bottom, top)   # swap top and bottom.
        r.setup()

        self.assertEqual(r.Q, Young([[1,2,6],[3,5],[4]]))
        self.assertEqual(r.P, Young([[1,2,5],[3,4],[6]]))

    def test_ex1(self):
        # P.120
        # Exercise

        perm = (3,2,6,1,4,5)

        r = RSK()
        r.set_array(perm)
        r.setup()

        self.assertEqual(r.P, [[1, 4, 5], [2, 6], [3]])
        self.assertEqual(r.Q, [[1, 3, 6], [2, 5], [4]])

    def test_ex2(self):
        # P.120
        # Exercise

        perm = (5,3,4,7,1,2,6)

        r = RSK()
        r.set_array(perm)
        r.setup()

        self.assertEqual(r.P, [[1, 2, 6], [3, 4, 7], [5]])
        self.assertEqual(r.Q, [[1, 3, 4], [2, 6, 7], [5]])

class TestRSKFulton(unittest.TestCase):
    def test_basic1(self):
        # P.37

        # set up P and Q from word w.

        w = Word((5,4,8,2,3,4,1,7,5,3,1))

        r = RSK()
        r.set_array(w)
        r.setup()

        self.assertEqual(r.P, Young([[1,1,3,5],[2,3],[4,4],[5,7],[8]]))
        self.assertEqual(r.Q, Young([[1,3,6,8],[2,5],[4,9],[7,10],[11]]))

    def test_basic2(self):
        # P.37
        # another way of constructing P and Q.

        top    = (1,2,3,4,5,6,7,8,9,10,11)
        bottom = (5,4,8,2,3,4,1,7,5,3,1)

        r = RSK()
        r.set_array(top, bottom)
        r.setup()

        self.assertEqual(r.P, Young([[1,1,3,5],[2,3],[4,4],[5,7],[8]]))
        self.assertEqual(r.Q, Young([[1,3,6,8],[2,5],[4,9],[7,10],[11]]))

    def test_ex1(self):
        # P.40
        # Exercise 1

        top    = (1,1,1,2,2,3,3,3,3)
        bottom = (1,2,2,1,2,1,1,1,2)

        r = RSK()
        r.set_array(top, bottom)
        r.setup()

        self.assertEqual(r.P, Young([[1,1,1,1,1,2],[2,2,2]]))
        self.assertEqual(r.Q, Young([[1,1,1,2,3,3],[2,3,3]]))

    def test_get_matrix1(self):
        # P.41

        top    = (1,1,1,2,2,3,3,3,3)
        bottom = (1,2,2,1,2,1,1,1,2)

        r = RSK()
        r.set_array(top, bottom)    # swap top and bottom

        self.assertEqual(r.get_matrix(), [[1,2],[1,1],[3,1]])


class TestSymmetry(unittest.TestCase):
    # test of symmetry theorem

    def test_basic_symmetry(self):
        # P.37

        top    = (1,2,3,4,5,6,7,8,9,10,11)
        bottom = (5,4,8,2,3,4,1,7,5,3,1)

        r = RSK()
        r.set_array(bottom, top)
        r.setup()

        self.assertEqual(r.Q, Young([[1,1,3,5],[2,3],[4,4],[5,7],[8]]))
        self.assertEqual(r.P, Young([[1,3,6,8],[2,5],[4,9],[7,10],[11]]))

    def test_ex2a(self):
        # P.41
        # Exercise 2

        # sort an array in advance.
        top    = (1,1,1,1,1,2,2,2,2)
        bottom = (1,2,3,3,3,1,1,2,3)

        r = RSK()
        r.set_array(top, bottom)
        r.setup()

        self.assertEqual(r.P, Young([[1,1,1,2,3,3],[2,3,3]]))
        self.assertEqual(r.Q, Young([[1,1,1,1,1,2],[2,2,2]]))

    def test_ex2b(self):
        # P.41
        # Exercise 2

        top    = (1,1,1,2,2,3,3,3,3)
        bottom = (1,2,2,1,2,1,1,1,2)

        r = RSK()
        r.set_array(bottom, top)    # swap top and bottom
        r.setup()

        self.assertEqual(r.P, Young([[1,1,1,2,3,3],[2,3,3]]))
        self.assertEqual(r.Q, Young([[1,1,1,1,1,2],[2,2,2]]))

class TestMatrix(unittest.TestCase):
    def test_get_matrix_Fulton1(self):
        # Fulton P.41

        top    = (1,1,1,2,2,3,3,3,3)
        bottom = (1,2,2,1,2,1,1,1,2)

        r = RSK()
        r.set_array(top, bottom)

        self.assertEqual(r.get_matrix(), [[1,2],[1,1],[3,1]])
        # [[1,2],[1,1],[3,1]] is a transpose of [[1,1,3],[2,1,1]].

    def test_get_matrix_Fulton2(self):
        # Fulton P.41

        top    = (1,1,1,2,2,3,3,3,3)
        bottom = (1,2,2,1,2,1,1,1,2)

        r = RSK()
        r.set_array(bottom, top)    # swap top and bottom

        self.assertEqual(r.get_matrix(), [[1,1,3],[2,1,1]])
        # [[1,1,3],[2,1,1]] is a transpose of [[1,2],[1,1],[3,1]].

    def test_get_matrix_Hotta1(self):
        # Hotta p.118

        top    = (1,2,3,4,5,6)
        bottom = (4,5,1,3,6,2)

        r = RSK()
        r.set_array(top, bottom)

        self.assertEqual(r.get_matrix(),
                [[0, 0, 0, 1, 0, 0],
                [0, 0, 0, 0, 1, 0],
                [1, 0, 0, 0, 0, 0],
                [0, 0, 1, 0, 0, 0],
                [0, 0, 0, 0, 0, 1],
                [0, 1, 0, 0, 0, 0],
                ])

    def test_get_matrix_Hotta2(self):
        # Hotta p.118

        top    = (1,2,3,4,5,6)
        bottom = (4,5,1,3,6,2)

        r = RSK()
        r.set_array(bottom, top)    # swap top and bottom

        self.assertEqual(r.get_matrix(),
                [[0, 0, 1, 0, 0, 0],
                [0, 0, 0, 0, 0, 1],
                [0, 0, 0, 1, 0, 0],
                [1, 0, 0, 0, 0, 0],
                [0, 1, 0, 0, 0, 0],
                [0, 0, 0, 0, 1, 0],
                ])

def test_main(verbose=None):
    testclasses = (
            TestRSKFulton,
            TestRSKHotta,
            TestSymmetry,
            TestMatrix,
            )

    test_support.run_unittest(*testclasses)

if __name__ == '__main__':
    test_main()
