# -*- coding: utf-8 -*-
# vi:ts=4:et
# $Id: util.py,v 1.3 2004/07/23 05:21:57 etale Exp $
# =====================================================
"""library for utility functions.
"""
__all__ = [
        'flatten_tableau',
        'is_integer',
        'is_non_negative_integer',
        'pairwise',
        'sequence_splitter',
        'transpose_matrix',
        'transpose_matrix_generator',
        ]

import itertools

try:
    reversed
except NameError:
    from compat import reversed, tee
    itertools.tee = tee

def is_integer(n):
    """is_integer(n) -> Tests whether or not n is an integer."""
    return isinstance(n, (int, long))

def is_non_negative_integer(n):
    """is_non_negative_integer(n) -> Tests whether or not n in a non-negative integer."""
    return is_integer(n) and n >= 0

def flatten_tableau(iterable, reverse=False):
    """This function is used to flatten a tableau"""
    if reverse:
        iterable = reversed(iterable)
    for row in iterable:
        for col in row:
            yield col

def pairwise(iterable):
    """s -> (s0,s1), (s1,s2), (s2, s3), ..."""
    # This function is taken from an example in the library.
    # See : http://docs.python.org/lib/itertools-example.html
    a, b = itertools.tee(iterable)
    try:
        b.next()
    except StopIteration:
        pass
    return itertools.izip(a, b)


def transpose_matrix(matrix):
    """Transpose a matrix
    """
    return map(list, transpose_matrix_generator(matrix))

def transpose_matrix_generator(matrix):
    """Transpose a matrix(generator based)
    """
    return itertools.imap(None, *matrix)

def sequence_splitter(sequence):
    """
    split a sequence by a weakly increasing order.
    """
    # For example, if a sequence is (1,2,2,1,2,1,1,1,2),
    # splitter splits it into [(1,2,2), (1,2), (1,1,1,2)].

    prev = None
    tmp  = []
    for x in sequence:
        if x >= prev:
            tmp.append(x)
        else:
            yield tmp
            tmp = [x]
        prev = x
    else:
        if tmp:
            yield tmp

