package map;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.print.PrinterException;
import java.io.File;
import java.io.IOException;
import java.util.Map;

import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JSplitPane;
import javax.swing.KeyStroke;
import javax.swing.filechooser.FileFilter;

/**
 * 地図を表示するフレームです。
 * @author Kumano Tatsuo
 * Created on 2003/11/01, 16:00
 */
public class MainFrame extends JFrame {
	/**
	 * 地図を表示するパネル
	 */
	final MapPanel panel;

	/**
	 * 新しくフレームを初期化します。
	 * @param maps 地図
	 * @param panel 地図を表示するパネル
	 * @throws IOException 入出力例外
	 */
	public MainFrame(final Map<String, MapData> maps, final MapPanel panel) throws IOException {
		this.panel = panel;
		this.setTitle("Map");
		this.setLayout(new BorderLayout());
		final JSplitPane splitPane = new JSplitPane();
		this.add(splitPane, BorderLayout.CENTER);
		final JPanel sidePanel = new SidePanel(maps, panel);
		sidePanel.setPreferredSize(new Dimension(200, 0));
		sidePanel.setMinimumSize(new Dimension(0, 0));
		splitPane.setLeftComponent(sidePanel);
		splitPane.setRightComponent(panel);
		splitPane.setOneTouchExpandable(true);
		final JMenuBar menuBar = new JMenuBar();
		this.add(menuBar, BorderLayout.NORTH);
		final JMenu fileMenu = new JMenu("ファイル(F)");
		fileMenu.setMnemonic('F');
		menuBar.add(fileMenu);
		final JMenuItem exportMenu = new JMenu("エクスポート(E)");
		exportMenu.setMnemonic('E');
		fileMenu.add(exportMenu);
		final JMenuItem imageExportItem = new JMenuItem("ラスタ画像（PNG、JPEG、BMPファイル）(I)...");
		imageExportItem.setMnemonic('I');
		imageExportItem.addActionListener(new ImageExportListener());
		exportMenu.add(imageExportItem);
		final JMenuItem psExportItem = new JMenuItem("PSファイル...");
		psExportItem.setMnemonic('P');
		psExportItem.addActionListener(new PSExportListener());
		exportMenu.add(psExportItem);
		fileMenu.addSeparator();
		final JMenuItem exitItem = new JMenuItem("終了(X)");
		exitItem.setMnemonic('X');
		exitItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F4, InputEvent.ALT_DOWN_MASK));
		exitItem.addActionListener(new ExitListener());
		fileMenu.add(exitItem);
		final JMenu viewMenu = new JMenu("表示(V)");
		viewMenu.setMnemonic('V');
		menuBar.add(viewMenu);
		final JMenu fontSizeMenu = new JMenu("文字サイズ(S)");
		fontSizeMenu.setMnemonic('S');
		viewMenu.add(fontSizeMenu);
		final JMenuItem fontSizeIncrementItem = new JMenuItem("大きく(I)");
		fontSizeIncrementItem.setMnemonic('I');
		fontSizeIncrementItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_SEMICOLON, InputEvent.SHIFT_DOWN_MASK | InputEvent.CTRL_DOWN_MASK));
		fontSizeIncrementItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				panel.incrementFontSize();
			}
		});
		fontSizeMenu.add(fontSizeIncrementItem);
		final JMenuItem fontSizeDecrementItem = new JMenuItem("小さく(D)");
		fontSizeDecrementItem.setMnemonic('D');
		fontSizeDecrementItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_MINUS, InputEvent.CTRL_DOWN_MASK));
		fontSizeDecrementItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				panel.decrementFontSize();
			}
		});
		fontSizeMenu.add(fontSizeDecrementItem);
		fontSizeMenu.addSeparator();
		final JMenuItem fontSizeResetItem = new JMenuItem("標準サイズ(N)");
		fontSizeResetItem.setMnemonic('N');
		fontSizeResetItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_0, InputEvent.CTRL_DOWN_MASK));
		fontSizeResetItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				panel.resetFontSize();
			}
		});
		fontSizeMenu.add(fontSizeResetItem);
	}

	/**
	 * PSファイルを出力する実装です。
	 * @author Kumano Tatsuo
	 * Created on 2005/03/20 16:42:34
	 */
	private class PSExportListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			try {
				final JFileChooser chooser = new JFileChooser(new File("."));
				chooser.addChoosableFileFilter(new FileFilter() {
					public boolean accept(File file) {
						return file.getName().toLowerCase().endsWith(".ps");
					}

					public String getDescription() {
						return "PSファイル（*.ps）";
					}
				});
				chooser.showDialog(MainFrame.this, "エクスポート");
				final File selectedFile = chooser.getSelectedFile();
				if (selectedFile != null) {
					final String fileName = selectedFile.getName();
					if (fileName != null) {
						if (fileName.toLowerCase().endsWith(".ps")) {
							MainFrame.this.panel.printPS(selectedFile);
						} else {
							MainFrame.this.panel.printPS(new File(fileName + ".ps"));
						}
					}
				}
			} catch (PrinterException e1) {
				e1.printStackTrace();
			} catch (IOException e1) {
				e1.printStackTrace();
			}
		}
	}

	/**
	 * ラスタ画像に出力する実装です。
	 * @author Kumano Tatsuo
	 * Created on 2005/02/28 15:39:42
	 */
	private class ImageExportListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			try {
				final JFileChooser chooser = new JFileChooser(new File("."));
				chooser.addChoosableFileFilter(new FileFilter() {
					public boolean accept(File file) {
						return file.getName().toLowerCase().endsWith(".png")
								|| file.getName().toLowerCase().endsWith(".jpg")
								|| file.getName().toLowerCase().endsWith(".bmp");
					}

					public String getDescription() {
						return "ラスタ画像ファイル（*.png、*.jpg、*.bmp）";
					}
				});
				chooser.showDialog(MainFrame.this, "エクスポート");
				final File selectedFile = chooser.getSelectedFile();
				if (selectedFile != null) {
					final String fileName = selectedFile.getName();
					if (fileName != null) {
						if (fileName.toLowerCase().endsWith(".bmp")) {
							MainFrame.this.panel.printBitmap(selectedFile, "bmp");
						} else if (fileName.toLowerCase().endsWith(".jpg")) {
							MainFrame.this.panel.printBitmap(selectedFile, "jpg");
						} else if (fileName.toLowerCase().endsWith(".png")) {
							MainFrame.this.panel.printBitmap(selectedFile, "png");
						}
					}
				}
			} catch (IOException e1) {
				e1.printStackTrace();
			}
		}
	}

	/**
	 * 終了の実装です。
	 * @author Kumano Tatsuo
	 * Created on 2005/02/28 17:49:02
	 */
	private class ExitListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			System.exit(0);
		}
	}
}
