package map;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.util.Collection;
import java.util.ConcurrentModificationException;
import java.util.Map;

/**
 * 地図の読み込み、ポリゴンの結合、塗り分け、属性配置位置の計算、再描画など、
 * バックグラウンドで行う作業を管理するクラスです。
 * @author Kumano Tatsuo
 * 作成日: 2004/01/19
 */
public class BackgroundThread extends Thread {
	/**
	 * 地図
	 */
	private final Map<String, MapData> maps; // 地図

	/**
	 * 都道府県の一覧
	 */
	private final Collection<Prefecture> prefectures;

	/**
	 * 地図を表示するパネル
	 */
	private final MapPanel panel;

	/**
	 * 地図の状態が変化したかどうか
	 */
	boolean isChanged; // 地図の状態が変化したかどうか

	/**
	 * 地図を読み込むためのオブジェクト
	 */
	private final LoadMap loadMap;
	
	/**
	 * バックグラウンドで行う処理を初期化します。
	 * @param maps 地図
	 * @param panel パネル
	 * @param loadMap 地図を読み込むためのオブジェクト
	 */
	public BackgroundThread(final Map<String, MapData> maps,
			final MapPanel panel, final LoadMap loadMap) {
		this.maps = maps;
		this.panel = panel;
		this.prefectures = this.panel.getPrefectures();
		this.isChanged = true;
		this.loadMap = loadMap;
		panel.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseReleased(MouseEvent e) {
				BackgroundThread.this.isChanged = true;
			}
		});
		panel.addMouseWheelListener(new MouseWheelListener() {
			public void mouseWheelMoved(MouseWheelEvent e) {
				BackgroundThread.this.isChanged = true;
			}
		});
		panel.addComponentListener(new ComponentAdapter() {
			@Override
			public void componentResized(ComponentEvent e) {
				BackgroundThread.this.isChanged = true;
			}
		});
		panel.setActionListner(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				BackgroundThread.this.isChanged = true;
			}
		});
	}

	@Override
	public void run() {
		try {
			int mapSize = this.maps.size();
			while (true) {
				if (this.maps.size() != mapSize) {
					this.isChanged = true;
				}
				if (this.panel.isIdle() && this.isChanged) {
					this.isChanged = false;
					this.panel.setIsBusy(true);
					loadMapPaintTyomeJoinTyome();
					this.panel.setIsBusy(false);
				}
				mapSize = this.maps.size();
				Thread.sleep(200);
			}
		} catch (InterruptedException e) {
			e.printStackTrace(System.err);
		}
	}

	/**
	 * 地図の読み込み、ポリゴンの結合、塗り分け、属性配置位置の計算を行います。
	 */
	void loadMapPaintTyomeJoinTyome() {
		try {
			Prefectures.loadCities(this.prefectures, this.panel, this.maps, this.loadMap);
			long startTime = System.currentTimeMillis();
			this.loadMap.loadMap(this.maps, this.panel);
			if ((System.currentTimeMillis() - startTime) > 200) {
				System.out.println("DEBUG: 地図の読み込み：" + (System.currentTimeMillis() - startTime)
						+ " ms");
			}
			startTime = System.currentTimeMillis();
			PaintTyome paintTyome = new PaintTyome();
			paintTyome.paintTyome(this.maps);
			if ((System.currentTimeMillis() - startTime) > 200) {
				System.out.println("DEBUG: 塗り分け：" + (System.currentTimeMillis() - startTime)
						+ " ms");
			}
			startTime = System.currentTimeMillis();
			JoinPolygon joinTyome = new JoinPolygon();
			joinTyome.joinPolygon(this.maps, this.panel.getVisibleRectangle());
			JoinTatemono joinTatemono = new JoinTatemono();
			joinTatemono.joinTatemono(this.maps);
			if ((System.currentTimeMillis() - startTime) > 200) {
				System.out.println("DEBUG: ポリゴンの結合：" + (System.currentTimeMillis() - startTime)
						+ " ms");
			}
			startTime = System.currentTimeMillis();
			FixAttributeLocation fixAttributeLocation = new FixAttributeLocation();
			fixAttributeLocation.fixAttributeLocation(this.maps, this.panel);
			if ((System.currentTimeMillis() - startTime) > 200) {
				System.out.println("DEBUG: 属性描画位置の計算：" + (System.currentTimeMillis() - startTime)
						+ " ms");
			}
			this.panel.setNeedsRepaint(true);
			this.panel.repaint();
		} catch (ConcurrentModificationException e) {
		} catch (Exception exception) {
			System.err.println("Failed to load map.");
			exception.printStackTrace(System.err);
		}
	}
}
