package map;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.Shape;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionListener;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.awt.geom.Area;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.awt.print.PageFormat;
import java.awt.print.Printable;
import java.awt.print.PrinterException;
import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.ConcurrentModificationException;
import java.util.Map;
import javax.imageio.ImageIO;
import javax.print.PrintException;
import javax.swing.JPanel;
import psout.PSOut;
import search.Search;

/**
 * 地図をパネルに描画するクラスです。
 * @author Kumano Tatsuo
 */
public class MapPanel extends JPanel implements Printable {
	/**
	 * 文字の大きさ
	 */
	private double fontZoom;

	/**
	 * イメージ
	 */
	Image image;

	/**
	 * ユーザ操作を禁止している状態かどうか
	 */
	boolean isBusy; // ユーザ操作ができない状態かどうか

	/**
	 * 描画を中止するべきかどうか
	 */
	private boolean isCancel;

	/**
	 * ユーザ操作がない状態かどうか
	 */
	private boolean isIdle; // ユーザ操作がない状態かどうか

	/**
	 * 倍率が変更されたかどうか
	 */
	boolean isZoomChanged; // 倍率が変更されたかどうか

	/**
	 * 直前にマウスのボタンが押されたx座標
	 */
	int lastMousePressedX;

	/**
	 * 直前にマウスのボタンが押されたy座標
	 */
	int lastMousePressedY;

	/**
	 * 直前のマウスカーソル座標
	 */
	double lastMouseX; // 直前のマウスカーソル座標

	/**
	 * 直前のマウスカーソル座標
	 */
	double lastMouseY; // 直前のマウスカーソル座標

	/**
	 * 描画が完了した時点でのオフセット
	 */
	double lastOffsetX;

	/**
	 * 描画が完了した時点でのオフセット
	 */
	double lastOffsetY;

	/**
	 * 描画が完了した時点で見えている長方形（仮想座標）
	 */
	Rectangle2D lastVisibleRectangle;

	/**
	 * 描画が完了した時点での表示倍率
	 */
	double lastZoom;

	/**
	 * 背景スレッドに再計算を要求するためのアクションリスナ
	 */
	private ActionListener listener;

	/**
	 * 地図の設定
	 */
	MapPreferences mapPreferences; // 地図の設定

	/**
	 * 地図
	 */
	private final Map<String, MapData> maps;

	/**
	 * x座標の最大値
	 */
	private double maxX;

	/**
	 * y座標の最大値
	 */
	private double maxY;

	/**
	 * x座標の最小値
	 */
	private double minX;

	/**
	 * y座標の最小値
	 */
	private double minY;

	/**
	 * 表示倍率の最小値
	 */
	double minZoom;

	/**
	 * マウスが動いた高さ
	 */
	int mouseMotionHeight;

	/**
	 * マウスが動いた幅
	 */
	int mouseMotionWidth;

	/**
	 * 再描画が必要かどうか
	 */
	private boolean needsRepaint; // 再描画が必要かどうか

	/**
	 * オフセット（実座標）
	 */
	double offsetX; // オフセット(実座標)

	/**
	 * オフセット（実座標）
	 */
	double offsetY; // オフセット(実座標)

	/**
	 * 都道府県の一覧
	 */
	private final Collection<Prefecture> prefectures;

	/**
	 * 彩度の差分
	 */
	private float saturationDifference;

	/**
	 * 地図を検索するためのデータ構造
	 */
	private Search search;

	/**
	 * パネルの幅と高さ
	 */
	private Dimension size;

	/**
	 * 表示倍率
	 */
	double zoom; // 表示倍率

	/**
	 * 地図を表示するパネルを初期化します。
	 * @param maps 地図
	 */
	public MapPanel(final Map<String, MapData> maps) {
		this.mapPreferences = new DefaultMapPreferences();
		this.setBackground(this.mapPreferences.getBackGroundColor());
		this.offsetX = 0;
		this.offsetY = 0;
		this.zoom = 1;
		this.fontZoom = 1;
		this.saturationDifference = 0;
		this.isZoomChanged = true;
		this.lastMouseX = this.offsetX;
		this.lastMouseY = this.offsetY;
		this.maps = maps;
		this.prefectures = Prefectures.loadPrefectures(this.mapPreferences);
		setIdle(true);
		setIsBusy(false);
		setNeedsRepaint(true);
		addMouseListener(new MouseAdapter() {
			@Override
			public void mousePressed(MouseEvent e) {
				if (!MapPanel.this.isBusy) {
					MapPanel.this.lastMouseX = e.getX();
					MapPanel.this.lastMouseY = e.getY();
					MapPanel.this.lastMousePressedX = e.getX();
					MapPanel.this.lastMousePressedY = e.getY();
					setIdle(false);
				}
			}

			@Override
			public void mouseReleased(MouseEvent e) {
				if (!MapPanel.this.isBusy) {
					MapPanel.this.mouseMotionWidth = 0;
					MapPanel.this.mouseMotionHeight = 0;
					setNeedsRepaint(true);
					setIdle(true);
				}
			}
		});
		addMouseMotionListener(new MouseMotionListener() {
			public void mouseDragged(MouseEvent e) {
				if (MapPanel.this.isBusy) {
					MapPanel.this.lastMouseX = e.getX();
					MapPanel.this.lastMouseY = e.getY();
					MapPanel.this.lastMousePressedX = e.getX();
					MapPanel.this.lastMousePressedY = e.getY();
				} else {
					MapPanel.this.offsetX -= (e.getX() - MapPanel.this.lastMouseX);
					MapPanel.this.offsetY -= (e.getY() - MapPanel.this.lastMouseY);
					MapPanel.this.lastMouseX = e.getX();
					MapPanel.this.lastMouseY = e.getY();
					MapPanel.this.mouseMotionWidth = e.getX() - MapPanel.this.lastMousePressedX;
					MapPanel.this.mouseMotionHeight = e.getY() - MapPanel.this.lastMousePressedY;
					repaint();
				}
			}

			public void mouseMoved(MouseEvent e) {
				if (e.getModifiersEx() == (InputEvent.SHIFT_DOWN_MASK | InputEvent.CTRL_DOWN_MASK | InputEvent.ALT_DOWN_MASK)) {
					final Point2D point = toVirtualLocation(new Point2D.Double(e.getX(), e.getY()));
					System.out.println("DEBUG: x = " + point.getX() + ", y = " + point.getY());
				}
			}
		});
		addMouseWheelListener(new MouseWheelListener() {
			public void mouseWheelMoved(MouseWheelEvent e) {
				final int wheelRotation = e.getWheelRotation();
				final int centerX = e.getX();
				final int centerY = e.getY();
				doWheelRotation(wheelRotation, centerX, centerY);
			}
		});
	}

	/**
	 * 座標の最大値と最小値を計算します。
	 */
	public void calcMinMaxXY() {
		this.minX = Double.POSITIVE_INFINITY;
		this.minY = Double.POSITIVE_INFINITY;
		this.maxX = Double.NEGATIVE_INFINITY;
		this.maxY = Double.NEGATIVE_INFINITY;
		if (this.maps != null) {
			if (this.maps.isEmpty()) {
				for (final Prefecture prefecture : this.prefectures) {
					final Rectangle2D boudns = prefecture.getBounds();
					this.minX = Math.min(this.minX, boudns.getMinX());
					this.minY = Math.min(this.minY, boudns.getMinY());
					this.maxX = Math.max(this.maxX, boudns.getMaxX());
					this.maxY = Math.max(this.maxY, boudns.getMaxY());
				}
			} else {
				for (MapData mapData : this.maps.values()) {
					Rectangle bounds = mapData.getBounds().getBounds();
					if (bounds.getMinX() < this.minX) {
						this.minX = bounds.getMinX();
					}
					if (bounds.getMinY() < this.minY) {
						this.minY = bounds.getMinY();
					}
					if (this.maxX < bounds.getMaxX()) {
						this.maxX = bounds.getMaxX();
					}
					if (this.maxY < bounds.getMaxY()) {
						this.maxY = bounds.getMaxY();
					}
				}
			}
		}
	}

	/**
	 * 文字を小さくします。
	 */
	public void decreaseFontSize() {
		this.fontZoom *= 0.9;
		this.mapPreferences.setFontZoom(this.fontZoom);
		if (this.listener != null) {
			this.listener.actionPerformed(new ActionEvent(this, this.hashCode(),
					"decrement font size"));
		}
	}

	/**
	 * 彩度を減らします。
	 */
	public void decreaseSaturation() {
		this.saturationDifference -= 0.05;
		this.setNeedsRepaint(true);
		repaint();
	}

	/**
	 * マウスホイールの回転をシミュレートします。
	 * @param wheelRotation マウスホイールの回転方向、1で手前、-1で向こう
	 * @param centerX マウスポインタのx座標（実座標）
	 * @param centerY マウスポインタのy座標（実座標）
	 */
	void doWheelRotation(final int wheelRotation, final int centerX, final int centerY) {
		setCancel(true);
		final double newZoom = Math.min(Const.ZOOM_MAX, Math.max(this.minZoom,
				wheelRotation > 0 ? this.zoom * Const.ZOOM_RATE : this.zoom / Const.ZOOM_RATE));
		zoom(newZoom, centerX, centerY);
	}

	/**
	 * 地図を描画します。
	 * @param thresholdTime 処理に要した時間を表示するしきい値
	 * @param g2 グラフィクスコンテキスト
	 */
	public void drawMap(final long thresholdTime, final Graphics2D g2) {
		final int width = (this.size == null) ? getWidth() : this.size.width;
		final int height = (this.size == null) ? getHeight() : this.size.height;
		this.isCancel = false;
		g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
		g2.setColor(this.mapPreferences.getMizuPreferences().getFillColor());
		g2.fillRect(0, 0, width, height);
		g2.translate(-this.offsetX, -this.offsetY);
		g2.scale(this.zoom, this.zoom);
		double x = this.offsetX / this.zoom;
		double y = this.offsetY / this.zoom;
		double w = width / this.zoom;
		double h = height / this.zoom;
		this.setIsBusy(true);
		try {
			// 都道府県を描画する
			for (final Prefecture prefecture : this.prefectures) {
				g2.setStroke(prefecture.hasFine() ? new BasicStroke(50f, BasicStroke.CAP_BUTT,
						BasicStroke.JOIN_BEVEL) : new BasicStroke(500f));
				if (prefecture.getBounds().intersects(x, y, w, h)) {
					final Shape shape = prefecture.hasFine() ? prefecture.getFineShape()
							: prefecture.getShape();
					if (shape.intersects(x, y, w, h)) {
						final Color color = prefecture.getColor();
						if (this.saturationDifference == 0) {
							g2.setColor(color);
						} else {
							final float[] hsb = Color.RGBtoHSB(color.getRed(), color.getGreen(),
									color.getBlue(), new float[] { 0, 0, 0 });
							hsb[1] = Math.min(1, Math.max(0, hsb[1] + this.saturationDifference));
							g2.setColor(Color.getHSBColor(hsb[0], hsb[1], hsb[2]));
						}
						g2.fill(shape);
						g2.setColor(Color.BLACK);
						g2.draw(shape);
					}
				}
			}
			// 市区町村を描画する
			for (final Prefecture prefecture : this.prefectures) {
				if (prefecture.hasCities()) {
					g2
							.setStroke(new BasicStroke(20f, BasicStroke.CAP_BUTT,
									BasicStroke.JOIN_BEVEL));
					for (final City city : prefecture.getCities()) {
						final Shape shape = city.hasFineShape() ? city.getFineShape() : city
								.getShape();
						if (shape.intersects(x, y, w, h)) {
							if (city.getURL() == null) {
								g2.setColor(Color.LIGHT_GRAY);
								g2.fill(shape);
							}
							g2.setColor(Color.BLACK);
							g2.draw(shape);
						}
					}
					g2.setStroke(prefecture.hasFine() ? new BasicStroke(50f, BasicStroke.CAP_BUTT,
							BasicStroke.JOIN_BEVEL) : new BasicStroke(500f));
					g2.draw(prefecture.hasFine() ? prefecture.getFineShape() : prefecture
							.getShape());
				}
			}
			if (this.maps != null) {
				g2.setStroke(new BasicStroke(1f));
				for (MapData mapData : this.maps.values()) {
					if (mapData.getBounds().intersects(x, y, w, h)) {
						// 海を描画する
						if (mapData.hasTyome()) {
							g2.setColor(this.mapPreferences.getMizuPreferences().getFillColor());
							g2.fill(mapData.getBounds());
							g2.draw(mapData.getBounds());
						}
					}
				}
				for (MapData mapData : this.maps.values()) {
					if (mapData.getBounds().intersects(x, y, w, h)) {
						// 丁目を描画する
						if (mapData.hasTyome()) {
							for (PolygonData polygon : mapData.getTyome().values()) {
								if (polygon.getArea() != null) {
									if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_TYOME) {
										final Color color = this.mapPreferences
												.getTyomeFillColor(polygon.getTyomeColorIndex());
										if (this.saturationDifference == 0) {
											g2.setColor(color);
										} else {
											final float[] hsb = Color.RGBtoHSB(color.getRed(),
													color.getGreen(), color.getBlue(), new float[] {
															0, 0, 0 });
											hsb[1] = Math.min(1, Math.max(0, hsb[1]
													+ this.saturationDifference));
											g2.setColor(Color.getHSBColor(hsb[0], hsb[1], hsb[2]));
										}
										g2.fill(polygon.getArea());
									}
								}
							}
						}
					}
				}
				if (this.isCancel) {
					System.out.println("DEBUG: 描画を中止しました。");
					this.isCancel = false;
					this.setIsBusy(false);
					return;
				}
				long startTime2 = System.currentTimeMillis();
				for (MapData mapData : this.maps.values()) {
					if (mapData.getBounds().intersects(x, y, w, h)) {
						// 場地を描画する
						if (mapData.hasZyouti()) {
							for (PolygonData polygon : mapData.getZyouti().values()) {
								if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_RAILROAD) {
									g2.setColor(this.mapPreferences.getZyoutiPreferences()
											.getFillColor());
									g2.fill(polygon.getArea());
								} else if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_PARK) {
									g2.setColor(this.mapPreferences.getParkPreferences()
											.getFillColor());
									g2.fill(polygon.getArea());
								} else if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_SCHOOL) {
									g2.setColor(this.mapPreferences.getZyoutiPreferences()
											.getFillColor());
									g2.fill(polygon.getArea());
								} else if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_TEMPLE) {
									g2.setColor(this.mapPreferences.getZyoutiPreferences()
											.getFillColor());
									g2.fill(polygon.getArea());
								} else if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_GRAVEYARD) {
									g2.setColor(this.mapPreferences.getZyoutiPreferences()
											.getFillColor());
									g2.fill(polygon.getArea());
								} else if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_OTHER) {
									g2.setColor(this.mapPreferences.getZyoutiPreferences()
											.getFillColor());
									g2.fill(polygon.getArea());
								}
							}
						}
						// 内水面を描画する
						if (mapData.hasMizu()) {
							for (PolygonData polygon : mapData.getMizu().values()) {
								if (polygon.getArea() != null) {
									g2.setColor(this.mapPreferences.getMizuPreferences()
											.getFillColor());
									g2.fill(polygon.getArea());
									g2.draw(polygon.getArea());
								}
							}
						}
					}
				}
				if (this.isCancel) {
					System.out.println("DEBUG: 描画を中止しました。");
					this.isCancel = false;
					this.setIsBusy(false);
					return;
				}
				for (MapData mapData : this.maps.values()) {
					if (mapData.getBounds().intersects(x, y, w, h)) {
						// 建物を描画する
						if (mapData.hasTatemono()) {
							for (PolygonData polygon : mapData.getTatemono().values()) {
								if (polygon.getArea() != null) {
									g2.setColor(this.mapPreferences.getTatemonoPreferences()
											.getFillColor());
									g2.fill(polygon.getArea());
								}
							}
						}
					}
				}
				if ((System.currentTimeMillis() - startTime2) > thresholdTime) {
					System.out.println("DEBUG: （場地、内水面、建物の塗りつぶし："
							+ (System.currentTimeMillis() - startTime2) + " ms）");
				}
				startTime2 = System.currentTimeMillis();
				g2
						.setStroke(new BasicStroke(this.mapPreferences.getZyoutiPreferences()
								.getWidth()));
				for (MapData mapData : this.maps.values()) {
					if (mapData.getBounds().intersects(x, y, w, h)) {
						// 場地界を描画する
						if (mapData.hasZyouti()) {
							for (ArcData arc : mapData.getOthers().values()) {
								if (arc.getClassification() != ArcData.TYPE_RAILWAY) {
									if (arc.getTag() == ArcData.TAG_NORMAL) {
										g2.setColor(this.mapPreferences.getZyoutiPreferences()
												.getBorderColor());
										g2.draw(arc.getPath());
									}
								}
							}
						}
						// 内水面界を描画する
						if (mapData.hasMizuArc()) {
							for (ArcData arc : mapData.getMizuArc().values()) {
								if (arc.getTag() == ArcData.TAG_NORMAL) {
									if (arc.getClassification() == ArcData.TYPE_MIZU_INSIDE) {
										g2.setColor(this.mapPreferences.getMizuPreferences()
												.getBorderColor());
										g2.draw(arc.getPath());
									}
								}
							}
						}
						// 建物界を描画する
						if (mapData.hasTatemonoArc()) {
							for (ArcData arc : mapData.getTatemonoArc().values()) {
								if (arc.getTag() == ArcData.TAG_NORMAL) {
									g2.setColor(this.mapPreferences.getTatemonoPreferences()
											.getBorderColor());
									g2.draw(arc.getPath());
								}
							}
						}
					}
				}
				if ((System.currentTimeMillis() - startTime2) > thresholdTime) {
					System.out.println("DEBUG: （場地界、内水面界、建物界の描画："
							+ (System.currentTimeMillis() - startTime2) + " ms）");
				}
				if (this.isCancel) {
					System.out.println("DEBUG: 描画を中止しました。");
					this.isCancel = false;
					this.setIsBusy(false);
					return;
				}
				startTime2 = System.currentTimeMillis();
				for (MapData mapData : this.maps.values()) {
					if (mapData.getBounds().intersects(x, y, w, h)) {
						// 道路の輪郭を描画する
						if (mapData.hasRoadArc() && mapData.hasTyome()) {
							for (ArcData arc : mapData.getRoadArc().values()) {
								if (arc.getRoadType() == ArcData.ROAD_HIGHWAY) {
									g2.setStroke(new BasicStroke(this.mapPreferences
											.getHighwayPreferences().getWidth() + 2));
									g2.setColor(this.mapPreferences.getHighwayPreferences()
											.getBorderColor());
									g2.draw(arc.getPath());
								} else if (arc.getRoadType() == ArcData.ROAD_KOKUDO) {
									g2.setStroke(new BasicStroke(this.mapPreferences
											.getKokudoPreferences().getWidth() + 2));
									g2.setColor(this.mapPreferences.getKokudoPreferences()
											.getBorderColor());
									g2.draw(arc.getPath());
								} else if (arc.getRoadType() == ArcData.ROAD_KENDO) {
									g2.setStroke(new BasicStroke(this.mapPreferences
											.getKendoPreferences().getWidth() + 2));
									g2.setColor(this.mapPreferences.getKendoPreferences()
											.getBorderColor());
									g2.draw(arc.getPath());
								} else if (arc.getRoadType() == ArcData.ROAD_CHIHODO) {
									g2.setStroke(new BasicStroke(this.mapPreferences
											.getChihodoPreferences().getWidth() + 2));
									g2.setColor(this.mapPreferences.getChihodoPreferences()
											.getBorderColor());
									g2.draw(arc.getPath());
								} else if (arc.getRoadType() == ArcData.ROAD_MAJOR) {
									g2.setStroke(new BasicStroke(this.mapPreferences
											.getMajorRoadPreferences().getWidth() + 2));
									g2.setColor(this.mapPreferences.getMajorRoadPreferences()
											.getBorderColor());
									g2.draw(arc.getPath());
								} else {
									g2.setStroke(new BasicStroke(this.mapPreferences
											.getNormalRoadPreferences().getWidth() + 2));
									g2.setColor(this.mapPreferences.getNormalRoadPreferences()
											.getBorderColor());
									g2.draw(arc.getPath());
								}
							}
						}
					}
				}
				if ((System.currentTimeMillis() - startTime2) > thresholdTime) {
					System.out.println("DEBUG: （道路の輪郭線の描画："
							+ (System.currentTimeMillis() - startTime2) + " ms）");
				}
				if (this.isCancel) {
					System.out.println("DEBUG: 描画を中止しました。");
					this.isCancel = false;
					this.setIsBusy(false);
					return;
				}
				startTime2 = System.currentTimeMillis();
				for (MapData mapData : this.maps.values()) {
					if (mapData.getBounds().intersects(x, y, w, h)) {
						// 道路の塗りつぶし部を描画する
						if (mapData.hasRoadArc() && mapData.hasTyome()) {
							// 一般の道路を描画する
							for (ArcData arc : mapData.getRoadArc().values()) {
								if (arc.getRoadType() == ArcData.ROAD_NORMAL) {
									g2.setColor(this.mapPreferences.getNormalRoadPreferences()
											.getFillColor());
									g2.setStroke(new BasicStroke(this.mapPreferences
											.getNormalRoadPreferences().getWidth()));
									g2.draw(arc.getPath());
								} else if (arc.getRoadType() == ArcData.ROAD_MAJOR) {
									g2.setColor(this.mapPreferences.getMajorRoadPreferences()
											.getFillColor());
									g2.setStroke(new BasicStroke(this.mapPreferences
											.getMajorRoadPreferences().getWidth()));
									g2.draw(arc.getPath());
								}
							}
							// 主要地方道、県道を描画する
							for (ArcData arc : mapData.getRoadArc().values()) {
								if (arc.getRoadType() == ArcData.ROAD_KENDO) {
									g2.setStroke(new BasicStroke(this.mapPreferences
											.getKendoPreferences().getWidth()));
									g2.setColor(this.mapPreferences.getKendoPreferences()
											.getFillColor());
									g2.draw(arc.getPath());
								} else if (arc.getRoadType() == ArcData.ROAD_CHIHODO) {
									g2.setStroke(new BasicStroke(this.mapPreferences
											.getChihodoPreferences().getWidth()));
									g2.setColor(this.mapPreferences.getChihodoPreferences()
											.getFillColor());
									g2.draw(arc.getPath());
								}
							}
							// 国道を描画する
							g2.setStroke(new BasicStroke(this.mapPreferences.getKokudoPreferences()
									.getWidth()));
							g2.setColor(this.mapPreferences.getKokudoPreferences().getFillColor());
							for (ArcData arc : mapData.getRoadArc().values()) {
								if (arc.getRoadType() == ArcData.ROAD_KOKUDO) {
									g2.draw(arc.getPath());
								}
							}
							// 高速道路を描画する
							g2.setStroke(new BasicStroke(this.mapPreferences
									.getHighwayPreferences().getWidth()));
							g2.setColor(this.mapPreferences.getHighwayPreferences().getFillColor());
							for (ArcData arc : mapData.getRoadArc().values()) {
								if (arc.getRoadType() == ArcData.ROAD_HIGHWAY) {
									g2.draw(arc.getPath());
								}
							}
						}
					}
				}
				if ((System.currentTimeMillis() - startTime2) > thresholdTime) {
					System.out.println("DEBUG: （道路の塗りつぶし部の描画："
							+ (System.currentTimeMillis() - startTime2) + " ms）");
				}
				if (this.isCancel) {
					System.out.println("DEBUG: 描画を中止しました。");
					this.isCancel = false;
					this.setIsBusy(false);
					return;
				}
				startTime2 = System.currentTimeMillis();
				for (MapData mapData : this.maps.values()) {
					if (mapData.getBounds().intersects(x, y, w, h)) {
						// 行政界を描画する
						if (mapData.hasGyousei() && mapData.hasTyome()) {
							for (ArcData arc : mapData.getGyousei().values()) {
								if (arc.getTag() == ArcData.TAG_NORMAL) {
									if ((arc.getClassification() == ArcData.TYPE_GYOUSEI_PREFECTURE)
											|| (arc.getClassification() == ArcData.TYPE_GYOUSEI_CITY)
											|| (arc.getClassification() == ArcData.TYPE_GYOUSEI_VILLAGE)) {
										g2.setColor(this.mapPreferences.getSi_tyoPreferences()
												.getBorderColor());
										g2.setStroke(new BasicStroke(this.mapPreferences
												.getSi_tyoPreferences().getWidth()));
									} else {
										g2.setColor(this.mapPreferences.getTyomePreferences()
												.getBorderColor());
										g2.setStroke(new BasicStroke(this.mapPreferences
												.getTyomePreferences().getWidth()));
									}
									g2.draw(arc.getPath());
								}
							}
						}
					}
				}
				if ((System.currentTimeMillis() - startTime2) > thresholdTime) {
					System.out.println("DEBUG: （行政界の描画："
							+ (System.currentTimeMillis() - startTime2) + " ms）");
				}
				float jrShinkansenRailwayWidth = this.mapPreferences.getJRShinkansenPreferences()
						.getWidth();
				float otherRailwayWidth = this.mapPreferences.getRailwayPreferences().getWidth();
				if (this.isCancel) {
					System.out.println("DEBUG: 描画を中止しました。");
					this.isCancel = false;
					this.setIsBusy(false);
					return;
				}
				for (MapData mapData : this.maps.values()) {
					if (mapData.getBounds().intersects(x, y, w, h)) {
						// 鉄道を描画する
						if (mapData.hasOthers() && mapData.hasTyome()) {
							g2.setStroke(new BasicStroke(this.mapPreferences.getJRPreferences()
									.getWidth() + 4));
							for (ArcData arc : mapData.getOthers().values()) {
								if (arc.getClassification() == ArcData.TYPE_RAILWAY) {
									if (arc.getRailwayType() == ArcData.RAILWAY_JR) {
										g2.setColor(this.mapPreferences.getJRPreferences()
												.getBorderColor());
										g2.draw(arc.getPath());
									} else if (arc.getRailwayType() == ArcData.RAILWAY_JR_SHINKANSEN) {
										g2.setColor(this.mapPreferences
												.getJRShinkansenPreferences().getBorderColor());
										g2.draw(arc.getPath());
									}
								}
							}
							for (ArcData arc : mapData.getOthers().values()) {
								if (arc.getClassification() == ArcData.TYPE_RAILWAY) {
									if (arc.getRailwayType() == ArcData.RAILWAY_JR) {
										g2.setStroke(new BasicStroke(this.mapPreferences
												.getJRPreferences().getWidth(),
												BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER, 10,
												new float[] { this.mapPreferences
														.getJRPreferences().getWidth() * 5 }, 0));
										g2.setColor(this.mapPreferences.getJRPreferences()
												.getFillColor());
										g2.draw(arc.getPath());
									} else if (arc.getRailwayType() == ArcData.RAILWAY_JR_SHINKANSEN) {
										g2.setStroke(new BasicStroke(jrShinkansenRailwayWidth,
												BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER, 10,
												new float[] { this.mapPreferences
														.getJRPreferences().getWidth() * 15 }, 0));
										g2.setColor(this.mapPreferences
												.getJRShinkansenPreferences().getFillColor());
										g2.draw(arc.getPath());
									} else {
										g2.setStroke(new BasicStroke(otherRailwayWidth));
										g2.setColor(this.mapPreferences.getRailwayPreferences()
												.getBorderColor());
										g2.draw(arc.getPath());
									}
								}
							}
						}
					}
				}
				if (this.isCancel) {
					System.out.println("DEBUG: 描画を中止しました。");
					this.isCancel = false;
					this.setIsBusy(false);
					return;
				}
				// 丁目の情報がないときは、鉄道、駅、高速道路、国道、市区町村界を描画する
				for (MapData mapData : this.maps.values()) {
					if (mapData.getBounds().intersects(x, y, w, h)) {
						if (!mapData.hasRoadArc()) {
							if (mapData.hasTyome()) {
								g2.setStroke(new BasicStroke(3f / (float) this.zoom));
								for (ArcData arc : mapData.getLargeRoadArc().values()) {
									if (arc.getRoadType() == ArcData.ROAD_HIGHWAY) {
										g2.setColor(this.mapPreferences.getHighwayPreferences()
												.getBorderColor());
									} else if (arc.getRoadType() == ArcData.ROAD_KOKUDO) {
										g2.setColor(this.mapPreferences.getKokudoPreferences()
												.getBorderColor());
									} else if (arc.getRoadType() == ArcData.ROAD_CHIHODO) {
										g2.setColor(this.mapPreferences.getChihodoPreferences()
												.getBorderColor());
									}
									g2.draw(arc.getPath());
								}
								g2.setStroke(new BasicStroke(2f / (float) this.zoom));
								for (ArcData arc : mapData.getLargeRoadArc().values()) {
									if (arc.getRoadType() == ArcData.ROAD_HIGHWAY) {
										g2.setColor(this.mapPreferences.getHighwayPreferences()
												.getFillColor());
									} else if (arc.getRoadType() == ArcData.ROAD_KOKUDO) {
										g2.setColor(this.mapPreferences.getKokudoPreferences()
												.getFillColor());
									} else if (arc.getRoadType() == ArcData.ROAD_CHIHODO) {
										g2.setColor(this.mapPreferences.getChihodoPreferences()
												.getFillColor());
									}
									g2.draw(arc.getPath());
								}
							} else {
								g2.setStroke(new BasicStroke(1f / (float) this.zoom));
								for (ArcData arc : mapData.getLargeRoadArc().values()) {
									if (arc.getRoadType() == ArcData.ROAD_HIGHWAY) {
										g2.setColor(this.mapPreferences.getHighwayPreferences()
												.getBorderColor());
									} else if (arc.getRoadType() == ArcData.ROAD_KOKUDO) {
										g2.setColor(this.mapPreferences.getKokudoPreferences()
												.getBorderColor());
									} else if (arc.getRoadType() == ArcData.ROAD_CHIHODO) {
										g2.setColor(this.mapPreferences.getChihodoPreferences()
												.getBorderColor());
									}
									g2.draw(arc.getLine());
								}
							}
						}
						if (!mapData.hasTyome()) {
							g2.setColor(this.mapPreferences.getRailwayPreferences()
									.getBorderColor());
							g2.setStroke(new BasicStroke(1f / (float) this.zoom));
							for (final ArcData arc : mapData.getOthers().values()) {
								if (arc.getClassification() == ArcData.TYPE_RAILWAY) {
									g2.draw(arc.getPath());
								}
							}
						}
						// for debug
						//						g2.setColor(this.mapPreferences.getMapBoundsColor());
						//						g2.setStroke(new BasicStroke(1f));
						//						g2.draw(mapData.getBounds());
						//						g2.setFont(new Font("default", Font.PLAIN, 300));
						//						g2.drawString(mapData.getMapName(), (float) mapData.getBounds().getBounds().getX(), (float) mapData.getBounds().getBounds().getMaxY());
					}
				}
				if (this.isCancel) {
					System.out.println("DEBUG: 描画を中止しました。");
					this.isCancel = false;
					this.setIsBusy(false);
					return;
				}
				startTime2 = System.currentTimeMillis();
				g2.scale(1 / this.zoom, 1 / this.zoom);
				g2.translate(this.offsetX, this.offsetY);
				g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
						RenderingHints.VALUE_ANTIALIAS_OFF);
				g2.setStroke(new BasicStroke(1f));
				for (MapData mapData : this.maps.values()) {
					// 建物のラベルを描画する
					if (isVisible(mapData.getBounds())) {
						if (mapData.hasTatemono()) {
							g2.setColor(this.mapPreferences.getTatemonoPreferences()
									.getAttributeColor());
							final Font tatemonoFont = this.mapPreferences.getTatemonoPreferences()
									.getFont();
							g2.setFont(tatemonoFont);
							FontMetrics metrics = this.getFontMetrics(tatemonoFont);
							int descent = metrics.getDescent();
							double pointSize = 4;
							for (PolygonData polygon : mapData.getTatemono().values()) {
								if (polygon.getAttribute() != null) {
									if (getVisibleRectangle().contains(polygon.getAttributeX(),
											polygon.getAttributeY())) {
										g2.fill(new Ellipse2D.Double((polygon.getX() * this.zoom)
												- this.offsetX - (pointSize / 2),
												(polygon.getY() * this.zoom) - this.offsetY
														- (pointSize / 2), pointSize, pointSize));
										g2.drawString(polygon.getAttribute(), (float) ((polygon
												.getAttributeX() * this.zoom) - this.offsetX),
												(float) ((polygon.getAttributeY() * this.zoom)
														- this.offsetY - descent));
									}
								}
							}
						}
						// 場地のラベルを描画する
						if (mapData.hasZyouti()) {
							g2.setColor(this.mapPreferences.getZyoutiPreferences()
									.getAttributeColor());
							final Font zyoutiFont = this.mapPreferences.getZyoutiPreferences()
									.getFont();
							g2.setFont(zyoutiFont);
							FontMetrics metrics = this.getFontMetrics(zyoutiFont);
							int descent = metrics.getDescent();
							for (PolygonData polygon : mapData.getZyouti().values()) {
								if (polygon.getAttribute() != null) {
									if (getVisibleRectangle().contains(polygon.getAttributeX(),
											polygon.getAttributeY())) {
										g2.drawString(polygon.getAttribute(), (float) ((polygon
												.getAttributeX() * this.zoom) - this.offsetX),
												(float) ((polygon.getAttributeY() * this.zoom)
														- this.offsetY - descent));
									}
								}
							}
						}
						// 内水面のラベルを描画する
						if (mapData.hasMizu()) {
							g2.setColor(this.mapPreferences.getMizuPreferences()
									.getAttributeColor());
							final Font mizuFont = this.mapPreferences.getMizuPreferences()
									.getFont();
							g2.setFont(mizuFont);
							FontMetrics metrics = this.getFontMetrics(mizuFont);
							int descent = metrics.getDescent();
							for (PolygonData polygon : mapData.getMizu().values()) {
								if (polygon.getAttribute() != null) {
									if (getVisibleRectangle().contains(polygon.getAttributeX(),
											polygon.getAttributeY())) {
										g2.drawString(polygon.getAttribute(), (float) ((polygon
												.getAttributeX() * this.zoom) - this.offsetX),
												(float) ((polygon.getAttributeY() * this.zoom)
														- this.offsetY - descent));
									}
								}
							}
						}
						// 丁目のラベルを描画する 
						if (mapData.hasTyome()) {
							g2.setColor(this.mapPreferences.getTyomePreferences()
									.getAttributeColor());
							for (PolygonData polygon : mapData.getTyome().values()) {
								final Font tyomeFont = polygon.getTyomeFont();
								if (tyomeFont != null) {
									g2.setFont(tyomeFont);
									FontMetrics metrics = this.getFontMetrics(tyomeFont);
									int descent = metrics.getDescent();
									if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_TYOME) {
										if (polygon.getAttribute() != null) {
											if (getVisibleRectangle().contains(
													polygon.getAttributeX(),
													polygon.getAttributeY())) {
												g2
														.drawString(
																polygon.getAttribute(),
																(float) ((polygon.getAttributeX() * this.zoom) - this.offsetX),
																(float) ((polygon.getAttributeY() * this.zoom)
																		- this.offsetY - descent));
											}
										}
									}
								}
							}
						}
						// 駅のラベルを描画する
						if (mapData.hasEki()) {
							double pointSize = this.mapPreferences.getEkiPreferences().getWidth();
							final Font ekiFont = this.mapPreferences.getEkiPreferences().getFont();
							g2.setFont(ekiFont);
							FontMetrics metrics = this.getFontMetrics(ekiFont);
							int descent = metrics.getDescent();
							for (PointData point : mapData.getEki().values()) {
								if (point.getAttribute() != null) {
									g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
											RenderingHints.VALUE_ANTIALIAS_ON);
									Ellipse2D ellipse = new Ellipse2D.Double(
											(point.getX() * this.zoom) - this.offsetX
													- (pointSize / 2), (point.getY() * this.zoom)
													- this.offsetY - (pointSize / 2), pointSize,
											pointSize);
									g2.setColor(this.mapPreferences.getEkiPreferences()
											.getFillColor());
									g2.fill(ellipse);
									g2.setColor(this.mapPreferences.getEkiPreferences()
											.getBorderColor());
									g2.draw(ellipse);
									if (point.getAttributeX() != 0 && point.getAttributeY() != 0) {
										g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
												RenderingHints.VALUE_ANTIALIAS_OFF);
										g2.drawString(point.getAttribute(), (float) ((point
												.getAttributeX() * this.zoom) - this.offsetX),
												(float) ((point.getAttributeY() * this.zoom)
														- this.offsetY - descent));
									}
								}
							}
						}
						// 道路のラベルを描画する
						g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
								RenderingHints.VALUE_ANTIALIAS_OFF);
						final Font roadFont = this.mapPreferences.getNormalRoadPreferences()
								.getFont();
						if (mapData.hasRoadArc()) {
							g2.setFont(roadFont);
							FontMetrics metrics = this.getFontMetrics(roadFont);
							int descent = metrics.getDescent();
							g2.setColor(this.mapPreferences.getNormalRoadPreferences()
									.getAttributeColor());
							for (ArcData arc : mapData.getRoadArc().values()) {
								if (arc.getAttribute() != null) {
									g2
											.drawString(arc.getAttribute(), (float) ((arc
													.getAttributeX() * this.zoom) - this.offsetX),
													(float) ((arc.getAttributeY() * this.zoom)
															- this.offsetY - descent));
								}
							}
						}
						// 高速道路、国道のラベルを描画する
						if (!mapData.hasRoadArc() && mapData.hasLargeRoadArc()) {
							g2.setFont(roadFont);
							FontMetrics metrics = this.getFontMetrics(roadFont);
							int descent = metrics.getDescent();
							g2.setColor(this.mapPreferences.getNormalRoadPreferences()
									.getAttributeColor());
							for (ArcData arc : mapData.getLargeRoadArc().values()) {
								if (arc.getAttribute() != null) {
									g2
											.drawString(arc.getAttribute(), (float) ((arc
													.getAttributeX() * this.zoom) - this.offsetX),
													(float) ((arc.getAttributeY() * this.zoom)
															- this.offsetY - descent));
								}
							}
						}
						// 鉄道のラベルを描画する
						if (mapData.hasOthers()) {
							final Font railFont = this.mapPreferences.getRailwayPreferences()
									.getFont();
							g2.setFont(railFont);
							FontMetrics metrics = this.getFontMetrics(railFont);
							int descent = metrics.getDescent();
							g2.setColor(this.mapPreferences.getRailwayPreferences()
									.getAttributeColor());
							for (ArcData arc : mapData.getOthers().values()) {
								if (arc.getAttribute() != null) {
									if (arc.getAttributeX() != 0 && arc.getAttributeY() != 0) {
										g2.drawString(arc.getAttribute(), (float) ((arc
												.getAttributeX() * this.zoom) - this.offsetX),
												(float) ((arc.getAttributeY() * this.zoom)
														- this.offsetY - descent));
									}
								}
							}
						}
					}
				}
				if ((System.currentTimeMillis() - startTime2) > thresholdTime) {
					System.out.println("DEBUG: （属性の描画：" + (System.currentTimeMillis() - startTime2)
							+ " ms）");
				}
			}
		} catch (ConcurrentModificationException e) {
		} catch (Exception e) {
			System.err.println("ERROR: Failed to draw map.");
			e.printStackTrace(System.err);
		}
		this.setIsBusy(false);
	}

	/**
	 * @return 文字の大きさ
	 */
	public double getFontZoom() {
		return this.fontZoom;
	}

	/**
	 * @return 地図の設定
	 */
	public MapPreferences getMapPreferences() {
		return this.mapPreferences;
	}

	/**
	 * オブジェクトが存在する最も大きい x 座標を取得します。
	 * @return オブジェクトが存在する最も大きい x 座標
	 */
	double getMaxX() {
		return this.maxX;
	}

	/**
	 * オブジェクトが存在する最も大きい y 座標を取得します。
	 * @return オブジェクトが存在する最も大きい y 座標
	 */
	double getMaxY() {
		return this.maxY;
	}

	/**
	 * オブジェクトが存在する最も小さい x 座標を取得します。
	 * @return オブジェクトが存在する最も小さい x 座標
	 */
	double getMinX() {
		return this.minX;
	}

	/** オブジェクトが存在する最も小さい y 座標を取得します。
	 * @return オブジェクトが存在する最も小さい x 座標
	 */
	double getMinY() {
		return this.minY;
	}

	/**
	 * @return 表示倍率の最小値
	 */
	public double getMinZoom() {
		return this.minZoom;
	}

	/**
	 * オブジェクトが存在する範囲を取得します。
	 * @return オブジェクトが存在する範囲（仮想座標）
	 */
	Rectangle2D getObjectArea() {
		return new Rectangle2D.Double(this.minX, this.minY, this.maxX - this.minX, this.maxY
				- this.minY);
	}

	/**
	 * @return x方向のオフセット
	 */
	public double getOffsetX() {
		return this.offsetX;
	}

	/**
	 * @return y方向のオフセット
	 */
	public double getOffsetY() {
		return this.offsetY;
	}

	/**
	 * @return 都道府県の一覧
	 */
	public Collection<Prefecture> getPrefectures() {
		return this.prefectures;
	}

	/**
	 * @return 地図を検索するためのデータ構造
	 */
	public Search getSearch() {
		return this.search;
	}

	/**
	 * 表示されている範囲を取得します。
	 * @return 表示されている範囲（仮想座標）
	 */
	Rectangle2D getVisibleRectangle() {
		final int width = (this.size == null) ? getWidth() : this.size.width;
		final int height = (this.size == null) ? getHeight() : this.size.height;
		return new Rectangle2D.Double(this.offsetX / this.zoom, this.offsetY / this.zoom, width
				/ this.zoom, height / this.zoom);
	}

	/**
	 * 倍率を取得します。
	 * @return 倍率
	 */
	public double getZoom() {
		return this.zoom;
	}

	/**
	 * 文字を大きくします。
	 */
	public void increaseFontSize() {
		this.fontZoom /= 0.9;
		this.mapPreferences.setFontZoom(this.fontZoom);
		if (this.listener != null) {
			this.listener.actionPerformed(new ActionEvent(this, this.hashCode(),
					"increment font size"));
		}
	}

	/**
	 * 彩度を増やします。
	 */
	public void increaseSaturation() {
		this.saturationDifference += 0.05;
		this.setNeedsRepaint(true);
		repaint();
	}

	/**
	 * ユーザ操作ができない状態かどうかを取得します。
	 * @return ユーザ操作ができない状態かどうか
	 */
	boolean isBusy() {
		return this.isBusy;
	}

	/**
	 * パネルが操作されていない状態かどうかを取得します。
	 * @return パネルが操作されていない状態かどうか
	 */
	boolean isIdle() {
		return this.isIdle;
	}

	/**
	 * 指定したオブジェクトが表示エリア内にあるかどうかを取得します。
	 * @param shape オブジェクト
	 * @return 指定したオブジェクトが表示エリア内にあるかどうか
	 */
	boolean isVisible(Shape shape) {
		return shape.intersects(getVisibleRectangle());
	}

	/**
	 * 倍率が変更されたかどうかを取得します。
	 * @return 倍率が変更されたかどうか
	 */
	boolean isZoomChanged() {
		if (this.isZoomChanged) {
			this.isZoomChanged = false;
			return true;
		} else {
			return false;
		}
	}

	/**
	 * 指定した点が画面の中央になるように、地図をスクロールさせます。
	 * @param x x座標（仮想座標）
	 * @param y y座標（仮想座標）
	 */
	public void moveTo(final double x, final double y) {
		final int width = (this.size == null) ? getWidth() : this.size.width;
		final int height = (this.size == null) ? getHeight() : this.size.height;
		this.offsetX = x * this.zoom - width / 2;
		this.offsetY = y * this.zoom - height / 2;
		setNeedsRepaint(true);
	}

	/**
	 * 地図の中央が画面の中央になるように、地図をスクロールさせます。
	 */
	public void moveToCenter() {
		final int width = (this.size == null) ? getWidth() : this.size.width;
		final int height = (this.size == null) ? getHeight() : this.size.height;
		this.offsetX = ((this.minX + this.maxX) / 2 * this.zoom) - width / 2;
		this.offsetY = ((this.minY + this.maxY) / 2 * this.zoom) - height / 2;
		setNeedsRepaint(true);
	}

	@Override
	public void paintComponent(Graphics g) {
		long startTime = System.currentTimeMillis();
		final long thresholdTime = 200;
		if (!this.isIdle || !this.needsRepaint) {
			if (this.image != null) {
				g.setColor(this.mapPreferences.getBackGroundColor());
				g.fillRect(0, 0, getWidth(), getHeight());
				g.drawImage(this.image, this.mouseMotionWidth, this.mouseMotionHeight, this);
				if (this.mouseMotionWidth != 0 || this.mouseMotionHeight != 0) {
					final Area clip = new Area(g.getClip());
					clip.subtract(new Area(new Rectangle(this.mouseMotionWidth,
							this.mouseMotionHeight, getWidth(), getHeight())));
					g.setClip(clip);
					drawMap(thresholdTime, (Graphics2D) g);
				}
			}
			return;
		}
		setNeedsRepaint(false);
		this.image = createImage(getWidth(), getHeight());
		Graphics2D g2 = (Graphics2D) this.image.getGraphics();
		drawMap(thresholdTime, g2);
		g.drawImage(this.image, 0, 0, this);
		if ((System.currentTimeMillis() - startTime) > thresholdTime) {
			System.out.println("DEBUG: 描画：" + (System.currentTimeMillis() - startTime + " ms"));
		}
		this.lastOffsetX = this.offsetX;
		this.lastOffsetY = this.offsetY;
		this.lastZoom = this.zoom;
		this.lastVisibleRectangle = this.getVisibleRectangle();
	}

	/**
	 * 現在表示されている地図をダイアログを表示して印刷します。
	 * @throws PrintException 印刷例外
	 */
	public void print() throws PrintException {
		PrintUtil.print(this);
	}

	public int print(Graphics graphics, PageFormat pageFormat, int pageIndex)
			throws PrinterException {
		if (pageIndex == 0) {
			final Graphics2D g = (Graphics2D) graphics;
			g.translate(pageFormat.getImageableX(), pageFormat.getImageableY());
			final double newZoom = Math.min(pageFormat.getImageableWidth() / getWidth(), pageFormat
					.getImageableHeight()
					/ getHeight());
			g.scale(newZoom, newZoom);
			g.setClip(0, 0, getWidth(), getHeight());
			drawMap(200, g);
			return Printable.PAGE_EXISTS;
		} else {
			return Printable.NO_SUCH_PAGE;
		}
	}

	/**
	 * 現在表示されている地図をビットマップファイルに出力します。
	 * @param file ファイル
	 * @param format ファイル形式（png、jpg、bmp）
	 * @throws IOException 入出力例外 
	 */
	public void printBitmap(final File file, final String format) throws IOException {
		BufferedImage tempImage = new BufferedImage(this.getWidth(), this.getHeight(),
				BufferedImage.TYPE_INT_RGB);
		drawMap(200, (Graphics2D) tempImage.getGraphics());
		ImageIO.write(tempImage, format, file);
	}

	/**
	 * 現在表示されている地図をPSファイルに出力します。
	 * @param file ファイル
	 * @throws IOException ファイル入出力例外
	 * @throws PrinterException 印刷例外
	 */
	public void printPS(final File file) throws PrinterException, IOException {
		PSOut.print(file, this);
	}

	/**
	 * 文字の大きさを標準に戻します。
	 */
	public void resetFontSize() {
		this.fontZoom = 1;
		this.mapPreferences.setFontZoom(this.fontZoom);
		if (this.listener != null) {
			this.listener
					.actionPerformed(new ActionEvent(this, this.hashCode(), "reset font size"));
		}
	}

	/**
	 * 彩度をリセットします。
	 */
	public void resetSaturation() {
		this.saturationDifference = 0;
		this.setNeedsRepaint(true);
		repaint();
	}

	/**
	 * スクロールします。
	 * @param dx x軸方向の移動距離（実座標）
	 * @param dy y軸方向の移動距離（実座標）
	 */
	public void scroll(double dx, double dy) {
		this.offsetX += dx;
		this.offsetY += dy;
		if (this.listener != null) {
			this.listener.actionPerformed(new ActionEvent(this, this.hashCode(), "move"));
		}
	}

	/**
	 * @param listener 背景スレッドに再計算を要求するためのアクションリスナ
	 */
	public void setActionListner(final ActionListener listener) {
		this.listener = listener;
	}

	/**
	 * @return 背景スレッドに再計算を要求するためのアクションリスナ
	 */
	public ActionListener getActionListener() {
		return this.listener;
	}

	/**
	 * @param isCancel 描画を中止するかどうか
	 */
	public void setCancel(boolean isCancel) {
		this.isCancel = isCancel;
	}

	/**
	 * パネルが操作されていない状態かどうかを設定します。
	 * @param b パネルが操作されていない状態かどうか
	 */
	void setIdle(boolean b) {
		this.isIdle = b;
	}

	/**
	 * ユーザ操作ができない状態かどうかを設定します。
	 * @param b ユーザ操作ができない状態かどうか
	 */
	public void setIsBusy(boolean b) {
		this.isBusy = b;
		if (b) {
			setCursor(new Cursor(Cursor.WAIT_CURSOR));
		} else {
			setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
		}
	}

	/**
	 * 指定された地図全てが収まる長方形を求めます。
	 * @param mapNames 地図の一覧
	 * @return 長方形
	 * @since 3.03
	 */
	public Rectangle2D getBounds(final Collection<String> mapNames) {
		Rectangle2D ret = null;
		for (final String mapName : mapNames) {
			if (this.maps.containsKey(mapName)) {
				final Rectangle bounds = this.maps.get(mapName).getBounds().getBounds();
				if (ret == null) {
					ret = bounds;
				} else {
					ret = ret.createUnion(bounds);
				}
			}
		}
		return ret;
	}
	
	/**
	 * @param minZoom 表示倍率の最小値
	 */
	public void setMinZoom(double minZoom) {
		this.minZoom = minZoom;
	}

	/**
	 * 再描画が必要かどうかを設定します。
	 * @param b 再描画が必要かどうか
	 */
	public void setNeedsRepaint(boolean b) {
		this.needsRepaint = b;
	}

	/**
	 * オフセットを設定します。
	 * @param offsetX x方向のオフセット
	 * @param offsetY y方向のオフセット
	 */
	public void setOffset(double offsetX, double offsetY) {
		this.offsetX = offsetX;
		this.offsetY = offsetY;
	}

	/**
	 * @param search 地図を検索するためのデータ構造
	 */
	public void setSearch(Search search) {
		this.search = search;
	}

	/**
	 * SWT版ではパネルの大きさが取得できないので、強制的に設定します。
	 * @param size パネルの大きさ
	 */
	public void setSWTSize(Dimension size) {
		this.size = size;
	}

	/**
	 * 縮尺を設定します。
	 * @param zoom 縮尺
	 */
	public void setZoom(double zoom) {
		this.zoom = zoom;
	}

	/**
	 * 実座標を取得します。
	 * @param location 仮想座標
	 * @return 実座標
	 */
	Point2D toRealLocation(Point2D location) {
		return new Point2D.Double((location.getX() * this.zoom) - this.offsetX,
				(location.getY() * this.zoom) - this.offsetY);
	}

	/**
	 * 仮想座標を取得します。
	 * @param location 実座標
	 * @return 仮想座標
	 */
	Point2D toVirtualLocation(Point2D location) {
		return new Point2D.Double((this.offsetX + location.getX()) / this.zoom,
				(this.offsetY + location.getY()) / this.zoom);
	}

	/**
	 * 地図の縮尺を設定します。
	 * @param newZoom 縮尺
	 */
	void zoom(double newZoom) {
		this.zoom = newZoom;
		this.isZoomChanged = true;
		setNeedsRepaint(true);
	}

	/**
	 * 倍率を変更します。
	 * @param newZoom 倍率
	 * @param x 中心のx座標（実座標）
	 * @param y 中心のy座標（実座標）
	 */
	private void zoom(final double newZoom, final int x, final int y) {
		final double newX = ((MapPanel.this.offsetX + x) / MapPanel.this.zoom * newZoom) - x;
		final double newY = ((MapPanel.this.offsetY + y) / MapPanel.this.zoom * newZoom) - y;
		MapPanel.this.offsetX = newX;
		MapPanel.this.offsetY = newY;
		MapPanel.this.zoom = newZoom;
		if (!isBusy() && isIdle()) {
			Image tempImage = createImage(getWidth(), getHeight());
			Graphics2D g = (Graphics2D) tempImage.getGraphics();
			g.setColor(MapPanel.this.mapPreferences.getBackGroundColor());
			g.fillRect(0, 0, getWidth(), getHeight());
			g.translate(-MapPanel.this.offsetX, -MapPanel.this.offsetY);
			g.scale(MapPanel.this.zoom / MapPanel.this.lastZoom, MapPanel.this.zoom
					/ MapPanel.this.lastZoom);
			g.translate(MapPanel.this.lastOffsetX, MapPanel.this.lastOffsetY);
			g.drawImage(MapPanel.this.image, 0, 0, MapPanel.this);
			getGraphics().drawImage(tempImage, 0, 0, MapPanel.this);
		}
		MapPanel.this.isZoomChanged = true;
		setNeedsRepaint(true);
	}

	/**
	 * 自動倍率設定します。
	 */
	public void zoomAutomaticaly() {
		final int width = (this.size == null) ? getWidth() : this.size.width;
		final int height = (this.size == null) ? getHeight() : this.size.height;
		final double zoomX = width / (this.maxX - this.minX);
		final double zoomY = height / (this.maxY - this.minY);
		if (zoomY < zoomX) {
			this.zoom = zoomY;
		} else {
			this.zoom = zoomX;
		}
		this.minZoom = this.zoom;
		this.isZoomChanged = true;
		setNeedsRepaint(true);
	}
	
	/**
	 * 指定された長方形が表示されるように表示倍率を変更し、表示位置を移動します。
	 * @param rectangle 長方形
	 * @since 3.03
	 */
	public void moveTo(final Rectangle2D rectangle) {
		final int width = (this.size == null) ? getWidth() : this.size.width;
		final int height = (this.size == null) ? getHeight() : this.size.height;
		final double zoomX = width / rectangle.getWidth();
		final double zoomY = height / rectangle.getHeight();
		if (zoomY < zoomX) {
			this.zoom = zoomY;
		} else {
			this.zoom = zoomX;
		}
		this.isZoomChanged = true;
		this.moveTo(rectangle.getCenterX(), rectangle.getCenterY());
	}

	/**
	 * 詳細表示します。
	 */
	public void zoomDetail() {
		zoom(Const.ZOOM_LOAD_ALL, getWidth() / 2, getHeight() / 2);
		if (this.listener != null) {
			this.listener.actionPerformed(new ActionEvent(this, this.hashCode(), "zoom whole"));
		}
	}

	/**
	 * 拡大します。
	 */
	public void zoomIn() {
		doWheelRotation(1, getWidth() / 2, getHeight() / 2);
		if (this.listener != null) {
			this.listener.actionPerformed(new ActionEvent(this, this.hashCode(), "zoom in"));
		}
	}

	/**
	 * 縮小します。
	 */
	public void zoomOut() {
		doWheelRotation(-1, getWidth() / 2, getHeight() / 2);
		if (this.listener != null) {
			this.listener.actionPerformed(new ActionEvent(this, this.hashCode(), "zoom out"));
		}
	}

	/**
	 * 全域表示します。
	 */
	public void zoomWhole() {
		zoomAutomaticaly();
		moveToCenter();
		if (this.listener != null) {
			this.listener.actionPerformed(new ActionEvent(this, this.hashCode(), "zoom whole"));
		}
	}

	/**
	 * 広域表示します。
	 */
	public void zoomWide() {
		zoom(Const.ZOOM_LOAD_GYOUSEI, getWidth() / 2, getHeight() / 2);
		if (this.listener != null) {
			this.listener.actionPerformed(new ActionEvent(this, this.hashCode(), "zoom whole"));
		}
	}
}
