package isj;

import java.awt.geom.Point2D;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URL;
import java.util.Enumeration;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Scanner;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;
import map.UTMUtil;
import web.WebUtilities;

/**
 * 街区レベル位置参照情報を取得するユーティリティクラスです。
 * @author Kumano Tatsuo
 * 2005/12/03
 */
public class ISJUtil {
	/**
	 * 古い展開済みファイルの接尾語
	 */
	public static final String OLD_CSV_SUFFIX = "_2003.csv";

	/**
	 * 古い圧縮ファイルの接尾語
	 */
	public static final String OLD_ZIP_SUFFIX = "-02.0a.zip";

	/**
	 * 街区レベル位置参照情報の配布URL
	 */
	public static final String BASE_URL = "http://nlftp.mlit.go.jp/isj/dls/data/";

	/**
	 * キャッシュディレクトリの相対パス
	 */
	public static final String CACHE_DIR = ".map" + File.separator + "isj";

	/**
	 * 圧縮ファイルの接頭語
	 */
	public static final String ZIP_PREFIX = "03.0a/";

	/**
	 * 圧縮ファイルの接尾語
	 */
	public static final String ZIP_SUFFIX = "-03.0a.zip";

	/**
	 * 展開済みファイルの接尾語
	 */
	public static final String CSV_SUFFIX = "_2004.csv";

	/**
	 * 古い圧縮ファイルの接頭語
	 */
	public static final String OLD_ZIP_PREFIX = "02.0a/";

	/**
	 * 街区レベル位置参照情報をダウンロードし、読み込みます。
	 * @param id 市区町村コード
	 * @return 街区レベル位置参照情報
	 * @throws FileNotFoundException 
	 * @throws IOException 
	 */
	public static Map<String, Point2D> loadIsj(final String id) throws IOException,
			FileNotFoundException {
		final Map<String, Point2D> ret = new LinkedHashMap<String, Point2D>();
		File csvFile = new File(ISJUtil.CACHE_DIR + File.separator + id + ISJUtil.CSV_SUFFIX);
		final File oldCsvFile = new File(ISJUtil.CACHE_DIR + File.separator + id
				+ ISJUtil.OLD_CSV_SUFFIX);
		final URL url = new URL(ISJUtil.BASE_URL + ISJUtil.ZIP_PREFIX + id + ISJUtil.ZIP_SUFFIX);
		final URL oldUrl = new URL(ISJUtil.BASE_URL + ISJUtil.OLD_ZIP_PREFIX + id
				+ ISJUtil.OLD_ZIP_SUFFIX);
		if (csvFile.exists()) {
			// 平成16年の展開済みファイルがあるとき
			System.out.println("DEBUG: skipped getting " + url);
		} else {
			if (oldCsvFile.exists()) {
				// 平成15年の展開済みファイルがあるとき
				System.out.println("DEBUG: skipped getting " + oldUrl);
				csvFile = oldCsvFile;
			} else {
				final File cacheDir = new File(ISJUtil.CACHE_DIR);
				if (!cacheDir.exists()) {
					cacheDir.mkdir();
				}
				try {
					url.openStream();
					// 平成16年の圧縮ファイルをダウンロードできるとき
					final File file = new File(ISJUtil.CACHE_DIR + File.separator + id
							+ ISJUtil.ZIP_SUFFIX);
					file.createNewFile();
					System.out.println("DEBUG: getting " + url);
					get(url, file);
					csvFile = new File(ISJUtil.CACHE_DIR + File.separator + id + ISJUtil.CSV_SUFFIX);
				} catch (FileNotFoundException e) {
					try {
						System.out.println("WARNING: failed to get " + url);
						oldUrl.openStream();
						// 平成15年の圧縮ファイルをダウンロードできるとき
						final File file = new File(ISJUtil.CACHE_DIR + File.separator + id
								+ ISJUtil.OLD_ZIP_SUFFIX);
						file.createNewFile();
						System.out.println("DEBUG: getting " + oldUrl);
						get(oldUrl, file);
						csvFile = new File(ISJUtil.CACHE_DIR + File.separator + id
								+ ISJUtil.OLD_CSV_SUFFIX);
					} catch (FileNotFoundException e1) {
						System.out.println("WARNING: failed to get " + oldUrl);
						e1.printStackTrace();
					}
				}
			}
		}
		{
			final Scanner scanner = new Scanner(new InputStreamReader(new FileInputStream(csvFile),
					"SJIS"));
			boolean isFirst = true;
			while (scanner.hasNextLine()) {
				final String line = scanner.nextLine();
				if (isFirst) {
					isFirst = false;
				} else {
					final Scanner scanner2 = new Scanner(line);
					scanner2.useDelimiter(",");
					final StringBuilder string = new StringBuilder();
					string.append(scanner2.next().replaceAll("\"", "") + ",");
					string.append(scanner2.next().replaceAll("\"", "") + ",");
					string.append(scanner2.next().replaceAll("\"", "") + ",");
					string.append(scanner2.next().replaceAll("\"", ""));
					scanner2.next();
					scanner2.next();
					scanner2.next();
					final String latitude = scanner2.next();
					final String longitude = scanner2.next();
					scanner2.next();
					final int rep = scanner2.nextInt();
					if (rep == 1) {
						if (longitude.length() == 10 && latitude.length() == 9) {
							final Point2D point = UTMUtil.toUTM(Double.parseDouble(longitude),
									-Double.parseDouble(latitude));
							ret.put(string.toString(), point);
						} else {
							System.out.println("WARNING: invalid longitude or latitude: " + line);
						}
					}
				}
			}
			scanner.close();
		}
		return ret;
	}

	/**
	 * ZIPファイルを取得して展開します。
	 * @param url URL
	 * @param file 出力ファイル
	 * @throws IOException
	 * @throws FileNotFoundException
	 * @throws ZipException
	 */
	private static void get(final URL url, final File file) throws IOException,
			FileNotFoundException, ZipException {
		WebUtilities.copy(url.openStream(), new FileOutputStream(file));
		final ZipFile zipFile = new ZipFile(file);
		for (final Enumeration<? extends ZipEntry> enumeration = zipFile.entries(); enumeration
				.hasMoreElements();) {
			final ZipEntry entry = enumeration.nextElement();
			if (entry.getName().endsWith(".csv")) {
				WebUtilities.copy(zipFile.getInputStream(entry), new FileOutputStream(
						ISJUtil.CACHE_DIR + File.separator + new File(entry.getName())));
			}
		}
	}
}
