package map;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionAdapter;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.TimerTask;
import zipcode.ZipCode;

/**
 * 地図の読み込み、ポリゴンの結合、塗り分け、属性配置位置の計算、再描画など、
 * バックグラウンドで行う作業を管理するクラスです。
 * @author Kumano Tatsuo
 * 作成日: 2004/01/19
 */
public class BackgroundThread extends TimerTask {
	/**
	 * 地図
	 */
	private final Map<String, MapData> maps; // 地図

	/**
	 * 都道府県の一覧
	 */
	private final Collection<Prefecture> prefectures;

	/**
	 * 地図を表示するパネル
	 */
	private final MapPanel panel;

	/**
	 * 地図の状態が変化したかどうか
	 */
	boolean isChanged; // 地図の状態が変化したかどうか

	/**
	 * 地図を読み込むためのオブジェクト
	 */
	private final LoadMap loadMap;

	/**
	 * 地図の数
	 */
	private int mapSize;

	/**
	 * バックグラウンドで行う処理を初期化します。
	 * @param maps 地図
	 * @param panel パネル
	 * @param loadMap 地図を読み込むためのオブジェクト
	 */
	public BackgroundThread(final Map<String, MapData> maps, final MapPanel panel,
		final LoadMap loadMap) {
		this.maps = maps;
		this.panel = panel;
		this.prefectures = this.panel.getPrefectures();
		this.isChanged = true;
		this.loadMap = loadMap;
		this.mapSize = maps.size();
		panel.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseReleased(MouseEvent e) {
				BackgroundThread.this.isChanged = true;
			}
		});
		panel.addMouseMotionListener(new MouseMotionAdapter() {
			@Override
			public void mouseDragged(MouseEvent e) {
				BackgroundThread.this.isChanged = true;
			}
		});
		panel.addMouseWheelListener(new MouseWheelListener() {
			public void mouseWheelMoved(MouseWheelEvent e) {
				BackgroundThread.this.isChanged = true;
			}
		});
		panel.addComponentListener(new ComponentAdapter() {
			@Override
			public void componentResized(ComponentEvent e) {
				BackgroundThread.this.isChanged = true;
			}
		});
		panel.setActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				BackgroundThread.this.isChanged = true;
			}
		});
	}

	/**
	 * 地図の読み込み、ポリゴンの結合、塗り分けを行います。
	 */
	void loadMapPaintTyomeJoinTyome() {
		try {
			this.panel.addMessage("地図を読み込んでいます。");
			Prefectures.loadCities(this.prefectures, this.panel, this.maps, this.loadMap);
			synchronized (this.maps) {
				this.loadMap.loadMap(this.maps, this.panel);
				loadYomi();
				PaintTyome paintTyome = new PaintTyome();
				paintTyome.paintTyome(this.maps);
				JoinPolygon joinTyome = new JoinPolygon();
				joinTyome.joinPolygon(this.maps, this.panel.getVisibleRectangle());
			}
			JoinTatemono joinTatemono = new JoinTatemono();
			joinTatemono.joinTatemono(this.maps);
			this.panel.removeMessage();
		} catch (Exception exception) {
			System.err.println("EXCEPTION: Failed to load map.");
			exception.printStackTrace();
		}
	}

	/**
	 * 町丁目の読みを調べます。
	 * @throws IOException
	 */
	private void loadYomi() throws IOException {
		if (this.panel.getZoom() >= Const.Zoom.LOAD_ALL) {
			final Map<String, ZipCode> zipCodes = new HashMap<String, ZipCode>();
			for (final Prefecture prefecture : this.prefectures) {
				if (prefecture.hasFine()) {
					if (this.panel.isVisible(prefecture.getFineShape())) {
						if (prefecture.hasFine()) {
							for (final City city : prefecture.getCities()) {
								if (city.hasFineShape()) {
									if (this.panel.isVisible(city.getFineShape())) {
										if (!zipCodes.containsKey(city.getId())) {
											zipCodes.put(city.getId(), new ZipCode(city
												.getId()));
										}
									}
								}
							}
						}
					}
				}
			}
			for (final MapData map : this.maps.values()) {
				if (map.hasTyome()) {
					for (final ZipCode zipCode : zipCodes.values()) {
						for (final PolygonData polygon : map.getTyome().values()) {
							if (!polygon.hasYomi()) {
								final String attribute = polygon.getAttribute()
									.replaceFirst("[0-9０-９]+$", "");
								final String yomi = zipCode.getYomi(attribute);
								if (yomi != null) {
									polygon.setYomi(yomi);
								}
							}
						}
					}
				}
			}
		}
	}

	@Override
	public void run() {
		if (this.maps.size() != this.mapSize) {
			this.isChanged = true;
			this.mapSize = this.maps.size();
		}
		if (this.isChanged) {
			this.isChanged = false;
			loadMapPaintTyomeJoinTyome();
			this.panel.setChanged();
			this.panel.repaint();
		}
	}
}
