package shop;

import isj.ISJUtil;
import java.awt.geom.Point2D;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.net.URLEncoder;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Scanner;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import map.MapPanel;

/**
 * タイムズの住所を取得するクラスです。
 * @author Kumano Tatsuo
 * @since 4.04
 * 2006/07/17
 */
public class Times {
	/**
	 * 最初のURL
	 */
	private static final String URL = "http://map.times-info.net/map/spot_addr.php?kind=1&key=";

	/**
	 * 最後のURL
	 */
	private static final String URL2 = "&lk=0&pg=1";
	
	/**
	 * エンコーディング
	 */
	private static final String ENCODING = "EUC-JP";

	/**
	 * キャッシュを保存するディレクトリ
	 */
	private static final String CACHE_DIR = ".map" + File.separator + "shops";

	/**
	 * キャッシュファイル名の接頭語
	 */
	private static final String PREFIX = "times_";

	/**
	 * キャッシュファイル名の接尾語
	 */
	private static final String SUFFIX = ".csv";

	/**
	 * タイムズの座標と店舗名の対応表をホームページ又はキャッシュから取得します。
	 * @param id 市区町村コード
	 * @param city 市区町村名
	 * @param prefecture 都道府県名
	 * @param isj 街区レベル位置参照情報
	 * @param panel 地図を描画するパネル
	 * @return 座標と店舗名の対応表
	 * @since 4.04
	 * @throws IOException 
	 */
	public static Map<Point2D, String> load(final String id, final String city,
			final String prefecture, final Map<String, Point2D> isj, final MapPanel panel) throws IOException {
		final Map<Point2D, String> ret = new LinkedHashMap<Point2D, String>();
		if (!new File(Times.CACHE_DIR).exists()) {
			new File(Times.CACHE_DIR).mkdirs();
		}
		final String cacheFile = Times.CACHE_DIR + File.separator + Times.PREFIX + id + Times.SUFFIX;
		if (!new File(cacheFile).exists()) {
			final PrintWriter out = new PrintWriter(new File(cacheFile), "SJIS");
			ISJUtil.parseAddresses(Times.getAddresses(city, prefecture, panel), out, isj);
			out.close();
		}
		final Scanner scanner = new Scanner(new InputStreamReader(new FileInputStream(new File(
				cacheFile)), "SJIS"));
		while (scanner.hasNextLine()) {
			final String line = scanner.nextLine();
			final String[] items = line.split(",");
			if (items.length == 4) {
				final double x = Double.parseDouble(items[2]);
				final double y = Double.parseDouble(items[3]);
				ret.put(new Point2D.Double(x, y), "タイムズ");
			} else {
				System.out.println("WARNING: データ形式が不正です。" + line);
			}
		}
		scanner.close();
		return ret;
	}

	/**
	 * タイムズの市区町村別ページを解析して住所の一覧を取得します。
	 * @param city 市区町村名
	 * @param prefecture 都道府県名
	 * @param panel 地図を描画するパネル
	 * @return 住所と店舗名の対応表
	 * @since 4.04
	 * @throws IOException 
	 * @throws UnsupportedEncodingException 
	 */
	public static Map<String, String> getAddresses(final String city, final String prefecture, final MapPanel panel)
			throws IOException {
		final Map<String, String> ret = new LinkedHashMap<String, String>();
		try {
			final URL url = new URL(URL + URLEncoder.encode(prefecture + city, Times.ENCODING) + URL2);
			panel.addMessage(url + "をダウンロードしています。");
			final Scanner scanner = new Scanner(new InputStreamReader(url.openStream(), ENCODING));
			final Pattern pattern1 = Pattern.compile("<p>([^0-9a-zA-Z]+[^<>]+)</p>");
			final Pattern pattern2 = Pattern.compile("<p><a href=[^<>]+>([^<>]+)</a></p>");
			String shopName = null;
			while (scanner.hasNextLine()) {
				final String line = scanner.nextLine();
				//System.out.println("Times: line = " + line);
				final Matcher matcher1 = pattern1.matcher(line);
				final Matcher matcher2 = pattern2.matcher(line);
				if (matcher1.find()) {
					String address = matcher1.group(1);
					if (!address.startsWith("検索結果") &&
							!address.startsWith("住所") &&
							!address.equals("駐車場名称") &&
							!address.equals("満空情報") &&
							!address.equals("時間貸台数") &&
							!address.equals("料金情報") &&
							!address.equals("住所")) {
						if (shopName != null) {
							ret.put(matcher1.group(1), shopName);
							shopName = null;
						}
					}
				} else if (matcher2.find()) {
					shopName = matcher2.group(1);
				}
			}
			scanner.close();
			panel.removeMessage();
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
		return ret;
	}
}
