################################################################################
##$Namespace: process
##$Summary: Provides components for dealing with operating system processes
##$Version: 1.0.0
##$Author: Michael E Allen
##$Copyright: Copyright(C)2001 Michael E Allen
##$License: GNU General Public License
##$Create Date: April 23, 2001
##$Description: 
##$Bugs: 
##+	Sometimes ::process::Get doesn't work
##$To Do: 
##+	Nothing
##$Changes:
##+  August 2, 2001(Michael E Allen):
##+		Renamed ::process::Kill to ::process::Signal
##+		Changed ::process::Signal parameters
##+  July 31, 2001(Michael E Allen):
##+		Added try/catch logic to all procedures
##$Disclaimer:
##+	This program is free software; you can redistribute it and/or modify
##+	it under the terms of the GNU General Public License as published by
##+	the Free Software Foundation; either version 2 of the License, or
##+	(at your option) any later version.
##+	This program is distributed in the hope that it will be useful,
##+	but WITHOUT ANY WARRANTY; without even the implied warranty of
##+	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##+	GNU General Public License for more details.
##+	
##+	You should have received a copy of the GNU General Public License
##+	along with this program; if not, write to the Free Software
##+	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
################################################################################
namespace eval process {
	package require Expect
	
	exception ProcessException Exception

	################################################################################
	##$Procedure: Signal
	##$Summary: Sends a signal to a running process
	##$Version: 1.0.0
	##$Author: Michael E Allen
	##$Create Date: April 23, 2001
	##$Description:
	##+	None
	##$Bugs: 
	##+	None
	##$To Do: 
	##+	Nothing
	##$Changes:
	##+	August 2, 2001(Michael E Allen):
	##+		Renamed from Kill to Signal
	##+		Changed order of parameters, can now be accepted in any order
	##+		Changed format of the 'signal' parameter, must now have a leading '-'
	##$Synopsis:
	##+	::process::Kill ?-pgroup? ?-signum? ?process_id_list?
	##+	::process::Kill ?-pgroup? ?-signum? ?process_id? 
	##$Parameters:
	##+	signum:
	##+		Specifies the signal to send to the process.  You can specify 
	##+		either a name, stripped of the SIG prefix (such as KILL), 
	##+		or a number (such as 9).  For information about signal names 
	##+		and numbers, see your operating system documentation
	##$Result:
	##+	Returns a list of any process that could not be signaled
	##+	TCL List in the form
	##+		{process_id error_message}
	################################################################################
	proc Signal {args} {
		set lPGroup 0
		set lSignal 15
		
		foreach lOption $args {
			if {($lOption == "-pgroup")} {
				set lPGroup 1
			} elseif {([string match "-*" $lOption])} {
				set lSignal $lOption
			} else {
				lappend lProcessIDList [eval concat $lOption]
			}
		}

		set lProcessIDList [eval concat $lProcessIDList]
		
		set lResult ""
		
		foreach lProcessID $lProcessIDList {
			if {($lPGroup == 1)} {
				set lProcessID "-$lProcessID"
			}
			if {([catch {set lError [exec /bin/kill $lSignal $lProcessID]} lCatch])} {
				lappend lResult [list process_id $lProcessID error_message $lCatch]
			}
		}
	
		return $lResult
	}
	
	################################################################################
	##$Procedure: Get
	##$Summary: Gets current process status
	##$Version: 1.0.0
	##$Author: Michael E Allen
	##$Create Date: April 23, 2001
	##$Description:
	##+	None
	##$Bugs: 
	##+	Sometimes fails
	##$To Do: 
	##+	Nothing
	##$Changes:
	##+	None
	##$Synopsis:
	##+	::process::Get process_info
	##$Parameters:
	##+	process_info
	##+		Text string containing user id, process id, command name, or parent id
	##$Result:
	##+	Returns a list of any process who matched the process_info string, in the
	##+	form
	##+		{process_id user_id parent_process_id command}
	################################################################################
	proc Get {pProcessInfo} {
		set lProcessList [exec /bin/ps -e -o "uid pid ppid cmd"]
		set lProcessList [split $lProcessList "\n"]
	
		set lNewProcessList ""
		
		foreach lProcess $lProcessList {
			if {([lsearch -glob $lProcess "*$pProcessInfo*"] != -1)} {
				set lNewProcess ""
				foreach lElement [split [string trim $lProcess] " "] {
					if {($lElement != "")} {
						lappend lNewProcess $lElement
					}
				}
	
				if {($lNewProcess != "")} {
					set lNewProcess [linsert $lNewProcess 0 "process_id"]
					set lNewProcess [linsert $lNewProcess 2 "user_id"]
					set lNewProcess [linsert $lNewProcess 4 "parent_process_id"]
					set lNewProcess [linsert $lNewProcess 6 "command"]
					set lCommand [join [lrange $lNewProcess 7 end] " "]
					set lNewProcess [lreplace $lNewProcess 7 end $lCommand]
					lappend lNewProcessList $lNewProcess
				}
			}
		}
		return $lNewProcessList
	}
		
	################################################################################
	##$Procedure: Top
	##$Summary: 
	##$Version: 1.0.0
	##$Author: Michael E Allen
	##$Create Date: April 23, 2001
	##$Description:
	##+	None
	##$Bugs: 
	##+	None
	##$To Do: 
	##+	Nothing
	##$Changes:
	##+	None
	##$Synopsis:
	##+	::process::Top
	##$Parameters:
	##+	None
	##$Result:
	##+	Not done
	################################################################################
	proc Top {} {
		set lTopList [exec /usr/local/bin/top]
		set lTopList [split $lTopList "\n"]
		
		set lTopList [lreplace $lTopList 0 6]
		set lTopList [lreplace $lTopList end end]
		
		return $lTopList
	}
	
	################################################################################
	##$Procedure: CountInstances
	##$Summary: Count the number of times the same program is running
	##$Version: 1.0.0
	##$Author: Michael E Allen
	##$Create Date: April 23, 2001
	##$Description:
	##+	None
	##$Bugs: 
	##+	None
	##$To Do: 
	##+	Nothing
	##$Changes:
	##+	None
	##$Synopsis:
	##+	::process::CountInstances process_info
	##$Parameters:
	##+	process_info
	##+		Text string containing user id, process id, command name, or parent id
	##$Result:
	##+	A number indicating the total number of running process that match the 
	##+	process_info criteria
	################################################################################
	proc CountInstances {pProcessInfo} {
		return [llength [Get $pProcessInfo]]
	}
	
	################################################################################
	##$Procedure: Background
	##$Summary: 
	##$Version: 1.0.0
	##$Author: Michael E Allen
	##$Create Date: April 23, 2001
	##$Description:
	##+	None
	##$Bugs: 
	##+	None
	##$To Do: 
	##+	Nothing
	##$Changes:
	##+	None
	##$Synopsis:
	##+	::process::Background
	##$Parameters:
	##+	None
	##$Result:
	##+	Raises a TCL error (Pandora exception) if the process could not be moved
	##+	into the background
	################################################################################
	proc BackGround {} {
		try {
			trap SIG_IGN {TTIN TSTP}
		} catch lErrorMessage {
			Error {
				throw ProcessException $lErrorMessage
			}
		}			

		try {
			set lFork [fork]
			if {($lFork == -1)} {
				throw ProcessException "failed to fork process"
			} elseif {($lFork > 0)} {
				exit
			}
		} catch lErrorMessage {
			Error {
				throw ProcessException $lErrorMessage
			}
		}

		try {
			disconnect
			trap SIG_IGN HUP
		} catch lErrorMessage {
			Error {
				throw ProcessException $lErrorMessage
			}
		}
		
		try {
			set lFork [fork]
			if {($lFork == -1)} {
				throw ProcessException "failed to fork process"
			} elseif {($lFork > 0)} {
				exit
			}
		} catch lErrorMessage {
			Error {
				throw ProcessException $lErrorMessage
			}
		}

		try {
			cd /
			trap SIG_IGN CHLD
		} catch lErrorMessage {
			Error {
				throw ProcessException $lErrorMessage
			}
		}			
	}
}