/*
	config.cpp

	The Config class implements all the functionality related to manipulating
	the pgjobs configuration settings. This file is built in both the agent
	and the configuration tool.

	Project: pgjobs
	Author: Zlatko Michailov
	Created: 19-Jul-2003
	Updated: 23-Jul-2003
	Updated: 24-Jul-2003
	Updated: 26-Jul-2003
	Updated: 27-Jul-2003
	Updated:  2-Aug-2003
	Updated:  3-Aug-2003
	Updated;  6-Oct-2003
	Updated: 15-Oct-2003
	Updated: 18-Jun-2004

	This file is provided as is, with no warranty. Use at your own risk.

	Copyright (c) 2003, Zlatko Michailov

*/



//--------------------------------------------------------------------------------

#include <stdio.h>
#include "config.h"

using namespace std;



//--------------------------------------------------------------------------------
// Static member initialization

const string	Config::__empty;
string			Config::__trash;



//--------------------------------------------------------------------------------
// Construction

Config::Config( const string& fileSpec )
{
	FileSpec	= fileSpec;
	In			= 0;
	Out			= 0;
}



Config::~Config()
{
	Close();
}



//--------------------------------------------------------------------------------
// Public API
bool Config::EnableDb( const DbRef& dbref, string& err )
{
	bool	setDb = false;
	string	dbid;

	if ( FindDbId( dbref, dbid ) )
	{
		err = "EnableDb: db already enabled";
	}
    else
    {
	    if ( setDb = NextDbId( dbid, err ) )
		{
			string row = dbid + RowMainChar +
						 dbref.Host + RowColChar +
						 dbref.Port + RowColChar +
						 dbref.Database + RowColChar +
						 "0" + RowColChar +			// Log level
						 Default + RowColChar +		// User
						 None + RowColChar +		// Notification
						 None;						// Recipient
			setDb = SetRow( dbid, row, err );
		}
	}

	return setDb;
}



bool Config::DisableDb( const DbRef& dbref, string& err )
{
	bool	delDb = false;
	string	dbid;

	if ( delDb = FindDbId( dbref, dbid, err ) )
	{
		delDb = DelRow( dbid, err );
	}

	return delDb;
}



bool Config::GetMaster( string& row, string& err )
{
	return GetRow( Master, row, err );
}



bool Config::SetMaster( const string row, string& err )
{
	return SetRow( Master, row, err );
}



bool Config::GetSystem( string& row, string& err )
{
	return GetRow( System, row, err );
}



bool Config::SetSystem( const string row, string& err )
{
	return SetRow( System, row, err );
}



bool Config::GetDb( const DbRef& dbref, string& row, string& err )
{
	bool	gotDb = false;
	string	dbid;

	if ( gotDb = FindDbId( dbref, dbid, err ) )
	{
		gotDb = GetRow( dbid, row, err );
	}

	return gotDb;
}



bool Config::SetDb( const DbRef& dbref, const string row, string& err )
{
	bool	setDb = false;
	string	dbid;

	if ( setDb = FindDbId( dbref, dbid, err ) )
	{
		setDb = SetRow( dbid, row, err );
	}

	return setDb;
}



bool Config::SeqGetDb( StreamPosition& pos, DbInfo& dbInfo, string& err )
{
	bool ok = false;
	
	err.clear();
	if ( Open( false, err ) )
	{
		string line;

		// Seek to the specified position
		Seek( pos );

		while ( ReadLine( line, err ) )
		{
			string	remain = line;
			string	extract;
			bool	isRow;

			// Look for extract=rowid
			isRow = FindChar( remain, RowMainChar, true, extract, err );
			if ( isRow && extract != Master && extract != System )
			{
				// Load DbInfo
				dbInfo.ID = extract;
				ok = GetCol( line, 1, dbInfo.Host );
				ok = ok && GetCol( line, 2, dbInfo.Port );
				ok = ok && GetCol( line, 3, dbInfo.Database );
				ok = ok && GetCol( line, 4, dbInfo.LogLevel );
				ok = ok && GetCol( line, 5, dbInfo.User );
				GetCol( line, 6, dbInfo.Notification );
				GetCol( line, 7, dbInfo.Recipient );

				if ( ok )
				{
					Tell( pos );
					break;
				}
			}
		} // while ReadLine()


		Close();
	} // if Open()
	else
	{
		// Could not Open
		ok = false;
		err = "SeqGetDb: could not Open\n" + err;
	}

	return ok;
}



//--------------------------------------------------------------------------------
// Internal support - File level

bool Config::NextDbId( string& dbid, string& err )
{
	return GetDbId( false, DbRef( __empty ), dbid, err );
}



bool Config::FindDbId( const DbRef& dbref, string& dbid, string& err )
{
	return GetDbId( true, dbref, dbid, err );
}



bool Config::GetDbId( bool exists, const DbRef& dbref, string& dbid, string& err )
{
	bool foundDbRef	= !exists;

	if ( Open( false, err ) )
	{
		unsigned int	maxDbId = 0;
		string			line;

		while ( ReadLine( line, err ) )
		{
			string	remain = line;
			string	extract;
			bool	isRow;

			isRow = FindChar( remain, RowMainChar, true, extract, err );
			if ( isRow && atoi( extract.c_str() ) )
			{
				string	host;
				string	port;
				string	database;

				isRow = isRow && GetCol( line, 1, host, err );
				isRow = isRow && GetCol( line, 2, port, err );
				isRow = isRow && GetCol( line, 3, database, err );

				// This is a row: rowid=...|...|...
				if ( isRow )
				{
					if ( exists )
					{
						// We are searching for an exact match
						if ( host	  == dbref.Host &&
							 port	  == dbref.Port &&
							 database == dbref.Database )
						{
							// We've found one match: break out of the loop
							foundDbRef = true;
							dbid = extract;
							break;
						}
					}
					else
					{
						// We have to scan ALL rowid's:
						// Get the current rowid and
						// save the current max rowid
						unsigned int currDbId = atoi( extract.c_str() );
						maxDbId = max( maxDbId, currDbId );
					}
				}

				// Clean up the err messages from this iteration
				err.clear();
			}
		} // while ReadLine()

		Close();

		// Finish work
		if ( exists )
		{
			if ( !foundDbRef )
			{
				err = "GetDbId: Not found\n" + err;
			}
		}
		else
		{
			char nextDbId[ 20 ];

			sprintf( nextDbId, "%u", ++maxDbId );
			dbid = nextDbId;
		}
		
	} // if Open()
	else
	{
		// Could not Open
		foundDbRef = false;
		err = "GetDbId: could not Open\n" + err;
	}

	return foundDbRef;
}
	


bool Config::GetRow( const string& rowid, string& row, string& err )
{
	return GetSetRow( false, rowid, __empty, row, err );
}



bool Config::SetRow( const string& rowid, const string& row, string& err )
{
	return GetSetRow( true, rowid, row, __trash, err );
}



bool Config::DelRow( const string& rowid, string& err )
{
	return GetSetRow( true, rowid, __empty, __trash, err );
}



bool Config::GetSetRow( bool mustSet, const string& rowid, const string& rowIn, string& rowOut, string& err )
{
	bool gotRow			= false;
	bool isInputGood	= true;

	// Do not allow empty rowid
	if ( rowid.empty() )
	{
		isInputGood = false;
		err = "GetSetRow: rowid is empty\n" + err;
	}

	if ( isInputGood )
	{
		if ( Open( mustSet, err ) )
		{
			bool	foundMatch	= false;
			bool	writeOK		= true;
			string	line;

			while ( ReadLine( line, err ) )
			{
				string	remain = line;
				string	extract;
				bool	isRow;

				// Look for extract=rowid
				isRow = FindChar( remain, RowMainChar, true, extract, err );
				if ( isRow && extract == rowid )
				{
					// We have a match!
					foundMatch = true;

					if ( mustSet )
					{
						// Copy the new line or do nothing to delete
						if ( !rowIn.empty() )
						{
							gotRow = writeOK = WriteLine( rowIn, err );
						}
						else
						{
							gotRow = true;
						}
					}
					else
					{
						// We've found one match: break out of the loop
						gotRow = true;
						rowOut = line;
						break;
					}
				}
    	        else if ( mustSet )
				{
					// When SET: we have to copy over every line that doesn't match
					writeOK = WriteLine( line, err );
				}

   				if ( !writeOK )
   				{
					break;
   				}
			} // while ReadLine()

			// We scanned through the whole file and did not find a match
			if ( !foundMatch )
			{
				// When SET: append the line
				if ( mustSet )
				{
					if ( !rowIn.empty() )
					{
						gotRow = writeOK = WriteLine( rowIn, err );
					}
				}
				else
				{
					// When GET: raise error
					gotRow = false;
					err = "GetSetRow: rowid not found\n" + err;
				}
			}

			if ( !writeOK )
			{
				// Write has failed
				gotRow = false;
				err = "GetSetRow: Could not WriteLine\n" + err;
			}

			Close();
		} // if Open()
		else
		{
			// Could not Open
			gotRow = false;
			err = "GetSetRow: could not Open\n" + err;
		}
	} // Input is good

	return gotRow;
}



//--------------------------------------------------------------------------------
// Internal support - Row level

bool Config::GetCol( const string& row, int i, string& col, string& err )
{
	return GetSetCol( false, row, i, __empty, col, err );
}



bool Config::SetCol( string& row, int i, const string& col, string& err )
{
	return GetSetCol( true, row, i, col, row, err );
}



bool Config::GetSetCol( bool mustSet, const string& row, int i, const string& colIn, string& out, string& err )
{
	bool	gotCol	= false;
	string	remain	= row;
	string	lead	= "";
	string	extract;

	//          1       2     3
	// agent=setting|setting|...
	gotCol = FindChar( remain, RowMainChar, true, extract, err );
	for ( ; gotCol && i; i-- )
	{
		lead  += extract + ( lead.empty() ? RowMainChar : RowColChar );
		gotCol = FindChar( remain, RowColChar, false, extract, err );
	}

	// Support appending a column
	if ( !gotCol && !i )
	{
		char last = lead[ lead.length() - 1 ];
		
		gotCol = true;
		if ( last != RowMainChar && last != RowColChar )
		{
			last += RowColChar;
		}
	}

	if ( gotCol )
	{
		if ( mustSet )
		{
			// When SET: out is the modified ROW
			out = lead + colIn + RowColChar + remain;
		}
		else
		{
			// When GET: out is the COL
			out = extract;
		}
	}

	return gotCol;
}



bool Config::FindChar( string& remain, char ch, bool mustFind, string& extract, string& err )
{
	bool 	foundChar	= false;
	int		len			= remain.length();
	int		i;

	for ( i = 0; i < len; i++ )
	{
		// The char is indeed found.
		// Break at this point.
		if ( remain[ i ] == ch )
		{
			foundChar = true;
			break;
		}

		// Break at this point.
		if ( remain[ i ] == RowCommentChar )
		{
			break;
		}

		// If there is at least one char in the string,
		// and the char is optional and we reach EOL or #, 
		// we assume the target char is found.
		foundChar = !mustFind;
	}

	if ( foundChar )
	{
		extract.clear();
		extract.append( remain, 0, i );	// excluding ch

		remain.erase( 0, i + 1 );		// including ch
	}
	else
	{
		err = "' ' not found\n" + err;
		err[ 1 ] = ch;
	}

	return foundChar;
}



//--------------------------------------------------------------------------------
// Internal support - I/O

bool Config::Open( bool withWrite, string& err )
{
	bool isOpen = false;

	err.clear();
	Close();

	if ( withWrite )
	{
		string BackupFileSpec = FileSpec + BackupExt;
		Backup( err );
		Close();

		// In: .bak
		// Out: conf
		In  = new ifstream( BackupFileSpec.c_str() );
		Out = new ofstream( FileSpec.c_str() );
		isOpen = ( In && Out );
	}
	else
	{
		// In: conf
		In = new ifstream( FileSpec.c_str() );
		isOpen = !!In;
	}

	if ( !isOpen )
	{
		err = "Open: could not open file\n" + err;
		Close();
	}

	return isOpen;
}



bool Config::Backup( string& err )
{
	bool	isBackedUp		= false;
	string	BackupFileSpec	= FileSpec + BackupExt;

	err.clear();
	Close();

	// In: conf
	// Out: .bak
	In  = new ifstream( FileSpec.c_str() );
	Out = new ofstream( BackupFileSpec.c_str() );

	if ( In && Out )
	{
		string line;
		while ( ReadLine( line, err ) )
		{
			WriteLine( line, err );
		}
	}
	else
	{
		err = "Backup: Could not open or create file\n" + err;
	}

	Close();

	return isBackedUp;
}



bool Config::ReadLine( string& line, string& err )
{
	bool isRead = false;

	err.clear();

	if ( In )
	{
		char buff[ MaxLineLen + 1 ];

		// getline() returns false at EOF
		if ( In->getline( buff, MaxLineLen ) )
		{
			line   = buff;
			isRead = true;
		}
	}
	else
	{
		err = "ReadLine: file not open\n" + err;
	}

	return isRead;
}



bool Config::WriteLine( const string& line, string& err )
{
	bool isWritten = false;

	err.clear();

	if ( Out )
	{
		*Out << line << endl;
		isWritten = true;
	}
	else
	{
		err = "WriteLine: file not open\n" + err;
	}

	return isWritten;
}



void Config::Seek( StreamPosition pos )
{
	if ( In )
	{
		In->seekg( pos );
	}
}



void Config::Tell( StreamPosition& pos )
{
	if ( In )
	{
		pos = In->tellg();
	}
}



void Config::Close()
{
	if ( In )
	{
		delete In;
		In = 0;
	}

	if ( Out )
	{
		delete Out;
		Out = 0;
	}
}



