/*
 * plpq.c
 *
 */
#include "plpq.h"

/*
 * Internal declarations
 */
static public_pq_connections *init_public_connections();
static public_pq_results *init_public_results();

static public_pq_connections *get_public_conn(int32 public_conn_id_index);
static void append_public_conn(public_pq_connections * connections);
static void remove_public_conn(public_pq_connections * connections);

static public_pq_results *get_public_res(int32 public_res_id_index);
static void append_public_res(public_pq_results * results);
static void remove_public_res(public_pq_results * results);

static PGresult* getResultPointer(int32 nResId, bool removePointer);
static PGconn* getConnPointer(int32 nConnId,bool removePointer);


/* Global */

List	   *public_conn_id = NIL;
int			public_conn_id_index = 0;
List	   *public_res_id = NIL;
int			public_res_id_index = 0;

#define GET_TEXT(cstrp) DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum(cstrp)))
#define GET_STR(textp) DatumGetCString(DirectFunctionCall1(textout, PointerGetDatum(textp)))

/*
 * init_public_connections
 */
static public_pq_connections *
init_public_connections()
{
	MemoryContext oldcontext;
	public_pq_connections *retval;
	oldcontext = MemoryContextSwitchTo(TopMemoryContext);

	retval = (public_pq_connections *) palloc(sizeof(public_pq_connections));
	MemSet(retval, 0, sizeof(public_pq_connections));

	retval->conn_id_index = -1;
	retval->conn = NULL;

	MemoryContextSwitchTo(oldcontext);

	return retval;
}

/*
 * init_public_results
 */
static public_pq_results *
init_public_results()
{
	MemoryContext oldcontext;
	public_pq_results *retval;
	oldcontext = MemoryContextSwitchTo(TopMemoryContext);

	retval = (public_pq_results *) palloc(sizeof(public_pq_results));
	MemSet(retval, 0, sizeof(public_pq_results));

	retval->res_id_index = -1;
	retval->res = NULL;

	MemoryContextSwitchTo(oldcontext);

	return retval;
}

/*
 * get_public_conn
 */
static public_pq_connections *
get_public_conn(int32 public_conn_id_index)
{
	List	   *ptr;
	MemoryContext oldcontext;
	public_pq_connections *this_public_conn_id;
	oldcontext = MemoryContextSwitchTo(TopMemoryContext);

	/*
	 * short circuit empty list
	 */
	if (public_conn_id== NIL){
		MemoryContextSwitchTo(oldcontext);
		return NULL;
	}

	/*
	 * OK, should be good to go
	 */
	foreach(ptr, public_conn_id)
	{
		this_public_conn_id = (public_pq_connections *) lfirst(ptr);
		if (this_public_conn_id->conn_id_index == public_conn_id_index){
			MemoryContextSwitchTo(oldcontext);
			return this_public_conn_id;
		}
	}
	MemoryContextSwitchTo(oldcontext);
	return NULL;
}

/*
 * get_public_conn
 */
static public_pq_results *
get_public_res(int32 public_res_id_index)
{
	List	   *ptr;
	public_pq_results *this_public_res_id;
	MemoryContext oldcontext;
	oldcontext = MemoryContextSwitchTo(TopMemoryContext);

	/*
	 * short circuit empty list
	 */
	if (public_res_id== NIL){
		MemoryContextSwitchTo(oldcontext);
		return NULL;
	}

	/*
	 * OK, should be good to go
	 */
	foreach(ptr, public_res_id)
	{
		this_public_res_id = (public_pq_results *) lfirst(ptr);
		if (this_public_res_id->res_id_index == public_res_id_index){
			MemoryContextSwitchTo(oldcontext);
			return this_public_res_id;
		}
	}
	MemoryContextSwitchTo(oldcontext);
	return NULL;
}

/*
 * Add node to global List public_conn_id
 */
static void
append_public_conn(public_pq_connections * connections)
{
	MemoryContext oldcontext;
	oldcontext = MemoryContextSwitchTo(TopMemoryContext);
	public_conn_id = lappend(public_conn_id, connections);
	MemoryContextSwitchTo(oldcontext);
}

/*
 * Add node to global List public_res_id
 */
static void
append_public_res(public_pq_results* results)
{
	MemoryContext oldcontext;
	oldcontext = MemoryContextSwitchTo(TopMemoryContext);
	public_res_id = lappend(public_res_id, results);
	MemoryContextSwitchTo(oldcontext);
}

/*
 * Remove node from global List
 * using public_conn_id_index
 */
static void
remove_public_conn(public_pq_connections * connections)
{
	MemoryContext oldcontext;
	oldcontext = MemoryContextSwitchTo(TopMemoryContext);
	public_conn_id= lremove(connections, public_conn_id);

	if (public_conn_id== NIL)
		public_conn_id_index = 0;
	MemoryContextSwitchTo(oldcontext);
}

/*
 * Remove node from global List
 * using public_conn_id_index
 */
static void
remove_public_res(public_pq_results * results)
{
	MemoryContext oldcontext;
	oldcontext = MemoryContextSwitchTo(TopMemoryContext);
	public_res_id= lremove(results, public_res_id);

	if (public_res_id== NIL)
		public_res_id_index = 0;
	MemoryContextSwitchTo(oldcontext);
}

/*
	Check connection id and return PGconn pointer, delete connection from list if removePointer = TRUE
*/
PGconn* getConnPointer(int32 nConnId,bool removePointer){
    PGconn* conn;
    public_pq_connections* connections;
	MemoryContext oldcontext;
	oldcontext = MemoryContextSwitchTo(TopMemoryContext);

	connections = get_public_conn(nConnId);
	if (connections == NULL){
		MemoryContextSwitchTo(oldcontext);
	   	return NULL;
	}
	conn = connections->conn;
    if (removePointer){
	    remove_public_conn(connections);
		pfree(connections);
	}
	MemoryContextSwitchTo(oldcontext);
	return conn;
}

/*
	Check result id and return PGresult pointer, delete result from list if removePointer = TRUE
*/
PGresult* getResultPointer(int32 nResId,bool removePointer){
    PGresult* res;
	public_pq_results *results;
	MemoryContext oldcontext;
	oldcontext = MemoryContextSwitchTo(TopMemoryContext);

	results = get_public_res(nResId);
	if (results == NULL){
		MemoryContextSwitchTo(oldcontext);
		return NULL;
	}
	res = results->res;
    if (removePointer){
	    remove_public_res(results);
		pfree(results);
	}
	MemoryContextSwitchTo(oldcontext);
	return res;
}


/*
	PGconn Connectdb(char *ConnectionString);
	--Connect to database
*/
PG_FUNCTION_INFO_V1(plpq_Connectdb);
Datum
plpq_Connectdb(PG_FUNCTION_ARGS)
{
	PGconn	*conn = NULL;
   	char *connString = GET_STR(PG_GETARG_TEXT_P(0));
	public_pq_connections* connections;

	conn = PQconnectdb(connString);
	if (PQstatus(conn) != CONNECTION_BAD)
	{
		public_conn_id_index++;
		connections = init_public_connections();
		connections->conn_id_index = public_conn_id_index;
		connections->conn=conn;
		append_public_conn(connections);
		PG_RETURN_INT32(public_conn_id_index);
	}else{
		PG_RETURN_INT32(0);
	}
}

/*
	PGconn SetdbLogin(char *host,char *port,char *options,char *tty,char *dbName,char *Login,char *Password);
	--Connect to database
*/
PG_FUNCTION_INFO_V1(plpq_SetdbLogin);
Datum
plpq_SetdbLogin(PG_FUNCTION_ARGS)
{
	PGconn	*conn = NULL;
	char		*pghost;
	char 		*pgport;
	char 		*pgoptions;
	char 		*pgtty;
	char 		*dbName;
	char 		*login;
	char 		*pwd;
	public_pq_connections* connections;

	pghost = GET_STR(PG_GETARG_TEXT_P(0));
	pgport = GET_STR(PG_GETARG_TEXT_P(1));
	pgoptions = GET_STR(PG_GETARG_TEXT_P(2));
	pgtty = GET_STR(PG_GETARG_TEXT_P(3));
	dbName = GET_STR(PG_GETARG_TEXT_P(4));
	login = GET_STR(PG_GETARG_TEXT_P(5));
	pwd	= GET_STR(PG_GETARG_TEXT_P(6));

	if (pgport == NULL)
		pgport="5432";

	conn = PQsetdbLogin(pghost,pgport,pgoptions,pgtty,dbName,login,pwd);
	if (PQstatus(conn) != CONNECTION_BAD)
	{
		public_conn_id_index++;
		connections = init_public_connections();
		connections->conn_id_index = public_conn_id_index;
		connections->conn=conn;
		append_public_conn(connections);
		PG_RETURN_INT32(public_conn_id_index);
	}else{
		PG_RETURN_INT32(0);
	}
}

/*
	int Status(PGconn *Connection);
	--Connection status
*/
PG_FUNCTION_INFO_V1(plpq_Status);
Datum
plpq_Status(PG_FUNCTION_ARGS)
{
    PGconn* conn;
    conn = getConnPointer(PG_GETARG_INT32(0),FALSE);
    if (conn==NULL)
		elog(ERROR, "plpq_Status: function called with invalid connection id");
	PG_RETURN_INT32(PQstatus(conn));
}

/*
	text StatusStr(PGconn *Connection);
	--Connection status
*/
PG_FUNCTION_INFO_V1(plpq_StatusStr);
Datum
plpq_StatusStr(PG_FUNCTION_ARGS)
{
	text*   ConStat = NULL;
    PGconn* conn;
    conn = getConnPointer(PG_GETARG_INT32(0),FALSE);

    if (conn==NULL)
    	elog(ERROR, "plpq_StatusStr: function called with invalid connection id");

	switch( PQstatus(conn) ){
		case 0 : {
			ConStat = DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum("CONNECTION_OK")));
			break;
		} case 1 : {
			ConStat = DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum("CONNECTION_BAD")));
			break;
		} case 2 : {
			ConStat = DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum("CONNECTION_STARTED")));
			break;
		} case 3 : {
			ConStat = DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum("CONNECTION_MADE")));
			break;
		} case 4 : {
			ConStat = DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum("CONNECTION_AWAITING_RESPONSE")));
			break;
		}case 5 : {
			ConStat = DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum("CONNECTION_AUTH_OK")));
			break;
		}case 6 : {
			ConStat = DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum("CONNECTION_SETENV")));
			break;
		}
	}
	PG_RETURN_TEXT_P(ConStat);
}

/*
	int Finish(PGconn *Connection);
	--Finish the connection
*/
PG_FUNCTION_INFO_V1(plpq_Finish);
Datum
plpq_Finish(PG_FUNCTION_ARGS)
{
    PGconn* conn;
    conn = getConnPointer(PG_GETARG_INT32(0),TRUE);
    if (conn==NULL)
    	elog(ERROR, "plpq_Finish: function called with invalid connection id");

	PQfinish(conn);
	PG_RETURN_INT32(0);
}

/*
	int Reset(PGconn *Connection);
	--Reset the connection
*/
PG_FUNCTION_INFO_V1(plpq_Reset);
Datum
plpq_Reset(PG_FUNCTION_ARGS)
{
    PGconn* conn;
    conn = getConnPointer(PG_GETARG_INT32(0),FALSE);
    if (conn==NULL)
    	elog(ERROR, "plpq_Reset: function called with invalid connection id");

	PQreset(conn);
	PG_RETURN_INT32(0);
}

/*
	text Db(PGconn *Connection);
	--Returns database name
*/
PG_FUNCTION_INFO_V1(plpq_Db);
Datum
plpq_Db(PG_FUNCTION_ARGS)
{
	text* DbName = NULL;
    PGconn* conn;
    conn = getConnPointer(PG_GETARG_INT32(0),FALSE);
    if (conn==NULL)
    	elog(ERROR, "plpq_Db: function called with invalid connection id");

	DbName = DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum(PQdb(conn))));
	PG_RETURN_TEXT_P(DbName);
}

/*
	text User(PGconn *Connection);
	--Returns user name
*/
PG_FUNCTION_INFO_V1(plpq_User);
Datum
plpq_User(PG_FUNCTION_ARGS)
{
	text* User = NULL;
    PGconn* conn;
    conn = getConnPointer(PG_GETARG_INT32(0),FALSE);
    if (conn==NULL)
    	elog(ERROR, "plpq_User: function called with invalid connection id");

	User = DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum(PQuser(conn))));
	PG_RETURN_TEXT_P(User);
}

/*
	text Password(PGconn *Connection);
	--Returns password
*/
PG_FUNCTION_INFO_V1(plpq_Password);
Datum
plpq_Password(PG_FUNCTION_ARGS)
{
	text* Password = NULL;
    PGconn* conn;
    conn = getConnPointer(PG_GETARG_INT32(0),FALSE);
    if (conn==NULL)
    	elog(ERROR, "plpq_Password: function called with invalid connection id");

	Password = DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum(PQpass(conn))));
	PG_RETURN_TEXT_P(Password);
}

/*
	text Host(PGconn *Connection);
	--Returns host name
*/
PG_FUNCTION_INFO_V1(plpq_Host);
Datum
plpq_Host(PG_FUNCTION_ARGS)
{
	text			*Host = NULL;
    PGconn* conn;
    conn = getConnPointer(PG_GETARG_INT32(0),FALSE);
    if (conn==NULL)
    	elog(ERROR, "plpq_Host: function called with invalid connection id");

	Host = DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum(PQhost(conn))));
	PG_RETURN_TEXT_P(Host);
}

/*
	text Port(PGconn *Connection);
	--Returns port
*/
PG_FUNCTION_INFO_V1(plpq_Port);
Datum
plpq_Port(PG_FUNCTION_ARGS)
{
	text* Port = NULL;
	PGconn* conn;
	conn = getConnPointer(PG_GETARG_INT32(0),FALSE);
	if (conn==NULL)
		elog(ERROR, "plpq_Port: function called with invalid connection id");

	Port = DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum(PQport(conn))));
	PG_RETURN_TEXT_P(Port);
}

/*
	text Tty(PGconn *Connection);
	--Returns tty
*/
PG_FUNCTION_INFO_V1(plpq_Tty);
Datum
plpq_Tty(PG_FUNCTION_ARGS)
{
	text* Tty = NULL;
    PGconn* conn;
    conn = getConnPointer(PG_GETARG_INT32(0),FALSE);
    if (conn==NULL)
    	elog(ERROR, "plpq_Tty: function called with invalid connection id");

	Tty = DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum(PQtty(conn))));
	PG_RETURN_TEXT_P(Tty);
}

/*
	text ErrorMessage(PGconn *Connection);
	--Returns connection error message
*/
PG_FUNCTION_INFO_V1(plpq_ErrorMessage);
Datum
plpq_ErrorMessage(PG_FUNCTION_ARGS)
{
	text* ErrorMsg = NULL;
    PGconn* conn;
    conn = getConnPointer(PG_GETARG_INT32(0),FALSE);
    if (conn==NULL)
    	elog(ERROR, "plpq_ErrorMessage: function called with invalid connection id");

	ErrorMsg = DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum(PQerrorMessage(conn))));
	PG_RETURN_TEXT_P(ErrorMsg);
}

/*
	int BackendPID(PGconn *Connection);
	--Returns backend PID of host server process
*/
PG_FUNCTION_INFO_V1(plpq_BackendPID);
Datum
plpq_BackendPID(PG_FUNCTION_ARGS)
{
    PGconn* conn;
    conn = getConnPointer(PG_GETARG_INT32(0),FALSE);
    if (conn==NULL)
    	elog(ERROR, "plpq_BackendPID: function called with invalid connection id");

	PG_RETURN_INT32(PQbackendPID(conn));
}

/*
	PGresult Exec(PGconn *Connection, char *SqlExecString);
	--Execute query and return pointer to PGresult
*/
PG_FUNCTION_INFO_V1(plpq_Exec);
Datum
plpq_Exec(PG_FUNCTION_ARGS)
{
    public_pq_results* results;
	PGresult* res = NULL;
	char* SqlExecStr;
    PGconn* conn;
    conn = getConnPointer(PG_GETARG_INT32(0),FALSE);
    if (conn==NULL)
    	elog(ERROR, "plpq_Exec: function called with invalid connection id");

	SqlExecStr = GET_STR(PG_GETARG_TEXT_P(1));
	res= PQexec(conn,SqlExecStr);

	public_res_id_index++;
	results = init_public_results();
	results->res_id_index = public_res_id_index;
	results->res=res;
	append_public_res(results);
	PG_RETURN_INT32(public_res_id_index);
}

/*
	int ResultStatus(PGresult *Res);
	--Status of PGresult
*/
PG_FUNCTION_INFO_V1(plpq_ResultStatus);
Datum
plpq_ResultStatus(PG_FUNCTION_ARGS)
{
    PGresult* res;
    res = getResultPointer(PG_GETARG_INT32(0),FALSE);
    if (res==NULL)
		elog(ERROR, "plpq_ResultStatus: function called with invalid resource id");

	PG_RETURN_INT32(PQresultStatus(res));
}

/*
	text ResStatus(PGresult *Res);
	--Status of PGresult as string (PGRES_COMMAND_OK for example)
*/
PG_FUNCTION_INFO_V1(plpq_ResStatus);
Datum
plpq_ResStatus(PG_FUNCTION_ARGS)
{
	int		statusFlag;
	text	*StatMsg;

	statusFlag = PG_GETARG_INT32(0);
	if (statusFlag < 0)
		elog(ERROR, "plpq_ResStatus: invalid status value");

	StatMsg = DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum(PQresStatus(statusFlag))));
	PG_RETURN_TEXT_P(StatMsg);
}

/*
	text ResultErrorMessage(PGresult *Res);
	--Return PGresult error message
*/
PG_FUNCTION_INFO_V1(plpq_ResultErrorMessage);
Datum
plpq_ResultErrorMessage(PG_FUNCTION_ARGS)
{
	text *ErrMsg;
    PGresult* res;
    res = getResultPointer(PG_GETARG_INT32(0),FALSE);
    if (res==NULL)
		elog(ERROR, "plpq_ResultErrorMessage: function called with invalid resource id");

	ErrMsg=DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum(PQresultErrorMessage(res))));
	PG_RETURN_TEXT_P(ErrMsg);
}

/*
	int Clear(PGresult *Res);
	--Clear the PGresult
*/
PG_FUNCTION_INFO_V1(plpq_Clear);
Datum
plpq_Clear(PG_FUNCTION_ARGS)
{
    PGresult* res;
    res = getResultPointer(PG_GETARG_INT32(0),TRUE);
    if (res==NULL)
		elog(ERROR, "plpq_ResultErrorMessage: function called with invalid resource id");

	PQclear(res);
	PG_RETURN_INT32(0);
}

/*
	text EscapeString(char *EscapeStr);
	--Escape the query string
*/
PG_FUNCTION_INFO_V1(plpq_EscapeString);
Datum
plpq_EscapeString(PG_FUNCTION_ARGS)
{
	char* InputStr;
	char* OutputStr;
	size_t InputLen;
	size_t OutputLen;

	InputStr = GET_STR(PG_GETARG_TEXT_P(0));
	InputLen=strlen(InputStr);

	if(InputStr==NULL){
		PG_RETURN_NULL();
	}
	OutputStr=(char*)palloc((2*InputLen)+1);
	OutputLen=PQescapeString (OutputStr, InputStr, InputLen);
	PG_RETURN_TEXT_P( DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum(OutputStr))) );
}

/*
	int Ntuples(PGresult *Res);
	--Return number of tuples in PGresult
*/
PG_FUNCTION_INFO_V1(plpq_Ntuples);
Datum
plpq_Ntuples(PG_FUNCTION_ARGS)
{
    PGresult* res;
    res = getResultPointer(PG_GETARG_INT32(0),FALSE);
    if (res==NULL)
		elog(ERROR, "plpq_Ntuples: function called with invalid resource id");

	PG_RETURN_INT32(PQntuples(res));
}

/*
	int Nfields(PGresult *Res);
	--Return number of fields(columns) in PGresult
*/
PG_FUNCTION_INFO_V1(plpq_Nfields);
Datum
plpq_Nfields(PG_FUNCTION_ARGS)
{
    PGresult* res;
    res = getResultPointer(PG_GETARG_INT32(0),FALSE);
    if (res==NULL)
		elog(ERROR, "plpq_Nfields: function called with invalid resource id");

	PG_RETURN_INT32(PQnfields(res));
}

/*
	text Fname(PGresult *Res, int FieldNumber);
	--Return field(column) name for given column index
*/
PG_FUNCTION_INFO_V1(plpq_Fname);
Datum
plpq_Fname(PG_FUNCTION_ARGS)
{
	int				FieldNumber;
	char			*FieldName;
    PGresult* res;
    res = getResultPointer(PG_GETARG_INT32(0),FALSE);
    if (res==NULL)
		elog(ERROR, "plpq_Fname: function called with invalid resource id");

	FieldNumber = PG_GETARG_INT32(1);
	if (FieldNumber > (PQnfields(res) - 1))
		PG_RETURN_NULL();

	FieldName=PQfname(res,FieldNumber);
	PG_RETURN_TEXT_P( DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum(FieldName))) );
}

/*
	text Fnumber(PGresult *Res, char *FieldName);
	--Return field(column) number for given column name
*/
PG_FUNCTION_INFO_V1(plpq_Fnumber);
Datum
plpq_Fnumber(PG_FUNCTION_ARGS)
{
	char			*FieldName;
	PGresult	*res = NULL;
    res = getResultPointer(PG_GETARG_INT32(0),FALSE);
    if (res==NULL)
		elog(ERROR, "plpq_Fnumber: function called with invalid resource id");

	FieldName = DatumGetCString(DirectFunctionCall1(textout, PointerGetDatum(PG_GETARG_TEXT_P(1))));
	PG_RETURN_INT32(PQfnumber(res,FieldName));
}

/*
	oid Ftype(PGresult *Res, int FieldNumber);
	--Return oid of data type for given field(column) number
*/
PG_FUNCTION_INFO_V1(plpq_Ftype);
Datum
plpq_Ftype(PG_FUNCTION_ARGS)
{
	int				FieldNumber;
	PGresult	*res = NULL;
    res = getResultPointer(PG_GETARG_INT32(0),FALSE);
    if (res==NULL)
		elog(ERROR, "plpq_Ftype: function called with invalid resource id");

	FieldNumber = PG_GETARG_INT32(1);
	if (FieldNumber > (PQnfields(res) - 1))
		PG_RETURN_NULL();
	PG_RETURN_OID(PQftype(res,FieldNumber));
}

/*
	int Fmod(PGresult *Res, int FieldNumber);
	--Return type-specific modificaton data for given field(column) number
*/
PG_FUNCTION_INFO_V1(plpq_Fmod);
Datum
plpq_Fmod(PG_FUNCTION_ARGS)
{
	int				FieldNumber;
	PGresult	*res = NULL;
    res = getResultPointer(PG_GETARG_INT32(0),FALSE);
    if (res==NULL)
		elog(ERROR, "plpq_Fmod: function called with invalid resource id");

	FieldNumber = PG_GETARG_INT32(1);
	if (FieldNumber > (PQnfields(res) - 1))
		PG_RETURN_NULL();
	PG_RETURN_INT32(PQfmod(res,FieldNumber));
}

/*
	int Fsize(PGresult *Res, int FieldNumber);
	--Return the size in bytes of the field
*/
PG_FUNCTION_INFO_V1(plpq_Fsize);
Datum
plpq_Fsize(PG_FUNCTION_ARGS)
{
	int				FieldNumber;
	PGresult	*res = NULL;
    res = getResultPointer(PG_GETARG_INT32(0),FALSE);
    if (res==NULL)
		elog(ERROR, "plpq_Fsize: function called with invalid resource id");

	FieldNumber = PG_GETARG_INT32(1);
	if (FieldNumber > (PQnfields(res) - 1))
		PG_RETURN_NULL();
	PG_RETURN_INT32(PQfsize(res,FieldNumber));
}

/*
	int BinaryTuples(PGresult *Res);
	--Return 1 if PGresult have binary tuples, 0 if not
*/
PG_FUNCTION_INFO_V1(plpq_BinaryTuples);
Datum
plpq_BinaryTuples(PG_FUNCTION_ARGS)
{
	PGresult	*res = NULL;
    res = getResultPointer(PG_GETARG_INT32(0),FALSE);
    if (res==NULL)
		elog(ERROR, "plpq_BinaryTuples: function called with invalid resource id");

	PG_RETURN_INT32(PQbinaryTuples(res));

}

/*
	text GetValue(PGresult *Res, int TupleNumber, int FieldNumber);
	--Return value of field
*/
PG_FUNCTION_INFO_V1(plpq_GetValue);
Datum
plpq_GetValue(PG_FUNCTION_ARGS)
{
	int				TupleNumber=0;
	int				FieldNumber=0;
	PGresult	*res = NULL;
    res = getResultPointer(PG_GETARG_INT32(0),FALSE);
    if (res==NULL)
		elog(ERROR, "plpq_GetValue: function called with invalid resource id");

	TupleNumber = PG_GETARG_INT32(1);
	if ( (TupleNumber<0) || (FieldNumber > (PQntuples(res) - 1)) )
		elog(ERROR, "PlPq: invalid tuple number: %i",TupleNumber);

	FieldNumber = PG_GETARG_INT32(2);
	if (FieldNumber > (PQnfields(res) - 1))
		elog(ERROR, "PlPq: invalid field number: %i",FieldNumber);

	if (PQgetisnull(res,TupleNumber,FieldNumber)==1){
		/*Return NULL*/
		PG_RETURN_NULL();
	}else{
		/*Return text*/
		PG_RETURN_TEXT_P( DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum( PQgetvalue(res,TupleNumber,FieldNumber) ))) );
	}
}

/*
	int Getisnull(PGresult *Res, int TupleNumber, int FieldNumber);
	--Return 1 if field has no value(NULL), 0 if not
*/
PG_FUNCTION_INFO_V1(plpq_Getisnull);
Datum
plpq_Getisnull(PG_FUNCTION_ARGS)
{
	int				TupleNumber=0;
	int				FieldNumber=0;
	PGresult	*res = NULL;
    res = getResultPointer(PG_GETARG_INT32(0),FALSE);
    if (res==NULL)
		elog(ERROR, "plpq_Getisnull: function called with invalid resource id");

	TupleNumber = PG_GETARG_INT32(1);
	if ( (TupleNumber<0) || (FieldNumber > (PQntuples(res) - 1)) )
		elog(ERROR, "PlPq: invalid tuple number: %i",TupleNumber);

	FieldNumber = PG_GETARG_INT32(2);
	if (FieldNumber > (PQnfields(res) - 1))
		elog(ERROR, "PlPq: invalid field number: %i",FieldNumber);

	PG_RETURN_INT32(PQgetisnull(res,TupleNumber,FieldNumber));
}

/*
	int Getlength(PGresult *Res, int TupleNumber, int FieldNumber);
	--Return actual size of field
*/
PG_FUNCTION_INFO_V1(plpq_Getlength);
Datum
plpq_Getlength(PG_FUNCTION_ARGS)
{
	int				TupleNumber=0;
	int				FieldNumber=0;
	PGresult	*res = NULL;
    res = getResultPointer(PG_GETARG_INT32(0),FALSE);
    if (res==NULL)
		elog(ERROR, "plpq_Getlength: function called with invalid resource id");

	TupleNumber = PG_GETARG_INT32(1);
	if ( (TupleNumber<0) || (FieldNumber > (PQntuples(res) - 1)) )
		elog(ERROR, "PlPq: invalid tuple number: %i",TupleNumber);

	FieldNumber = PG_GETARG_INT32(2);
	if (FieldNumber > (PQnfields(res) - 1))
		elog(ERROR, "PlPq: invalid field number: %i",FieldNumber);

	PG_RETURN_INT32(PQgetlength(res,TupleNumber,FieldNumber));
}

/*
	text CmdStatus(PGresult *Res);
	--Return command status string from PGresult
*/
PG_FUNCTION_INFO_V1(plpq_CmdStatus);
Datum
plpq_CmdStatus(PG_FUNCTION_ARGS)
{
	PGresult	*res = NULL;
    res = getResultPointer(PG_GETARG_INT32(0),FALSE);
    if (res==NULL)
		elog(ERROR, "plpq_CmdStatus: function called with invalid resource id");

	PG_RETURN_TEXT_P( DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum( PQcmdStatus(res) ))) );
}

/*
	int CmdTuples(PGresult *Res);
	--Return number of rows affected by the SQL command
*/
PG_FUNCTION_INFO_V1(plpq_CmdTuples);
Datum
plpq_CmdTuples(PG_FUNCTION_ARGS)
{
	PGresult	*res = NULL;
    res = getResultPointer(PG_GETARG_INT32(0),FALSE);
    if (res==NULL)
		elog(ERROR, "plpq_CmdTuples: function called with invalid resource id");

	PG_RETURN_INT32(atoi(PQcmdTuples(res)));
}

/*
	oid OidValue(PGresult *Res);
	--Return last inserted OID
*/
PG_FUNCTION_INFO_V1(plpq_OidValue);
Datum
plpq_OidValue(PG_FUNCTION_ARGS)
{
	PGresult	*res = NULL;
    res = getResultPointer(PG_GETARG_INT32(0),FALSE);
    if (res==NULL)
		elog(ERROR, "plpq_OidValue: function called with invalid resource id");

	PG_RETURN_OID(PQoidValue(res));
}

/*
	text OidStatus(PGresult *Res);
	--Return OID status as string
*/
PG_FUNCTION_INFO_V1(plpq_OidStatus);
Datum
plpq_OidStatus(PG_FUNCTION_ARGS)
{
	PGresult	*res = NULL;
    res = getResultPointer(PG_GETARG_INT32(0),FALSE);
    if (res==NULL)
		elog(ERROR, "plpq_OidStatus: function called with invalid resource id");

	PG_RETURN_TEXT_P( DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum( PQoidStatus(res) ))) );
}
