/****************************************************************************
**
** Copyright (C) 2018 The Qt Company Ltd.
** Contact: https://www.qt.io/licensing/
**
** This file is part of the QtBluetooth module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company. For licensing terms
** and conditions see https://www.qt.io/terms-conditions. For further
** information use the contact form at https://www.qt.io/contact-us.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 3 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL3 included in the
** packaging of this file. Please review the following information to
** ensure the GNU Lesser General Public License version 3 requirements
** will be met: https://www.gnu.org/licenses/lgpl-3.0.html.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 2.0 or (at your option) the GNU General
** Public license version 3 or any later version approved by the KDE Free
** Qt Foundation. The licenses are as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL2 and LICENSE.GPL3
** included in the packaging of this file. Please review the following
** information to ensure the GNU General Public License requirements will
** be met: https://www.gnu.org/licenses/gpl-2.0.html and
** https://www.gnu.org/licenses/gpl-3.0.html.
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include "qbluetoothutils_winrt_p.h"
#include <QtBluetooth/private/qtbluetoothglobal_p.h>
#include <QtCore/private/qfunctions_winrt_p.h>
#include <QtCore/QLoggingCategory>

#include <robuffer.h>
#include <wrl.h>
#include <winrt/windows.foundation.metadata.h>
#include <windows.storage.streams.h>

using namespace Microsoft::WRL;
using namespace Microsoft::WRL::Wrappers;
using namespace winrt::Windows::Foundation::Metadata;
using namespace ABI::Windows::Storage::Streams;

QT_BEGIN_NAMESPACE

Q_DECLARE_LOGGING_CATEGORY(QT_BT_WINDOWS)

QByteArray byteArrayFromBuffer(const ComPtr<NativeBuffer> &buffer, bool isWCharString)
{
    if (!buffer) {
        qErrnoWarning("nullptr passed to byteArrayFromBuffer");
        return QByteArray();
    }
    ComPtr<Windows::Storage::Streams::IBufferByteAccess> byteAccess;
    HRESULT hr = buffer.As(&byteAccess);
    RETURN_IF_FAILED("Could not cast buffer", return QByteArray())
    char *data;
    hr = byteAccess->Buffer(reinterpret_cast<byte **>(&data));
    RETURN_IF_FAILED("Could not obtain buffer data", return QByteArray())
    UINT32 size;
    hr = buffer->get_Length(&size);
    RETURN_IF_FAILED("Could not obtain buffer size", return QByteArray())
    if (isWCharString) {
        QString valueString = QString::fromUtf16(reinterpret_cast<char16_t *>(data)).left(size / 2);
        return valueString.toUtf8();
    }
    return QByteArray(data, qint32(size));
}

static QSet<void*> successfulInits;
static QThread *mainThread = nullptr;

void mainThreadCoInit(void* caller)
{
    Q_ASSERT(caller);

    if (QThread::currentThread() != QCoreApplication::instance()->thread()) {
        qCWarning(QT_BT_WINDOWS) << "Main thread COM init tried from another thread";
        return;
    }

    if (successfulInits.contains(caller)) {
        qCWarning(QT_BT_WINDOWS) << "Multiple COM inits by same object";
        return;
    }

    Q_ASSERT_X(!mainThread || mainThread == QThread::currentThread(),
               __FUNCTION__, "QCoreApplication's thread has changed!");

    // This executes in main thread which may run Gui => use apartment threaded
    if (!SUCCEEDED(CoInitializeEx(NULL, COINIT_APARTMENTTHREADED))) {
        qCWarning(QT_BT_WINDOWS) << "Unexpected COM initialization result";
        return;
    }
    successfulInits.insert(caller);
    if (!mainThread)
        mainThread = QThread::currentThread();
}

void mainThreadCoUninit(void* caller)
{
    Q_ASSERT(caller);

    if (!successfulInits.contains(caller)) {
        qCWarning(QT_BT_WINDOWS) << "COM uninitialization without initialization";
        return;
    }

    if (QThread::currentThread() != mainThread) {
        qCWarning(QT_BT_WINDOWS) << "Main thread COM uninit tried from another thread";
        return;
    }

    CoUninitialize();
    successfulInits.remove(caller);

}

QT_END_NAMESPACE
