# Copyright (C) 2005 FishGrove Inc.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#
# $Id: Handshaker_s.pm,v 1.20 2005/07/03 11:42:51 slash5234 Exp $

use strict;
use lib("../../../extlib/");
use XMLRPC::Transport::HTTP;
use DBI;
use Crypt::RC5;
use Crypt::DH;
use MIME::Base64::Perl;
use Error qw(:try);
use lib("../../");
use Affelio;
use Affelio::SNS::FriendManager;
use Affelio::SNS::Handshaker_tmpDB;
use Affelio::SNS::Handshaker_c qw(get_F2List);
use Affelio::Managing::MessageManager;
use Affelio::misc::CGIError;
use Affelio::misc::Debug;
use Affelio::misc::MyCrypt;
use Affelio::misc::NetMisc;
use Affelio::misc::Time;
use Affelio::misc::Sanitizer;

########################################################################
package Affelio::SNS::Handshaker_s::Util;
{
    sub af_new {
	my $cfg_dir = $Affelio::SNS::Handshaker_s::AF_DIR;
	Affelio::misc::Debug::debug_print("Starting AF($cfg_dir)...");
	my $af = Affelio->new( ConfigDir => $cfg_dir )
	    or die "Cannot start Affelio";
	return($af);
    }
}

########################################################################
package Affelio::SNS::Handshaker_s;
{
    use strict;
    use Exporter;
    @Affelio::SNS::Handshaker_s::ISA = "Exporter";
    @Affelio::SNS::Handshaker_s::EXPORT = qw (HandShake HandShakeReply F2List post_Message get_services);

    use vars qw( $AF_DIR);

    ##################################################################
    #server.get_services
    # proto_ver: 1
    ##################################################################
    sub get_services{
	my $self = shift;
	my ($proto_ver) = @_;

	if($proto_ver > 1.0){
	    return {
		flerror => XMLRPC::Data->type('boolean', 1),
		message => "ERR:102 UnsupportedProtoVer 1.0"
		};
	}

	my $msg= <<EOT;
<?xml version="1.0"?>
<services type="core">
<service name="AffelioHandshaker">
<version>1.1</version>
</service>
<service name="AffelioMessaging">
<version>1.0</version>
</service>
<service name="AffelioFriendList">
<version>1.0</version>
</service>
</services>
EOT
          return {
	    flerror => XMLRPC::Data->type('boolean', 0),
	    message => $msg
	    };
    }

    ##################################################################
    #server.F2List
    # proto_ver: 1
    ##################################################################
    sub F2List {
	my $self = shift;
	my ($proto_ver, $req_timestamp) = @_;

	my $af = Affelio::SNS::Handshaker_s::Util::af_new();

	$req_timestamp 
	    = Affelio::misc::Sanitizer::sanitize_number($req_timestamp);

	Affelio::misc::Debug::debug_print("server.F2List: proto_ver=$proto_ver, timestamp=$req_timestamp");

	##################################
	#Retrieve friends whose
	#     timestamp > req_timestamp
	#Retrieve erased friends whose
	#     timestamp > req_timestamp
	my $retmsg = $af->{fm}->get_updated_friends($req_timestamp);

	Affelio::misc::Debug::debug_print("server.F2List: ret=[$retmsg]");
	Affelio::misc::Debug::debug_print("server.F2List: end.");

	return {
	    flerror => XMLRPC::Data->type('boolean', 0),
	    message => $retmsg
	    };
    }

    ##################################################################
    #server.post_Message
    # proto_ver:1
    ##################################################################
    sub post_Message {
	my $self = shift;
	my $af = Affelio::SNS::Handshaker_s::Util::af_new();

	Affelio::misc::Debug::debug_print("server.postMesg: starg.");

        ##################################################
	#Distill args
        ##################################################
	my ($proto_ver, 
	    $timestamp, 
	    $peer_afid, 
	    $MIMed_mesg) = @_;

	my $passAB = $af->{fm}->get_attribute_by_afid($peer_afid, "password");
	if(!defined($passAB) || $passAB eq ""){
	    return {
		flerror => XMLRPC::Data->type('boolean', 1),
		message => "ERR:100 Youre not my friend."
	    };
	}

        ##################################################
	#Decode MIME
        ##################################################
	my $encrypted = MIME::Base64::Perl::decode_base64($MIMed_mesg);

        ##################################################
	#Decrypt
        ##################################################
	my $rc5 = Crypt::RC5->new($passAB, 12 );
	my $plain = $rc5->decrypt($encrypted);

        ##################################################
	#Parse XML
        ##################################################
	my $src;
	if($plain =~ /<src>(.+)<\/src>/){
	    $src = $1;
	}
	Affelio::misc::Debug::debug_print("server.postMesg: src: $src"); 

	my $msg_from;
	if($plain =~ /<from>(.+)<\/from>/){
	    $msg_from = $1;
	}
	Affelio::misc::Debug::debug_print("server.postMesg: from: $msg_from"); 

	my $msg_from_nickname;
	if($plain =~ /<from_nickname>(.+)<\/from_nickname>/){
	    $msg_from_nickname = $1;
	}
	Affelio::misc::Debug::debug_print("server.postMesg: nick: $msg_from_nickname"); 

	my $msg_to;
	if($plain =~ /<to>(.+)<\/to>/){
	    $msg_to = $1;
	}
	Affelio::misc::Debug::debug_print("server.postMesg: to: $msg_to");

	my $msg_timestamp;
	if($plain =~ /<timestamp>(.+)<\/timestamp>/){
	    $msg_timestamp = $1;
	}
	Affelio::misc::Debug::debug_print("server.postMesg: TIME: $msg_timestamp");

	my $msg_MIMed_title;
	if($plain =~ /<title>(.*)<\/title>/ms){
	    $msg_MIMed_title = $1;
	}
	Affelio::misc::Debug::debug_print("server.postMesg: title: $msg_MIMed_title");

	my $msg_MIMed_body;
	if($plain =~ /<text>(.*)<\/text>/ms){
	    $msg_MIMed_body = $1;
	}
	Affelio::misc::Debug::debug_print("server.postMesg: body: $msg_MIMed_body");

        ##################################################
	#Encryption integrity check
        ##################################################
	if($src ne $peer_afid){
	    return {
		flerror => XMLRPC::Data->type('boolean', 1),
		message => "ERR:101 Invalid Encryption."
		};
	}

        ##################################################
	#MIM decode for title and body
        ##################################################
	my $msg_title = MIME::Base64::Perl::decode_base64($msg_MIMed_title);
	my $msg_body  = MIME::Base64::Perl::decode_base64($msg_MIMed_body);

        ###########################################
        # Post this message to Message Manager
        ###########################################
	Affelio::misc::Debug::debug_print("server.PostMesg: $proto_ver, $msg_from $msg_from_nickname $msg_to $msg_timestamp $msg_title $msg_body");

	$msg_from = '<A HREF="' .  $msg_from . '">' . $msg_from_nickname . '</A>';

	$af->{mesgm}->post_message($msg_from,
				   $msg_title,
				   "UserToUser/OneToOne",
				   $msg_body);
	undef($af);
	
        ###########################################
        # Reply to client
        ###########################################
	my $msg = "OK: Thanks for your message.";
	return {
	    flerror => XMLRPC::Data->type('boolean', 0),
	    message => $msg
	    };
    }


    ##################################################################
    #server.HandShake
    # proto_ver:1.1
    # Accept HandShake from a client
    ##################################################################
    sub HandShake {
	my $self = shift;

        ##################################################
	#Distill args
        ##################################################
	my ($proto_ver, $timestamp, $peer_nickname, 
	    $peer_af_id, $peer_DH_pub_key_str, $MIMed_mesg) = @_;
	my $peer_domain = Affelio::misc::NetMisc::get_remote_domain(%ENV);
	my $sessionid = "$$" . "$timestamp";
	Affelio::misc::Debug::debug_print("server.HandShake: size=["
					  . length($MIMed_mesg));

	Affelio::misc::Debug::debug_print("server.HandShake: $proto_ver, $timestamp, $peer_domain, $peer_nickname, $peer_af_id, $peer_DH_pub_key_str [$MIMed_mesg]\n");
	##################################################
	#Version check
	##################################################
	if($proto_ver > 1.1){
	    return {
		flerror => XMLRPC::Data->type('boolean', 1),
		message => "ERR:102 UnsupportedProtoVer 1.1"
		};
	}

        ###########################################
        # Instantiate Affelio
        ###########################################
	my $af = Affelio::SNS::Handshaker_s::Util::af_new();

        ###########################################
        # Send a message to MessageManager
        ###########################################
	my $message_body= 
	    MIME::Base64::Perl::encode_base64("You got a link request from $peer_nickname ( $peer_af_id ). \n\nMessage from the user is...\n")
	    . "$MIMed_mesg\n"
	    . MIME::Base64::Perl::encode_base64("\n\nClick following link to approve this request.\n\n$af->{site__web_root}/bin/recv_mail_ack.cgi?id=$sessionid");

	$af->{mesgm}->post_message("Your Affelio",
				   "Link Request from $peer_nickname",
				   "SystemToUser/LinkRequest/Encode-Base64",
				   $message_body);

        ###########################################
        # Save peer's info into "received_Handshake" DB
        ###########################################
	my $tmpdb= new Affelio::SNS::Handshaker_tmpDB($af);
	$tmpdb->add_received_Handshake($sessionid,
				       $peer_af_id,
				       $peer_nickname,
				       $timestamp,
				       $peer_DH_pub_key_str);

	Affelio::misc::Debug::debug_print("server.HandShake: DB(W) $sessionid\n");

	undef($af);
        ###########################################
        # Reply to client
        ###########################################
	my $msg = "OK: Thanks for your HandShake.";
	return {
	    flerror => XMLRPC::Data->type('boolean', 0),
	    message => $msg
	    };
	
    }#method


    ##################################################################
    #server.HandShakeReply
    # proto_ver:1.1
    # Accept HandshakeReply from a client
    ##################################################################
    sub HandShakeReply {
	my $self = shift;
	
        ###########################################
	#Distill args
        ###########################################
	my ($proto_ver, $timestamp, $peer_nickname, 
	    $peer_af_id, $peer_DH_pub_key_str, $mesg) = @_;
	Affelio::misc::Debug::debug_print("server.HandShakeReply: $proto_ver, $timestamp, $peer_nickname, $peer_af_id, $peer_DH_pub_key_str $mesg\n");
	
	##################################################
	#Version check
	##################################################
	if($proto_ver > 1.1){
	    return {
		flerror => XMLRPC::Data->type('boolean', 1),
		message => "ERR:102 UnsupportedProtoVer 1.1"
		};
	}

        ###########################################
        # Instantiate Affelio
        ###########################################
	my $af = Affelio::SNS::Handshaker_s::Util::af_new();

        ###########################################
        # check peer's info in "received_Handshake" DB
        ###########################################
	Affelio::misc::Debug::debug_print("server.HandShakeReply: searching... $peer_af_id => $timestamp\n");

	my $tmpdb= new Affelio::SNS::Handshaker_tmpDB($af);
	my @ret= $tmpdb->remove_sent_Handshake($timestamp);

	my $my_DH_pri_key_str;
	my $dummy1;
	my $dummy2;
	my $dummy3;
	my $dummy4;

	if(!defined(@ret)){
	    #No such session exists!!
	    Affelio::misc::Debug::debug_print("server.HandShakeReply: sent-Handshake session NOT Found!\n");
	      return {
		  flerror => XMLRPC::Data->type('boolean', 1),
		  message => "ERR:103 HandShakeReply denied."
		  };
	}else{
	    ($dummy1, $dummy2, $dummy3, $dummy4, $my_DH_pri_key_str) = @ret;
	    Affelio::misc::Debug::debug_print("server.HandShakeReply: my DH_pri_key = $my_DH_pri_key_str");
	    Affelio::misc::Debug::debug_print("server.HandShakeReply: sent-Handshake session Found. OK. Let's Move on.");
	}
	
        ###########################################
        #generate password
        ###########################################
	my $mydh = Crypt::DH->new;
        #RFC 2412 - The OAKLEY Key Determination Protocol
        #Group 1:  A 768 bit prime
	my $DH_g="2";
	my $DH_p="1552518092300708935130918131258481755631334049434514313202351194902966239949102107258669453876591642442910007680288864229150803718918046342632727613031282983744380820890196288509170691316593175367469551763119843371637221007210577919";
	$mydh->g($DH_g);
	$mydh->p($DH_p);

	$mydh->priv_key(Math::BigInt->new($my_DH_pri_key_str) );
	my $pass 
	    = $mydh->compute_key(Math::BigInt->new($peer_DH_pub_key_str))->bstr;
	Affelio::misc::Debug::debug_print("server.HandShakeReply: PASSWORD=[$pass]\n");

        ###########################################
        #Add peer to my friends list.
        ###########################################
	my $uid = $af->{fm}->add_friend($peer_af_id, 
					$peer_nickname,
					$timestamp,
					$pass);
	Affelio::misc::Debug::debug_print("server.HandShakeReply: add_friend finished.\n");

	eval{
	    $af->{db}->commit;
	    $af->{db}->disconnect;
	    undef($af);
	};
	if($@){
	    Affelio::misc::Debug::debug_print($@);
	}

        ###########################################
        #Get peer's friends list.
        ###########################################
	# "peer's friends" = my F2 friends
	Affelio::misc::Debug::debug_print("server.HandshakeReply: Let's download peer's flist!");
	my $ret = Affelio::SNS::Handshaker_c::get_F2List(dest_uri =>  "$peer_af_id/bin/xml-rpc-serv.cgi", timestamp => 0);
	Affelio::misc::Debug::debug_print("server.HandshakeReply: get_F2List finished.");
	Affelio::misc::Debug::debug_print("server.HandshakeReply: List I've got is [$ret]");
	
        ###########################################
        #Save the F2 list into my DB
        ###########################################
	$af = Affelio::SNS::Handshaker_s::Util::af_new();
	Affelio::misc::Debug::debug_print("server.HandshakeReply: Let's save peer's flist!");
        $af->{fm}->save_F2List($ret, $peer_af_id);
	Affelio::misc::Debug::debug_print("server.HandshakeReply: save_F2List finished.");
	#Make a new instance of Affelio

        ###########################################
        #Reply to client
        ###########################################
	return {
	    flerror => XMLRPC::Data->type('boolean', 0),
	    message => "OK: Thanks for your HandShakeReply." 
	    };
    }

}


########################################################################
{    
    package affelio;
    BEGIN { @affelio::ISA = qw( Affelio::SNS::Handshaker_s ); }
}


########################################################################
1;
